/*******************************************************************************
 * Copyright (c) 2022 Contributors
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/org/documents/epl-2.0/EPL-2.0.txt
 *******************************************************************************/
package org.aspectj.weaver.patterns;

import org.aspectj.weaver.UnresolvedType;

/**
 * Check if there are illegal 'void[]' , 'void[][]' etc. occurrences anywhere in type patterns
 *
 * @author Alexander Kriegisch
 */
public class VoidArrayFinder extends AbstractPatternNodeVisitor {
  boolean found = false;
  PatternNode foundNode = null;

  public boolean hasVoidArray() {
    return found;
  }

  public PatternNode getFoundNode() {
    return foundNode;
  }

  // ------------------------------------------------------------------------
  // Signature patterns
  // ------------------------------------------------------------------------

//  public Object visit(SignaturePattern node, Object data) {
//    if (!found && node.getReturnType() != null)
//      node.getReturnType().accept(this, data);
//    if (!found && node.getDeclaringType() != null)
//      node.getDeclaringType().accept(this, data);
//    if (!found && node.getParameterTypes() != null)
//      node.getParameterTypes().accept(this, data);
//    return node;
//  }

  // ------------------------------------------------------------------------
  // Type patterns
  // ------------------------------------------------------------------------

//  public Object visit(NotTypePattern node, Object data) {
//    if (!found && node.getNegatedPattern() != null)
//      node.getNegatedPattern().accept(this, data);
//    return node;
//  }
//
//  public Object visit(AndTypePattern node, Object data) {
//    if (!found && node.getLeft() != null)
//      node.getLeft().accept(this, data);
//    if (!found && node.getRight() != null)
//      node.getRight().accept(this, data);
//    return node;
//  }
//
//  public Object visit(OrTypePattern node, Object data) {
//    if (!found && node.getLeft() != null)
//      node.getLeft().accept(this, data);
//    if (!found && node.getRight() != null)
//      node.getRight().accept(this, data);
//    return node;
//  }

  public Object visit(ExactTypePattern node, Object data) {
    if (found)
      return node;
    UnresolvedType exactType = node.getExactType();
    if (exactType != null && exactType.isArray() && exactType.toString().startsWith("void[")) {
      found = true;
      foundNode = node;
    }
    return node;
  }

  public Object visit(WildTypePattern node, Object data) {
    if (!found && node.isArray() && node.toString().startsWith("void[")) {
      found = true;
      foundNode = node;
    }
    if (!found && node.getLowerBound() != null)
      node.getLowerBound().accept(this, data);
    if (!found && node.getUpperBound() != null)
      node.getUpperBound().accept(this, data);
    if (found)
      return node;
    TypePattern[] additionalIntefaceBounds = node.getAdditionalIntefaceBounds();
    if (additionalIntefaceBounds != null) {
      for (TypePattern typePattern : additionalIntefaceBounds) {
        if (found)
          break;
        typePattern.accept(this, data);
      }
    }
    return node;
  }

  // ------------------------------------------------------------------------
  // Annotation type patterns
  // ------------------------------------------------------------------------

//  public Object visit(NotAnnotationTypePattern node, Object data) {
//    if (!found && node.getNegatedPattern() != null)
//      node.getNegatedPattern().accept(this, data);
//    return node;
//  }
//
//  public Object visit(AndAnnotationTypePattern node, Object data) {
//    if (!found && node.getLeft() != null)
//      node.getLeft().accept(this, data);
//    if (!found && node.getRight() != null)
//      node.getRight().accept(this, data);
//    return node;
//  }
//
//  public Object visit(OrAnnotationTypePattern node, Object data) {
//    if (!found && node.getLeft() != null)
//      node.getLeft().accept(this, data);
//    if (!found && node.getRight() != null)
//      node.getRight().accept(this, data);
//    return node;
//  }

  public Object visit(ExactAnnotationTypePattern node, Object data) {
    if (!found) {
      UnresolvedType exactType = node.getAnnotationType();
      if (exactType != null && exactType.isArray() && exactType.toString().startsWith("void[")) {
        found = true;
        foundNode = node;
      }
    }
    return node;
  }

  // ExactAnnotationFieldTypePattern.getAnnotationType() throws BCException("unimplemented")
  /*
  public Object visit(ExactAnnotationFieldTypePattern node, Object data) {
    if (!found) {
      UnresolvedType exactType = node.getAnnotationType(); // unimplemented
      if (exactType != null && exactType.isArray() && exactType.toString().startsWith("void[")) {
        found = true;
        foundNode = node;
      }
    }
    return node;
  }
  */

  public Object visit(BindingAnnotationFieldTypePattern node, Object data) {
    if (!found) {
      UnresolvedType exactType = node.getAnnotationType();
      if (exactType != null && exactType.isArray() && exactType.toString().startsWith("void[")) {
        found = true;
        foundNode = node;
      }
    }
    return node;
  }

  public Object visit(BindingAnnotationTypePattern node, Object data) {
    if (!found) {
      UnresolvedType exactType = node.getAnnotationType();
      if (exactType != null && exactType.isArray() && exactType.toString().startsWith("void[")) {
        found = true;
        foundNode = node;
      }
    }
    return node;
  }

//  public Object visit(WildAnnotationTypePattern node, Object data) {
//    if (!found && node.getTypePattern() != null)
//      node.getTypePattern().accept(this, data);
//    return node;
//  }

  // ------------------------------------------------------------------------
  // Type pattern lists
  // ------------------------------------------------------------------------

//  public Object visit(TypePatternList node, Object data) {
//    if (!found) {
//      TypePattern[] typePatterns = node.getTypePatterns();
//      if (!found && typePatterns != null) {
//        for (TypePattern typePattern : typePatterns) {
//          if (found)
//            break;
//          typePattern.accept(this, data);
//        }
//      }
//    }
//    return node;
//  }

}
