/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.chromattic.core;

import org.chromattic.common.logging.Logger;
import org.chromattic.api.Status;
import org.chromattic.api.DuplicateNameException;
import org.chromattic.api.NameConflictResolution;
import org.chromattic.core.jcr.info.MixinTypeInfo;
import org.chromattic.core.jcr.info.PrimaryTypeInfo;
import org.chromattic.core.mapper.NodeTypeKind;
import org.chromattic.core.mapper.ObjectMapper;
import org.chromattic.core.jcr.SessionWrapper;
import org.chromattic.core.jcr.LinkType;

import javax.jcr.RepositoryException;
import javax.jcr.Node;
import javax.jcr.ItemNotFoundException;
import javax.jcr.PathNotFoundException;
import javax.jcr.nodetype.NodeType;
import java.util.*;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class DomainSessionImpl extends DomainSession {

  /** . */
  final Domain domain;

  /** . */
  private Map<String, EntityContext> contexts;

  /** . */
  private final Logger log = Logger.getLogger(DomainSession.class);

  public DomainSessionImpl(Domain domain, SessionWrapper sessionWrapper) {
    super(domain, sessionWrapper);

    //
    this.domain = domain;
    this.contexts = new HashMap<String, EntityContext>();
  }

  protected void _setName(EntityContext ctx, String name) {
    if (ctx == null) {
      throw new NullPointerException();
    }

    //
    ctx.state.setName(name);
  }

  @Override
  protected String _getName(EntityContext ctx) throws RepositoryException {
    if (ctx == null) {
      throw new NullPointerException();
    }

    //
    switch (ctx.getStatus()) {
      default:
        return ctx.state.getName();
      case PERSISTENT:
        Node node = ctx.state.getNode();
        Node parentNode = node.getParent();
        return domain.decodeName(parentNode, node.getName(), NameKind.OBJECT);
    }
  }

  protected <O> O _findByPath(EntityContext ctx, Class<O> clazz, String relPath) throws RepositoryException {
    Node origin;
    if (ctx != null) {
      origin = ctx.state.getNode();
    } else {
      origin = getRoot();
      nodeRead(origin);
    }
    try {
      Node node = origin.getNode(relPath);
      nodeRead(node);
      return findByNode(clazz, node);
    }
    catch (PathNotFoundException e) {
      return null;
    }
  }

  protected String _persist(EntityContext ctx, String name) throws RepositoryException {
    if (ctx == null) {
      throw new NullPointerException("No null object context accepted");
    }
    if (name == null) {
      throw new NullPointerException("No relative path specified");
    }

    //
    if (ctx.getStatus() != Status.TRANSIENT) {
      String msg = "Attempt to persist non transient object " + ctx;
      log.error(msg);
      throw new IllegalArgumentException(msg);
    }

    //
    log.trace("Setting context {} for insertion", ctx);
    log.trace("Adding node for context {} and node type {}", ctx, ctx.mapper);

    //
    return _persist(getRoot(), name, ctx);
  }

  /**
   * Insert a context as a child of a parent context.
   *
   * @param srcCtx the source context
   * @param name the destination path relative to the source context
   * @param dstCtx the destination context
   * @return the id of the inserted context
   * @throws NullPointerException
   * @throws IllegalArgumentException
   * @throws IllegalStateException
   * @throws RepositoryException
   */
  protected String _persist(EntityContext srcCtx, String name, EntityContext dstCtx) throws
    NullPointerException,
    IllegalArgumentException,
    IllegalStateException,
    RepositoryException {
    if (srcCtx == null) {
      String msg = "Cannot insert context " + dstCtx + " as a child of a null context";
      log.error(msg);
      throw new NullPointerException(msg);
    }
    if (dstCtx.getStatus() != Status.TRANSIENT) {
      String msg = "Attempt to insert non transient context " + dstCtx + " as child of " + srcCtx;
      log.error(msg);
      throw new IllegalStateException(msg);
    }
    if (name == null) {
      String msg = "Attempt to insert context " + dstCtx + " with no relative path to " + srcCtx;
      log.error(msg);
      throw new NullPointerException(msg);
    }
    if (srcCtx.getStatus() != Status.PERSISTENT) {
      String msg = "Attempt to insert context " + dstCtx + " as child of non persistent context " + srcCtx;
      log.error(msg);
      throw new IllegalStateException(msg);
    }

    //
    Node parentNode = srcCtx.state.getNode();

    //
    return _persist(parentNode, name, dstCtx);
  }

  private String _persist(Node srcNode, String name, EntityContext dstCtx) throws RepositoryException {
    ObjectMapper mapper = dstCtx.mapper;

    //
    Object parent = findByNode(Object.class, srcNode);
    EntityContext parentCtx = parent != null ? unwrapEntity(parent) : null;

    //
    name = domain.encodeName(parentCtx, name, NameKind.OBJECT);

    //
    NameConflictResolution onDuplicate = NameConflictResolution.FAIL;
    NodeType parentNodeType = srcNode.getPrimaryNodeType();
    ObjectMapper parentTypeMapper = domain.getTypeMapper(parentNodeType.getName());
    if (parentTypeMapper != null) {
      onDuplicate = parentTypeMapper.getOnDuplicate();
    }

    // Check insertion capability
    Node previousNode = sessionWrapper.getNode(srcNode, name);
    if (previousNode != null) {
      log.trace("Found existing child with same name {}", name);
      if (onDuplicate == NameConflictResolution.FAIL) {
        String msg = "Attempt to insert context " + dstCtx + " as an existing child with name " + name + " child of node " + srcNode.getPath();
        log.error(msg);
        throw new DuplicateNameException(msg);
      } else {
        log.trace("About to remove same name {} child with id {}", previousNode.getPath(), previousNode.getName());
        remove(previousNode);
      }
    }

    //
    String primaryNodeTypeName = mapper.getNodeTypeName();
    log.trace("Setting context {} for insertion", dstCtx);
    log.trace("Adding node for context {} and node type {} as child of node {}", dstCtx, primaryNodeTypeName, srcNode.getPath());

    //
    Node dstNode = sessionWrapper.addNode(srcNode, name, primaryNodeTypeName, Collections.<String>emptyList());

    // If the node is not referenceable, make it so
    if (!domain.nodeInfoManager.isReferenceable(dstNode)) {
      dstNode.addMixin("mix:referenceable");
    }

    //
    nodeAdded(dstNode, dstCtx);

    //
    String relatedId = dstCtx.getId();

    //
    log.trace("Added context {} for path {}", dstCtx, relatedId, dstNode.getPath());
    return relatedId;
  }

  @Override
  protected void _addMixin(EntityContext entityCtx, EmbeddedContext mixinCtx) throws RepositoryException {
    if (entityCtx == null) {
      throw new NullPointerException();
    }
    if (mixinCtx == null) {
      throw new NullPointerException();
    }

    // Maybe they are already wired
    if (mixinCtx.relatedEntity != null) {
      if (mixinCtx.relatedEntity != entityCtx) {
        throw new IllegalArgumentException();
      }
    } else {
      EmbeddedContext previousMixinCtx = entityCtx.embeddeds.get(mixinCtx.mapper);
      if (previousMixinCtx != null) {
        if (previousMixinCtx != mixinCtx) {
          throw new IllegalStateException();
        }
      } else {

        //
        String mixinTypeName = mixinCtx.mapper.getNodeTypeName();
        Node node = entityCtx.state.getNode();

        //
        if (!sessionWrapper.canAddMixin(node, mixinTypeName)) {
          throw new IllegalArgumentException("Cannot add mixin " + mixinCtx + " to context " + entityCtx);
        }

        // Add mixin
        sessionWrapper.addMixin(node, mixinTypeName);

        //
        NodeType mixinType = sessionWrapper.getNodeType(mixinTypeName);
        MixinTypeInfo mixinTypeInfo = domain.nodeInfoManager.getMixinTypeInfo(mixinType);

        // Perform wiring
        entityCtx.embeddeds.put(mixinCtx.mapper, mixinCtx);
        mixinCtx.relatedEntity = entityCtx;
        mixinCtx.typeInfo = mixinTypeInfo;
      }
    }
  }

  @Override
  protected EmbeddedContext _getEmbedded(EntityContext entityCtx, Class<?> embeddedClass) throws RepositoryException {
    if (entityCtx == null) {
      throw new NullPointerException();
    }
    if (embeddedClass == null) {
      throw new NullPointerException();
    }

    // That's a necessary evil
    ObjectMapper<EmbeddedContext> mapper = (ObjectMapper<EmbeddedContext>)domain.getTypeMapper(embeddedClass);

    //
    EmbeddedContext embeddedCtx = null;
    if (mapper != null) {
      embeddedCtx = entityCtx.embeddeds.get(mapper);

      //
      if (embeddedCtx == null) {
        Node node = entityCtx.state.getNode();
        if (mapper.getKind() == NodeTypeKind.MIXIN) {
          String mixinTypeName = mapper.getNodeTypeName();
          if (sessionWrapper.haxMixin(node, mixinTypeName)) {
            NodeType mixinType = sessionWrapper.getNodeType(mixinTypeName);
            MixinTypeInfo mixinTypeInfo = domain.nodeInfoManager.getMixinTypeInfo(mixinType);

            //
            embeddedCtx = new EmbeddedContext(mapper, this);
            entityCtx.embeddeds.put(embeddedCtx.mapper, embeddedCtx);
            embeddedCtx.relatedEntity = entityCtx;
            embeddedCtx.typeInfo = mixinTypeInfo;
          }
        } else {
          PrimaryTypeInfo typeInfo = entityCtx.state.getTypeInfo();
          PrimaryTypeInfo superTI = (PrimaryTypeInfo)typeInfo.getSuperType(mapper.getNodeTypeName());
          if (superTI != null) {
            embeddedCtx = new EmbeddedContext(mapper, this);
            entityCtx.embeddeds.put(embeddedCtx.mapper, embeddedCtx);
            embeddedCtx.relatedEntity = entityCtx;
            embeddedCtx.typeInfo = superTI;
          }
        }
      }
    }

    //
    return embeddedCtx;
  }

  @Override
  protected void _move(EntityContext srcCtx, EntityContext dstCtx) throws RepositoryException {
    if (srcCtx == null) {
      String msg = "Cannot move null context";
      log.error(msg);
      throw new NullPointerException(msg);
    }
    if (dstCtx == null) {
      String msg = "Cannot move to null context";
      log.error(msg);
      throw new NullPointerException(msg);
    }
    if (srcCtx.getStatus() != Status.PERSISTENT) {
      String msg = "Attempt to move non persistent context " + srcCtx + " as child of " + dstCtx;
      log.error(msg);
      throw new IllegalStateException(msg);
    }
    if (dstCtx.getStatus() != Status.PERSISTENT) {
      String msg = "Attempt to move child " + srcCtx + " to a non persistent context " + dstCtx;
      log.error(msg);
      throw new IllegalStateException(msg);
    }

    //
    Node dstNode = dstCtx.state.getNode();
    Node srcNode = srcCtx.state.getNode();
    String name = srcNode.getName();

    //
    NameConflictResolution onDuplicate = NameConflictResolution.FAIL;
    NodeType parentNodeType = dstNode.getPrimaryNodeType();
    ObjectMapper parentTypeMapper = domain.getTypeMapper(parentNodeType.getName());
    if (parentTypeMapper != null) {
      onDuplicate = parentTypeMapper.getOnDuplicate();
    }

    // Check insertion capability
    Node previousNode = sessionWrapper.getNode(dstNode, name);
    if (previousNode != null) {
      log.trace("Found existing child with same name {}", name);
      if (onDuplicate == NameConflictResolution.FAIL) {
        String msg = "Attempt to move context " + dstCtx + " as an existing child with name " + name + " child of node " + dstNode.getPath();
        log.error(msg);
        throw new DuplicateNameException(msg);
      } else {
        log.trace("About to remove same name {} child with id {}", previousNode.getPath(), previousNode.getName());
        //previousNode.remove();
        throw new UnsupportedOperationException("Do that properly");
      }
    }

    //
    sessionWrapper.move(srcNode, dstNode);

    // Generate some kind of event ????
  }

  protected void _orderBefore(EntityContext parentCtx, EntityContext srcCtx, EntityContext dstCtx) throws RepositoryException {

    if (parentCtx == null) {
      throw new NullPointerException();
    }
    if (srcCtx == null) {
      throw new NullPointerException();
    }

    //
    Node parentNode = parentCtx.state.getNode();
    Node srcNode = srcCtx.state.getNode();
    Node dstNode = dstCtx != null ? dstCtx.state.getNode() : null;

    //
    sessionWrapper.orderBefore(parentNode, srcNode, dstNode);
  }

  protected <O> O _create(Class<O> clazz, String name) throws NullPointerException, IllegalArgumentException, RepositoryException {
    if (clazz == null) {
      throw new NullPointerException();
    }

    //
    ObjectMapper<?> typeMapper = domain.getTypeMapper(clazz);
    TransientEntityContextState state = new TransientEntityContextState(this);

    //
    ObjectContext octx;
    if (typeMapper.getKind() == NodeTypeKind.PRIMARY) {
      EntityContext ctx = new EntityContext((ObjectMapper<EntityContext>)typeMapper, state);

      //
      if (name != null) {
        ctx.setName(name);
      }

      //
      broadcaster.created(ctx.getObject());

      //
      octx = ctx;
    } else {
      if (name != null) {
        throw new IllegalArgumentException("Cannot create a mixin type with a name");
      }
      octx = new EmbeddedContext((ObjectMapper<EmbeddedContext>)typeMapper, this);
    }
    return clazz.cast(octx.getObject());
  }

  protected <O> O _findById(Class<O> clazz, String id) throws RepositoryException {
    if (clazz == null) {
      throw new NullPointerException();
    }
    if (id == null) {
      throw new NullPointerException();
    }

    // Attempt to load the object
    try {
      log.trace("About to load node with id {} and class {}", id, clazz.getName());
      Node node = sessionWrapper.getNodeByUUID(id);
      return _findByNode(clazz, node);
    }
    catch (ItemNotFoundException e) {
      log.trace("Could not find node with id {}", id, clazz.getName());
      return null;
    }
  }

  protected <O> O _findByNode(Class<O> clazz, Node node) throws RepositoryException {
    if (clazz == null) {
      throw new NullPointerException();
    }

    //
    EntityContext ctx = _findByNode(node);

    //
    if (ctx == null) {
      return null;
    } else {
      Object object = ctx.object;
      if (clazz.isInstance(object)) {
        return clazz.cast(object);
      } else {
        String msg = "Could not cast context " + ctx + " with class " + object.getClass().getName() + " to class " + clazz.getName();
        throw new ClassCastException(msg);
      }
    }
  }

  private EntityContext _findByNode(Node node) throws RepositoryException {
    if (node == null) {
      throw new NullPointerException();
    }

    //
    if (!domain.nodeInfoManager.isReferenceable(node)) {
      log.trace("Cannot map non referenceable node {} to a chromattic object", node.getPath());
      return null;
    }

    // Attempt to get the object
    EntityContext ctx = contexts.get(node.getUUID());

    //
    if (ctx == null) {
      try {
        log.trace("About to read node with path {}", node.getPath());
        nodeRead(node);
        log.trace("Loaded node with path {}", node.getPath());
        ctx = contexts.get(node.getUUID());
        log.trace("Obtained context {} node for path {}", ctx, node.getPath());
      }
      catch (ItemNotFoundException e) {
        log.trace("Could not find node with path {}", node.getPath());
        return null;
      }
    }

    //
    return ctx;
  }

  protected void _save() throws RepositoryException {
    sessionWrapper.save();
  }

  protected void _remove(EntityContext context) throws RepositoryException {
    if (context == null) {
      throw new NullPointerException();
    }
    switch (context.state.getStatus()) {
      case TRANSIENT:
        throw new IllegalStateException("Cannot remove transient node");
      case PERSISTENT:
        Node node = context.state.getNode();
        remove(node);
        break;
      case REMOVED:
        throw new IllegalStateException("Cannot remove removed node");
      default:
        throw new AssertionError();
    }
  }

  private static class Removed {

    private final String id;
    private final String path;
    private final String name;
    private final EntityContext ctx;

    private Removed(String id, String path, String name, EntityContext ctx) {
      this.id = id;
      this.path = path;
      this.name = name;
      this.ctx = ctx;
    }
  }

  private void remove(Node node) throws RepositoryException {
    List<Removed> removeds = new LinkedList<Removed>();
    String pathToRemove = node.getPath();
    for (Map.Entry<String, EntityContext> ctxEntry : contexts.entrySet()) {
      EntityContext ctx = ctxEntry.getValue();
      Node ctxNode = ctx.state.getNode();
      if (ctxNode.getPath().startsWith(pathToRemove)) {
        removeds.add(new Removed(ctx.getId(), ctx.getPath(), ctx.getName(), ctx));
      }
    }

    // Perform removal
    sessionWrapper.remove(node);

    //
    Collection<EntityContext> ctxs = contexts.values();

    //
    for (Removed removed : removeds) {

      String path = removed.path;
      log.trace("Removing context for path {}", path);
      removed.ctx.state = new RemovedEntityContextState(path);
      ctxs.remove(removed.ctx);
      broadcaster.removed(removed.id, removed.path, removed.name, removed.ctx.getObject());
      log.trace("Removed context {} for path {}", removed.ctx, path);
    }
  }

  protected Object _getReferenced(EntityContext referentCtx, String name, LinkType linkType) throws RepositoryException {
    if (referentCtx.getStatus() != Status.PERSISTENT) {
      throw new IllegalStateException();
    }
    Node referent = referentCtx.state.getNode();
    Node referenced = sessionWrapper.getReferenced(referent, name, linkType);
    if (referenced != null) {
      return findByNode(Object.class, referenced);
    } else {
      return null;
    }
  }

  protected boolean _setReferenced(EntityContext referentCtx, String name, EntityContext referencedCtx, LinkType linkType) throws RepositoryException {
    if (referentCtx.getStatus() != Status.PERSISTENT) {
      throw new IllegalStateException("Cannot create a relationship with a non persisted context " + this);
    }

    //
    Node referent = referentCtx.state.getNode();

    // Then create
    if (referencedCtx != null) {
      if (referencedCtx.getStatus() != Status.PERSISTENT) {
        throw new IllegalStateException();
      }

      // Should do some type checking probably!!!!

      //
      Node referenced = referencedCtx.state.getNode();

      //
      return referenced != sessionWrapper.setReferenced(referent, name, referenced, linkType);
    } else {
      return null != sessionWrapper.setReferenced(referent, name, null, linkType);
    }
  }

  protected <T> Iterator<T> _getReferents(EntityContext referencedCtx, String name, Class<T> filterClass, LinkType linkType) throws RepositoryException {
    Node referenced = referencedCtx.state.getNode();
    Iterator<Node> referents = sessionWrapper.getReferents(referenced, name, linkType);
    return new ReferentCollectionIterator<T>(this, referents, filterClass, name);
  }

  protected void _removeChild(EntityContext ctx, String name) throws RepositoryException {
    name = domain.encodeName(ctx, name, NameKind.OBJECT);
    Node node = ctx.state.getNode();
    Node childNode = sessionWrapper.getNode(node, name);
    if (childNode != null) {
      remove(childNode);
    }
  }

  protected Object _getChild(EntityContext ctx, String name) throws RepositoryException {
    name = domain.encodeName(ctx, name, NameKind.OBJECT);
    Node node = ctx.state.getNode();
    log.trace("About to load the name child {} of context {}", name, this);
    Node child = sessionWrapper.getChild(node, name);
    if (child != null) {
      log.trace("Loaded named child {} of context {} with path {}", name, this, child.getPath());
      return findByNode(Object.class, child);
    } else {
      log.trace("No child named {} to load for context {}", name, this);
      return null;
    }
  }

  protected <T> Iterator<T> _getChildren(EntityContext ctx, Class<T> filterClass) throws RepositoryException {
    Node node = ctx.state.getNode();
    Iterator<Node> iterator = sessionWrapper.getChildren(node);
    return new ChildCollectionIterator<T>(this, iterator, filterClass);
  }

  protected Object _getParent(EntityContext ctx) throws RepositoryException {
    if (ctx.getStatus() != Status.PERSISTENT) {
      throw new IllegalStateException();
    }
    Node node = ctx.state.getNode();
    Node parent = sessionWrapper.getParent(node);
    return _findByNode(Object.class, parent);
  }

  protected Node _getRoot() throws RepositoryException {
    if ("/".equals(domain.rootNodePath)) {
      return sessionWrapper.getSession().getRootNode();
    } else {
      return (Node)sessionWrapper.getSession().getItem(domain.rootNodePath);
    }
  }

  private void nodeRead(Node node) throws RepositoryException {
    NodeType nodeType = node.getPrimaryNodeType();
    String nodeTypeName = nodeType.getName();
    ObjectMapper mapper = domain.getTypeMapper(nodeTypeName);
    if (mapper != null) {
      EntityContext ctx = contexts.get(node.getUUID());
      if (ctx == null) {
        ctx = new EntityContext((ObjectMapper<EntityContext>)mapper, new PersistentEntityContextState(node, this));
        log.trace("Inserted context {} loaded from node path {}", ctx, node.getPath());
        contexts.put(node.getUUID(), ctx);
        broadcaster.loaded(ctx, ctx.getObject());
      }
      else {
        log.trace("Context {} is already present for path ", ctx, node.getPath());
      }
    }
    else {
      log.trace("Could not find mapper for node type {}", nodeTypeName);
    }
  }

  private void nodeAdded(Node node, EntityContext ctx) throws RepositoryException {
    NodeType nodeType = node.getPrimaryNodeType();
    String nodeTypeName = nodeType.getName();
    ObjectMapper mapper = domain.getTypeMapper(nodeTypeName);
    if (mapper != null) {
      if (contexts.containsKey(node.getUUID())) {
        String msg = "Attempt to replace an existing context " + ctx + " with path " + node.getPath();
        log.error(msg);
        throw new AssertionError(msg);
      }
      log.trace("Inserted context {} for path {}", ctx, node.getPath());
      contexts.put(node.getUUID(), ctx);
      ctx.state = new PersistentEntityContextState(node, this);
      broadcaster.added(ctx, ctx.getObject());
    }
    else {
      log.trace("Could not find mapper for node type {}", nodeTypeName);
    }
  }

  public void close() {
    sessionWrapper.close();
  }
}
