/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2004 Andy Jefferson - added description of addition process.
2004 Andy Jefferson - added constructor, and try-catch on initialisation
2006 Andy Jefferson - renamed to PersistenceConfiguration so that it is API agnostic
2008 Andy Jefferson - rewritten to have properties map and not need Java beans setters/getters
    ...
**********************************************************************/
package org.datanucleus;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Collections;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TimeZone;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.plugin.ConfigurationElement;
import org.datanucleus.plugin.PluginManager;
import org.datanucleus.properties.PersistencePropertyValidator;
import org.datanucleus.properties.PropertyTypeInvalidException;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;

/**
 * Class providing configuration for persistence.
 * Persistence properties should be defined in the "plugin.xml" of the plugin that originates them. This class
 * will read the available defined plugin properties (and their defaults, validators etc) and loaded up
 * accordingly. Then any user-provided properties are superimposed over the top of these. Components can
 * then access these properties using any of the convenience accessors for boolean, Boolean, long, int, Object,
 * String types.
 */
public abstract class PersistenceConfiguration
{
    /** Localisation of messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        ObjectManager.class.getClassLoader());

    /** Map of persistence properties. */
    private Map<String, Object> properties = new HashMap<String, Object>();

    /** Mapping for the properties of the plugins, PropertyMapping, keyed by the property name. */
    private Map<String, PropertyMapping> propertyMappings = new HashMap<String, PropertyMapping>();

    /** Calendar for this datastore. */
    private transient Calendar dateTimezoneCalendar = null;

    /** Flag for whether this object is still configurable. */
    private transient boolean configurable = true;

    /**
     * Convenience class wrapping the plugin property specification information.
     */
    class PropertyMapping
    {
        String name;
        String internalName;
        String value;
        String validatorName;
        public PropertyMapping(String name, String intName, String val, String validator)
        {
            this.name = name;
            this.internalName = intName;
            this.value = val;
            this.validatorName = validator;
        }
    }

    /**
     * Constructor.
     */
    public PersistenceConfiguration()
    {
    }

    /**
     * Accessor for the names of the supported persistence properties.
     * @return The persistence properties that we support
     */
    public Set<String> getSupportedProperties()
    {
        return propertyMappings.keySet();
    }

    /**
     * Method to set the persistence property defaults based on what is defined for plugins.
     * @param pluginMgr The plugin manager
     */
    public void setDefaultProperties(PluginManager pluginMgr)
    {
        ConfigurationElement[] propElements =
            pluginMgr.getConfigurationElementsForExtension("org.datanucleus.persistence_properties", null, null);
        if (propElements != null)
        {
            for (int i=0;i<propElements.length;i++)
            {
                String name = propElements[i].getAttribute("name");
                String intName = propElements[i].getAttribute("internal-name");
                String value = propElements[i].getAttribute("value");
                propertyMappings.put(name, new PropertyMapping(name, intName, value, 
                    propElements[i].getAttribute("validator")));

                Object systemValue = System.getProperty(name);
                if (systemValue != null)
                {
                    // System value provided so add property with that value
                    if (intName != null)
                    {
                        setPropertyInternal(intName, systemValue);
                    }
                    else
                    {
                        setPropertyInternal(name, systemValue);
                    }
                }
                else
                {
                    if (!properties.containsKey(name) && value != null)
                    {
                        // Property has default value and not yet set so add property with that value
                        if (intName != null)
                        {
                            setPropertyInternal(intName, value);
                        }
                        else
                        {
                            setPropertyInternal(name, value);
                        }
                    }
                }
            }
        }
    }

    /**
     * Accessor for the persistence properties.
     * @return The persistence properties
     */
    public Map<String, Object> getPersistenceProperties()
    {
        return Collections.unmodifiableMap(properties);
    }

    /**
     * Accessor for the specified property as an Object.
     * If the specified property isn't found returns null.
     * @param name Name of the property
     * @return Value for the property
     */
    public Object getProperty(String name)
    {
        return properties.get(name.toLowerCase());
    }

    /**
     * Accessor for whether a particular property is defined.
     * @param name Property name
     * @return Whether the property is defined
     */
    public boolean hasProperty(String name)
    {
        return getProperty(name) != null;
    }

    /**
     * Accessor for the specified property as a long.
     * If the specified property isn't found returns 0.
     * @param name Name of the property
     * @return Long value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public long getLongProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Number)
            {
                return ((Number)obj).longValue();
            }
            else if (obj instanceof String)
            {
                Long longVal = Long.valueOf((String)obj);
                setPropertyInternal(name, longVal); // Replace String value with Long
                return longVal.longValue();
            }
        }
        else
        {
            return 0;
        }
        throw new PropertyTypeInvalidException(name, "long");
    }

    /**
     * Accessor for the specified property as an int.
     * If the specified property isn't found returns 0.
     * @param name Name of the property
     * @return Int value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public int getIntProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Number)
            {
                return ((Number)obj).intValue();
            }
            else if (obj instanceof String)
            {
                Integer intVal = Integer.valueOf((String)obj);
                setPropertyInternal(name, intVal); // Replace String value with Integer
                return intVal.intValue();
            }
        }
        else
        {
            return 0;
        }
        throw new PropertyTypeInvalidException(name, "int");
    }

    /**
     * Accessor for the specified property as a boolean.
     * If the specified property isn't found returns false.
     * @param name Name of the property
     * @return Boolean value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public boolean getBooleanProperty(String name)
    {
        return getBooleanProperty(name, false);
    }

    /**
     * Accessor for the specified property as a boolean.
     * @param name Name of the property
     * @param resultIfNotSet The value to return if no value for the specified property is found.
     * @return Boolean value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public boolean getBooleanProperty(String name, boolean resultIfNotSet)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Boolean)
            {
                return ((Boolean)obj).booleanValue();
            }
            else if (obj instanceof String)
            {
                Boolean boolVal = new Boolean((String)obj);
                setPropertyInternal(name, boolVal); // Replace String value with Boolean
                return boolVal.booleanValue();
            }
        }
        else
        {
            return resultIfNotSet;
        }
        throw new PropertyTypeInvalidException(name, "boolean");
    }

    /**
     * Accessor for the specified property as a Boolean.
     * If the specified property isn't found returns false.
     * @param name Name of the property
     * @return Boolean value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public Boolean getBooleanObjectProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof Boolean)
            {
                return ((Boolean)obj);
            }
            else if (obj instanceof String)
            {
                Boolean boolVal = new Boolean((String)obj);
                setPropertyInternal(name, boolVal); // Replace String value with Boolean
                return boolVal;
            }
        }
        else
        {
            return null;
        }
        throw new PropertyTypeInvalidException(name, "Boolean");
    }

    /**
     * Accessor for the specified property as a String.
     * If the specified property isn't found returns null.
     * @param name Name of the property
     * @return String value for the property
     * @throws PropertyTypeInvalidException thrown when the property is not available as this type
     */
    public String getStringProperty(String name)
    {
        Object obj = getProperty(name);
        if (obj != null)
        {
            if (obj instanceof String)
            {
                return ((String)obj);
            }
        }
        else
        {
            return null;
        }
        throw new PropertyTypeInvalidException(name, "String");
    }

    /**
     * Set the value for this persistence property.
     * @param name Name of the property
     * @param value The value
     */
    private void setPropertyInternal(String name, Object value)
    {
        this.properties.put(name.toLowerCase(), value);
    }

    /**
     * Accessor for the Calendar to be used in handling all timezone issues with the datastore.
     * Utilises the "serverTimeZoneID" in providing this Calendar used in time/date conversions.
     * @return The calendar to use for dateTimezone issues.
     */
    public Calendar getCalendarForDateTimezone()
    {
        if (dateTimezoneCalendar == null)
        {
            TimeZone tz;
            String serverTimeZoneID = getStringProperty("datanucleus.ServerTimeZoneID");
            if (serverTimeZoneID != null)
            {
                tz = TimeZone.getTimeZone(serverTimeZoneID);
            }
            else
            {
                tz = TimeZone.getDefault();
            }
            dateTimezoneCalendar = new GregorianCalendar(tz);
        }
        // This returns a clone because Oracle JDBC driver was taking the Calendar and modifying it
        // in calls. Hence passing a clone gets around that. May be best to just return it direct here
        // and then in Oracle usage we pass in a clone to its JDBC driver
        return (Calendar) dateTimezoneCalendar.clone();
    }

    /**
     * Mutator to set the primary class loader.
     * Setter provided since the input is an object and so cannot go through property input
     * @param loader Loader
     */
    public void setPrimaryClassLoader(ClassLoader loader)
    {
        setProperty("datanucleus.primaryClassLoader", loader);
    }

    /**
     * Accessor for the primary class loader
     * @return primary class loader
     */
    public ClassLoader getPrimaryClassLoader()
    {
        return (ClassLoader)getProperty("datanucleus.primaryClassLoader");
    }

    /**
     * Method to set the persistence properties using those defined in a file.
     * @param filename Name of the file containing the properties
     */
    public synchronized void setPropertiesUsingFile(String filename)
    {
        assertConfigurable();
        if (filename == null)
        {
            return;
        }

        // try to load the properties file
        Properties props = new Properties();
        File file = new File(filename);
        if (file.exists())
        {
            setPropertyInternal("datanucleus.propertiesFile", filename);
            try
            {
                InputStream is = new FileInputStream(file);
                props.load(is);
                is.close();
            }
            catch (FileNotFoundException e)
            {
                properties.remove("datanucleus.propertiesFile");
                throw new NucleusUserException(LOCALISER.msg("008014", filename), e).setFatal();
            }
            catch (IOException e)
            {
                properties.remove("datanucleus.propertiesFile");
                throw new NucleusUserException(LOCALISER.msg("008014", filename), e).setFatal();
            }
        }
        else
        {
            // Try to load it as a resource in the CLASSPATH
            try
            {
                InputStream is = PersistenceConfiguration.class.getClassLoader().getResourceAsStream(filename);
                props.load(is);
                is.close();

                setPropertyInternal("datanucleus.propertiesFile", filename);
            }
            catch (Exception e)
            {
                // Still not loadable so throw exception
                throw new NucleusUserException(LOCALISER.msg("008014", filename), e).setFatal();
            }
        }

        setPersistenceProperties(props);
    }

    /**
     * Set the properties for this configuration.
     * Note : this has this name so it has a getter/setter pair for use by things like Spring.
     * @param props The persistence properties
     */
    public void setPersistenceProperties(Map props)
    {
        assertConfigurable();

        Set keys = props.keySet();
        Iterator keyIter = keys.iterator();
        while (keyIter.hasNext())
        {
            Object keyObj = keyIter.next();
            if (keyObj instanceof String)
            {
                String key = (String)keyObj;
                Object valueObj = props.get(keyObj);
                setProperty(key, valueObj);
            }
        }
    }

    /**
     * Convenience method to set a persistence property.
     * Uses any validator defined for the property to govern whether the value is suitable.
     * @param name Name of the property
     * @param value Value
     */
    public void setProperty(String name, Object value)
    {
        if (name != null)
        {
            String propertyName = name.trim();
            PropertyMapping mapping = propertyMappings.get(propertyName);
            if (mapping != null)
            {
                if (mapping.validatorName != null)
                {
                    // TODO Use ClassLoaderResolver
                    PersistencePropertyValidator validator = null;
                    try
                    {
                        Class validatorCls = Class.forName(mapping.validatorName);
                        validator = (PersistencePropertyValidator)validatorCls.newInstance();
                    }
                    catch (Exception e)
                    {
                        NucleusLogger.PERSISTENCE.warn("Error creating validator of type " + mapping.validatorName, e);
                    }

                    if (validator != null)
                    {
                        boolean validated = (mapping.internalName != null ? 
                                validator.validate(mapping.internalName, value) : 
                                    validator.validate(propertyName, value));
                        if (!validated)
                        {
                            throw new IllegalArgumentException(LOCALISER.msg("008012", propertyName, value));
                        }
                    }
                }

                if (mapping.internalName != null)
                {
                    setPropertyInternal(mapping.internalName, value);
                }
                else
                {
                    setPropertyInternal(mapping.name, value);
                }

                // Special behaviour properties
                if (propertyName.equals("datanucleus.propertiesFile"))
                {
                    // Load all properties from the specified file
                    setPropertiesUsingFile((String)value);
                }
                else if (propertyName.equals("datanucleus.localisation.messageCodes"))
                {
                    // Set global log message code flag
                    boolean included = getBooleanProperty("datanucleus.localisation.messageCodes");
                    Localiser.setDisplayCodesInMessages(included);
                }
                else if (propertyName.equals("datanucleus.localisation.language"))
                {
                    String language = getStringProperty("datanucleus.localisation.language");
                    Localiser.setLanguage(language);
                }
            }
            else
            {
                // Unknown property so just add it.
                setPropertyInternal(propertyName, value);
                if (propertyMappings.size() > 0)
                {
                    NucleusLogger.PERSISTENCE.info(LOCALISER.msg("008015", propertyName));
                }
            }
        }
    }

    /**
     * Equality operator.
     * @param obj Object to compare against.
     * @return Whether the objects are equal.
     */
    public synchronized boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof PersistenceConfiguration))
        {
            return false;
        }

        PersistenceConfiguration config = (PersistenceConfiguration)obj;
        if (properties == null)
        {
            if (config.properties != null)
            {
                return false;
            }
        }
        else if (!properties.equals(config.properties))
        {
            return false;
        }
        return true;
    }

    /**
     * Accessor for whether this is still configurable (can set more properties etc).
     * @return Whether it is configurable
     */
    protected boolean isConfigurable()
    {
        return configurable;
    }

    /**
     * Method to set that this is no longer configurable.
     * Can no longer become configurable.
     */
    protected void setIsNotConfigurable()
    {
        this.configurable = false;
    }

    /**
     * Asserts that a change to a configuration property is allowed.
     * @throws NucleusUserException if not configurable. Override in subclass if wanting other exception
     */
    protected void assertConfigurable()
    {
        if (!configurable)
        {
            throw new NucleusUserException(LOCALISER.msg("008016"));
        }
    }
}