/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - merged Compiler to make more understandable and compact
2008 Andy Jefferson - added compile() method with return object
    ...
 **********************************************************************/
package org.datanucleus.query.compiler;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.query.JDOQLQueryHelper;
import org.datanucleus.query.compiler.JDOQLParser;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.query.symbol.PropertySymbol;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolResolver;
import org.datanucleus.query.symbol.SymbolTable;
import org.datanucleus.util.Imports;

/**
 * Implementation of a compiler for JDOQL (JSR0012, JSR0243).
 */
public class JDOQLCompiler extends JavaQueryCompiler implements SymbolResolver
{
    public JDOQLCompiler(MetaDataManager metaDataManager, ClassLoaderResolver clr, 
            String from, Class candidateClass, Collection candidates, 
            String filter, Imports imports, String ordering, String result, String grouping, String having, 
            String params, String variables)
    {
        super(metaDataManager, clr, from, candidateClass, candidates, 
            filter, imports, ordering, result, grouping, having, params, variables, null);
    }

    /**
     * Method to compile the query, and return the compiled results.
     * @param parameters the parameter map of values keyed by param name
     * @param subqueryMap Map of subquery variables, keyed by the subquery name
     * @return The compiled query
     */
    public QueryCompilation compile(Map parameters, Map subqueryMap)
    {
        Map parseOptions = new HashMap();
        if (this.parameters != null)
        {
            parseOptions.put("explicitParameters", true);
        }
        else
        {
            parseOptions.put("implicitParameters", true);
        }
        parser = new JDOQLParser(parseOptions);
        symtbl = new SymbolTable(clr);
        symtbl.setSymbolResolver(this);
        if (parentCompiler != null)
        {
            symtbl.setParentSymbolTable(parentCompiler.symtbl);
        }

        if (subqueryMap != null && !subqueryMap.isEmpty())
        {
            // Load subqueries into symbol table so the compilation knows about them
            Iterator<String> subqueryIter = subqueryMap.keySet().iterator();
            while (subqueryIter.hasNext())
            {
                String subqueryName = subqueryIter.next();
                Symbol sym = new PropertySymbol(subqueryName);
                sym.setType(Symbol.VARIABLE);
                symtbl.addSymbol(sym);
            }
        }

        Expression[] exprFrom = compileFrom();
        compileCandidatesParametersVariables(parameters);
        Expression exprFilter = compileFilter();
        Expression[] exprOrdering = compileOrdering();
        Expression[] exprResult = compileResult();
        Expression[] exprGrouping = compileGrouping();
        Expression exprHaving = compileHaving();
        QueryCompilation compilation = new QueryCompilation(candidateClass, candidateAlias, symtbl,
            exprResult, exprFrom, exprFilter, exprGrouping, exprHaving, exprOrdering, null);
        compilation.setQueryLanguage("JDOQL");

        return compilation;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.symbol.SymbolResolver#supportsVariables()
     */
    public boolean supportsImplicitVariables()
    {
        if (variables != null)
        {
            // Query uses explicit variables, so don't allow implicit
            return false;
        }
        return true;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.symbol.SymbolResolver#caseSensitiveSymbolNames()
     */
    public boolean caseSensitiveSymbolNames()
    {
        return true;
    }
    
    /**
     * Accessor for the query language name.
     * @return Name of the query language.
     */
    public String getLanguage()
    {
        return "JDOQL";
    }

    /**
     * Method to return if the supplied name is a keyword.
     * Keywords can only appear at particular places in a query so we need to detect for valid queries.
     * @param name The name
     * @return Whether it is a keyword
     */
    protected boolean isKeyword(String name)
    {
        if (name == null)
        {
            return false;
        }
        else if (JDOQLQueryHelper.isKeyword(name))
        {
            return true;
        }
      /*else if (name.equals("this"))
        {
            return true;
        }*/
        return false;
    }
}
