/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards
2005 Andy Jefferson - added contrib from Guillaume Bodet for escape of single-quote
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigInteger;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a string literal.
 */
public class StringLiteral extends StringExpression implements Literal
{
    private final String value;

    /** Raw value that this literal represents. */
    Object rawValue;

    private StatementText stUsingParameter = new StatementText();

    /**
     * Constructor that takes a char. TODO we should not accept chars in String literal
     * @param qs The QueryExpression
     * @param mapping The mapping to the java type
     * @param value The value that is further converted to a String
     */
    public StringLiteral(QueryExpression qs, JavaTypeMapping mapping, char value)
    {
        this(qs,mapping,String.valueOf(value));
    }

    /**
     * Constructor
     * @param qs The QueryExpression
     * @param mapping The mapping to the java type. null mapping is accepted 
     * @param value The value
     */
    public StringLiteral(QueryExpression qs, JavaTypeMapping mapping, String value)
    {
        super(qs);

        this.mapping = mapping;
        this.value = value;
        // Escape any single-quotes
        st.append('\'').append(value.replace("'", "''")).append('\'');

        if (mapping == null)
        {
            stUsingParameter.appendParameter(qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
                String.class, false, false, qs.getClassLoaderResolver()), value);
        }
        else
        {
            stUsingParameter.appendParameter(mapping,value);
        }
    }

    /**
     * Convenience method to generate the statement without any quotes.
     * This is called when we create a literal using a mapping, and dont want quotes
     * because the string is an SQL keyword.
     */
    public void generateStatementWithoutQuotes()
    {
        st.clearStatement();
        st.append(value.replace("'", "''"));
    }

    public Object getValue()
    {
        return value;
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, StringExpression.class);

        if (expr instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.equals(((StringLiteral)expr).value));
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, StringExpression.class);

        if (expr instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, !value.equals(((StringLiteral)expr).value));
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((StringLiteral)expr).value) < 0);
        }
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((StringLiteral)expr).value) <= 0);
        }
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((StringLiteral)expr).value) > 0);
        }
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((StringLiteral)expr).value) >= 0);
        }
        else
        {
            return super.gteq(expr);
        }
    }

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            return new StringLiteral(qs, mapping, value.concat(((StringLiteral)expr).value));
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new StringLiteral(qs, mapping, value.concat(((Literal)expr).getValue().toString()));
        }
        else if (expr instanceof IntegerLiteral || expr instanceof FloatingPointLiteral || expr instanceof BooleanLiteral )
        {
            return new StringLiteral(qs, mapping, value.concat(((Literal)expr).getValue().toString()));
        }        
        else
        {
            return super.add(expr);
        }
    }

    public BooleanExpression endsWithMethod(ScalarExpression str)
    {
        if (str instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.endsWith(((StringLiteral)str).value));
        }
        else
        {
            return super.endsWithMethod(str);
        }
    }

    public NumericExpression indexOfMethod(ScalarExpression str)
    {
        if (str instanceof StringLiteral)
        {
            return new IntegerLiteral(qs, mapping, BigInteger.valueOf(value.indexOf(((StringLiteral)str).value)));
        }
        else
        {
            return super.indexOfMethod(str);
        }
    }

    public NumericExpression lengthMethod()
    {
        return new IntegerLiteral(qs, mapping, BigInteger.valueOf(value.length()));
    }

    public BooleanExpression startsWithMethod(ScalarExpression str)
    {
        if (str instanceof StringLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.startsWith(((StringLiteral)str).value));
        }
        else
        {
            return super.startsWithMethod(str);
        }
    }

    public BooleanExpression startsWithMethod(ScalarExpression str, ScalarExpression toffset)
    {
        if (str instanceof StringLiteral && toffset instanceof IntegerLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.startsWith(((StringLiteral)str).value,
                            ((Number) ((IntegerLiteral)toffset).getValue()).intValue()));
        }
        else
        {
            return super.startsWithMethod(str, toffset);
        }
    }

    /**
     * Method to handle the substring operation.
     * @param begin The start position
     * @return The expression.
     **/    
    public StringExpression substringMethod(NumericExpression begin)
    {
        if (begin instanceof IntegerLiteral)
        {
            return new StringLiteral(qs, mapping, value.substring(((Number)((IntegerLiteral)begin).getValue()).intValue()));
        }
        else
        {
            return super.substringMethod(begin);
        }
    }

    /**
     * Method to handle the substring operation.
     * @param begin The start position
     * @param end The end position
     * @return The expression.
     **/    
    public StringExpression substringMethod(NumericExpression begin, NumericExpression end)
    {
        if (begin instanceof IntegerLiteral && end instanceof IntegerLiteral)
        {
            return new StringLiteral(qs, mapping, value.substring(((Number)((IntegerLiteral)begin).getValue()).intValue(),
                									     ((Number)((IntegerLiteral)end).getValue()).intValue()));
        }
        else
        {
            return super.substringMethod(begin, end);
        }
    }

    /**
     * Method to handle the lower case operation.
     * @return The expression.
     **/    
    public StringExpression toLowerCaseMethod()
    {
        return new StringLiteral(qs, mapping, value.toLowerCase());
    }

    /**
     * Method to handle the upper case operation.
     * @return The expression.
     **/    
    public StringExpression toUpperCaseMethod()
    {
        return new StringLiteral(qs, mapping, value.toUpperCase());
    }

    /**
     * Method to handle trimming of a string.
     * @return The expression.
     */    
    public StringExpression trimMethod()
    {
        return new StringLiteral(qs, mapping, value.trim());
    }
    
    public StatementText toStatementText(int mode)
    {
        if (mode == ScalarExpression.FILTER)
        {
            return stUsingParameter;
        }
        return super.toStatementText(mode);
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    {
        this.rawValue = val;
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return rawValue;
    }
}