/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.cbi.common.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.UnsignedInteger;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.util.Enumeration;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipException;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.jar.JarArchiveEntry;
import org.apache.commons.compress.archivers.jar.JarArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.eclipse.cbi.common.util.MorePosixFilePermissions;

public class Zips {
    private static final int PERM_MASK = 511;
    private static final String ZIP_ENTRY_NAME_SEPARATOR = "/";
    private static final String BACKSLASH_ESCAPE_REPLACEMENT = "\\\\\\\\";
    private static final Pattern BACKSLASH_PATTERN = Pattern.compile("\\\\");

    public static int unpackZip(Path source, Path outputDir) throws IOException {
        Zips.checkPathExists(source, "'source' path must exists");
        try (ZipFile zipFile = new ZipFile(Files.newByteChannel(source, new OpenOption[0]));){
            int n = Zips.unpack(zipFile, outputDir, Zips::fixPosixPermissions);
            return n;
        }
    }

    private static void fixPosixPermissions(ZipArchiveEntry entry, Path entryPath) {
        PosixFileAttributeView attributes = Files.getFileAttributeView(entryPath, PosixFileAttributeView.class, new LinkOption[0]);
        if (attributes != null) {
            try {
                Files.setPosixFilePermissions(entryPath, MorePosixFilePermissions.fromFileMode(entry.getUnixMode() & 0x1FF));
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private static int unpack(ZipFile zipFile, Path outputDir, BiConsumer<ZipArchiveEntry, Path> entryFixer) throws IOException, ZipException {
        int unpack = 0;
        Enumeration entries = zipFile.getEntries();
        while (entries.hasMoreElements()) {
            ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
            Path entryPath = Zips.unpackZipEntry(zipFile, entry, outputDir);
            if (entryFixer != null) {
                entryFixer.accept(entry, entryPath);
            }
            ++unpack;
        }
        return unpack;
    }

    private static Path unpackZipEntry(ZipFile zipFile, ZipArchiveEntry entry, Path outputDir) throws IOException, ZipException {
        Path entryPath = outputDir.resolve(entry.getName());
        if (entry.isDirectory()) {
            Files.createDirectories(entryPath, new FileAttribute[0]);
        } else {
            Path parentPath = entryPath.normalize().getParent();
            Files.createDirectories(parentPath, new FileAttribute[0]);
            Files.copy(zipFile.getInputStream(entry), entryPath, StandardCopyOption.REPLACE_EXISTING);
        }
        Files.setLastModifiedTime(entryPath, FileTime.from(entry.getTime(), TimeUnit.MILLISECONDS));
        return entryPath;
    }

    public static int unpackJar(Path source, Path outputDir) throws IOException {
        Zips.checkPathExists(source, "'source' path must exists");
        try (ZipFile zipFile = new ZipFile(Files.newByteChannel(source, new OpenOption[0]));){
            int n = Zips.unpack(zipFile, outputDir, null);
            return n;
        }
    }

    public static int unpackTarGz(Path sourcePath, Path outputDir) throws IOException {
        try (TarArchiveInputStream tarArchiveInputStream = new TarArchiveInputStream((InputStream)new GZIPInputStream(Files.newInputStream(sourcePath, new OpenOption[0])));){
            int n = Zips.unpack(tarArchiveInputStream, outputDir);
            return n;
        }
    }

    @VisibleForTesting
    static int unpack(TarArchiveInputStream zis, Path outputDir) throws IOException {
        int unpackedEntries = 0;
        TarArchiveEntry entry = zis.getNextTarEntry();
        while (entry != null) {
            Path entryPath = outputDir.resolve(entry.getName());
            if (entry.isDirectory()) {
                Files.createDirectories(entryPath, new FileAttribute[0]);
            } else if (entry.isLink()) {
                Files.createLink(entryPath, outputDir.resolve(entry.getLinkName()));
            } else if (entry.isSymbolicLink()) {
                Files.createSymbolicLink(entryPath, outputDir.resolve(entry.getLinkName()), new FileAttribute[0]);
            } else if (entry.isFile()) {
                Path parentPath = entryPath.normalize().getParent();
                Files.createDirectories(parentPath, new FileAttribute[0]);
                Files.copy((InputStream)zis, entryPath, StandardCopyOption.REPLACE_EXISTING);
            } else {
                throw new IOException("Type of a Tar entry is not supported");
            }
            if (!Files.isSymbolicLink(entryPath)) {
                Zips.setPermissions(entry, entryPath);
                Files.setLastModifiedTime(entryPath, FileTime.from(entry.getLastModifiedDate().getTime(), TimeUnit.MILLISECONDS));
            }
            ++unpackedEntries;
            entry = zis.getNextTarEntry();
        }
        return unpackedEntries;
    }

    private static void setPermissions(TarArchiveEntry entry, Path entryPath) throws IOException {
        PosixFileAttributeView attributes = Files.getFileAttributeView(entryPath, PosixFileAttributeView.class, new LinkOption[0]);
        if (attributes != null) {
            attributes.setPermissions(MorePosixFilePermissions.fromFileMode(entry.getMode()));
        }
    }

    public static int packZip(Path source, Path targetZip, boolean preserveRoot) throws IOException {
        Zips.checkPathExists(source, "'source' path must exists");
        try (ZipArchiveOutputStream zos = new ZipArchiveOutputStream((OutputStream)Zips.newBufferedOutputStream(targetZip));){
            int n = Zips.packEntries(source, zos, preserveRoot, (Set<Path>)ImmutableSet.of());
            return n;
        }
    }

    public static int packJar(Path source, Path targetJar, boolean preserveRoot) throws IOException {
        Zips.checkPathExists(source, "'source' path must exists");
        try (JarArchiveOutputStream jos = new JarArchiveOutputStream((OutputStream)Zips.newBufferedOutputStream(targetJar));){
            Set<Path> pathToExcludes = !preserveRoot ? Zips.packManifestIfAny(source, jos) : ImmutableSet.of();
            int n = Zips.packEntries(source, (ZipArchiveOutputStream)jos, preserveRoot, pathToExcludes) + pathToExcludes.size();
            return n;
        }
    }

    private static Set<Path> packManifestIfAny(Path source, JarArchiveOutputStream jos) throws IOException {
        ImmutableSet ret;
        if (Files.isDirectory(source, new LinkOption[0])) {
            Path metaInf = source.resolve("META-INF");
            Path manifest = metaInf.resolve("MANIFEST.MF");
            if (Files.exists(manifest, new LinkOption[0])) {
                Zips.putDirectoryEntry(metaInf, (ZipArchiveOutputStream)jos, source.relativize(metaInf));
                Zips.putFileEntry(manifest, (ZipArchiveOutputStream)jos, source.relativize(manifest));
                ret = ImmutableSet.of((Object)metaInf, (Object)manifest);
            } else {
                ret = ImmutableSet.of();
            }
        } else {
            ret = ImmutableSet.of();
        }
        return ret;
    }

    private static Path checkPathExists(Path source, String msg) {
        if (!Files.exists(source, new LinkOption[0])) {
            throw new IllegalArgumentException(msg);
        }
        return source;
    }

    private static BufferedOutputStream newBufferedOutputStream(Path path) throws IOException {
        return new BufferedOutputStream(Files.newOutputStream(path, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING));
    }

    private static int packEntries(Path source, ZipArchiveOutputStream zos, boolean preserveRoot, Set<Path> pathToExcludes) throws IOException {
        if (Files.isDirectory(source, new LinkOption[0])) {
            PathMapper pathMapper = preserveRoot ? new PreserveRootPathMapper(source) : new NoPreserveRootPathMapper(source);
            PackerFileVisitor packerFileVisitor = new PackerFileVisitor(zos, pathMapper, pathToExcludes);
            Files.walkFileTree(source, packerFileVisitor);
            return packerFileVisitor.packedEntries();
        }
        Zips.putFileEntry(source, zos, source.getFileName());
        return 1;
    }

    private static String entryNameFrom(Path path, boolean isDirectoryw) {
        String escapedEntryName;
        String pathFsSeparator = path.getFileSystem().getSeparator();
        if (pathFsSeparator != ZIP_ENTRY_NAME_SEPARATOR) {
            String separatorRegex = BACKSLASH_PATTERN.matcher(pathFsSeparator).replaceAll(BACKSLASH_ESCAPE_REPLACEMENT);
            escapedEntryName = path.toString().replaceAll(separatorRegex, ZIP_ENTRY_NAME_SEPARATOR);
        } else {
            escapedEntryName = path.toString();
        }
        if (isDirectoryw && !escapedEntryName.endsWith(ZIP_ENTRY_NAME_SEPARATOR)) {
            return escapedEntryName + ZIP_ENTRY_NAME_SEPARATOR;
        }
        return escapedEntryName;
    }

    private static void putFileEntry(Path file, ZipArchiveOutputStream zos, Path entryPath) throws IOException {
        ZipArchiveEntry zipEntry = Zips.createArchiveEntry(zos, Zips.entryNameFrom(entryPath, false));
        zipEntry.setTime(Files.getLastModifiedTime(file, new LinkOption[0]).toMillis());
        zipEntry.setSize(Files.size(file));
        PosixFileAttributeView posixFileAttributeView = Files.getFileAttributeView(file, PosixFileAttributeView.class, new LinkOption[0]);
        if (posixFileAttributeView != null) {
            PosixFileAttributes posixFileAttributes = posixFileAttributeView.readAttributes();
            zipEntry.setUnixMode(UnsignedInteger.valueOf((long)MorePosixFilePermissions.toFileMode(posixFileAttributes.permissions())).intValue());
        }
        zos.putArchiveEntry((ArchiveEntry)zipEntry);
        Files.copy(file, (OutputStream)zos);
        zos.closeArchiveEntry();
    }

    private static void putDirectoryEntry(Path dir, ZipArchiveOutputStream zos, Path entryPath) throws IOException {
        ZipArchiveEntry zipEntry = Zips.createArchiveEntry(zos, Zips.entryNameFrom(entryPath, true));
        zipEntry.setTime(Files.getLastModifiedTime(dir, new LinkOption[0]).toMillis());
        PosixFileAttributeView posixFileAttributeView = Files.getFileAttributeView(dir, PosixFileAttributeView.class, new LinkOption[0]);
        if (posixFileAttributeView != null) {
            PosixFileAttributes posixFileAttributes = posixFileAttributeView.readAttributes();
            zipEntry.setUnixMode(UnsignedInteger.valueOf((long)MorePosixFilePermissions.toFileMode(posixFileAttributes.permissions())).intValue());
        }
        zos.putArchiveEntry((ArchiveEntry)zipEntry);
        zos.closeArchiveEntry();
    }

    private static ZipArchiveEntry createArchiveEntry(ZipArchiveOutputStream zos, String entryName) {
        if (zos instanceof JarArchiveOutputStream) {
            return new JarArchiveEntry(entryName);
        }
        return new ZipArchiveEntry(entryName);
    }

    private static final class PackerFileVisitor
    extends SimpleFileVisitor<Path> {
        private final PathMapper pathMapper;
        private final ZipArchiveOutputStream zos;
        private final Set<Path> pathToExcludes;
        private int packedEntries;

        private PackerFileVisitor(ZipArchiveOutputStream zos, PathMapper pathMapper, Set<Path> pathToExcludes) {
            this.pathMapper = pathMapper;
            this.zos = zos;
            this.pathToExcludes = pathToExcludes;
        }

        int packedEntries() {
            return this.packedEntries;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (!this.pathToExcludes.contains(dir)) {
                Path entryPath = this.pathMapper.mapTo(dir);
                if (!Strings.isNullOrEmpty((String)entryPath.toString())) {
                    Zips.putDirectoryEntry(dir, this.zos, entryPath);
                    ++this.packedEntries;
                }
            } else {
                return FileVisitResult.CONTINUE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (!this.pathToExcludes.contains(file)) {
                Path entryPath = this.pathMapper.mapTo(file);
                Zips.putFileEntry(file, this.zos, entryPath);
                ++this.packedEntries;
            }
            return FileVisitResult.CONTINUE;
        }
    }

    private static final class PreserveRootPathMapper
    implements PathMapper {
        private final Path source;

        private PreserveRootPathMapper(Path source) {
            this.source = source;
        }

        @Override
        public Path mapTo(Path path) {
            return this.source.getFileName().resolve(this.source.relativize(path));
        }
    }

    private static final class NoPreserveRootPathMapper
    implements PathMapper {
        private final Path source;

        private NoPreserveRootPathMapper(Path source) {
            this.source = source;
        }

        @Override
        public Path mapTo(Path path) {
            return this.source.relativize(path);
        }
    }

    private static interface PathMapper {
        public Path mapTo(Path var1);
    }
}

