/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.io;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.WritePendingException;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public abstract class WriteFlusher {
    private static final Logger LOG = Log.getLogger(WriteFlusher.class);
    private static final boolean DEBUG = LOG.isDebugEnabled();
    private static final ByteBuffer[] EMPTY_BUFFERS = new ByteBuffer[0];
    private static final EnumMap<StateType, Set<StateType>> __stateTransitions = new EnumMap(StateType.class);
    private static final State __IDLE = new IdleState();
    private static final State __WRITING = new WritingState();
    private static final State __COMPLETING = new CompletingState();
    private final EndPoint _endPoint;
    private final AtomicReference<State> _state = new AtomicReference();

    protected WriteFlusher(EndPoint endPoint) {
        this._state.set(__IDLE);
        this._endPoint = endPoint;
    }

    private boolean updateState(State previous, State next) {
        if (!this.isTransitionAllowed(previous, next)) {
            throw new IllegalStateException();
        }
        boolean updated = this._state.compareAndSet(previous, next);
        if (DEBUG) {
            LOG.debug("update {}:{}{}{}", new Object[]{this, previous, updated ? "-->" : "!->", next});
        }
        return updated;
    }

    private void fail(PendingState pending) {
        FailedState failed;
        State current = this._state.get();
        if (current.getType() == StateType.FAILED && this.updateState(failed = (FailedState)current, __IDLE)) {
            pending.fail(failed.getCause());
            return;
        }
        throw new IllegalStateException();
    }

    private void ignoreFail() {
        State current = this._state.get();
        while (current.getType() == StateType.FAILED) {
            if (this.updateState(current, __IDLE)) {
                return;
            }
            current = this._state.get();
        }
    }

    private boolean isTransitionAllowed(State currentState, State newState) {
        Set<StateType> allowedNewStateTypes = __stateTransitions.get((Object)currentState.getType());
        if (!allowedNewStateTypes.contains((Object)newState.getType())) {
            LOG.warn("{}: {} -> {} not allowed", new Object[]{this, currentState, newState});
            return false;
        }
        return true;
    }

    protected abstract void onIncompleteFlushed();

    public void write(Callback callback, ByteBuffer ... buffers) throws WritePendingException {
        if (DEBUG) {
            LOG.debug("write: {} {}", new Object[]{this, BufferUtil.toDetailString((ByteBuffer[])buffers)});
        }
        if (!this.updateState(__IDLE, __WRITING)) {
            throw new WritePendingException();
        }
        try {
            buffers = this.flush(buffers);
            if (buffers != null) {
                PendingState pending;
                if (DEBUG) {
                    LOG.debug("flushed incomplete", new Object[0]);
                }
                if (this.updateState(__WRITING, pending = new PendingState(buffers, callback))) {
                    this.onIncompleteFlushed();
                } else {
                    this.fail(pending);
                }
                return;
            }
            if (!this.updateState(__WRITING, __IDLE)) {
                this.ignoreFail();
            }
            if (callback != null) {
                callback.succeeded();
            }
        }
        catch (IOException e) {
            if (DEBUG) {
                LOG.debug("write exception", (Throwable)e);
            }
            if (this.updateState(__WRITING, __IDLE)) {
                if (callback != null) {
                    callback.failed((Throwable)e);
                }
            }
            this.fail(new PendingState(buffers, callback));
        }
    }

    public void completeWrite() {
        State previous;
        if (DEBUG) {
            LOG.debug("completeWrite: {}", new Object[]{this});
        }
        if ((previous = this._state.get()).getType() != StateType.PENDING) {
            return;
        }
        PendingState pending = (PendingState)previous;
        if (!this.updateState(pending, __COMPLETING)) {
            return;
        }
        try {
            ByteBuffer[] buffers = pending.getBuffers();
            buffers = this.flush(buffers);
            if (buffers != null) {
                if (DEBUG) {
                    LOG.debug("flushed incomplete {}", new Object[]{BufferUtil.toDetailString((ByteBuffer[])buffers)});
                }
                if (buffers != pending.getBuffers()) {
                    pending = new PendingState(buffers, pending._callback);
                }
                if (this.updateState(__COMPLETING, pending)) {
                    this.onIncompleteFlushed();
                } else {
                    this.fail(pending);
                }
                return;
            }
            if (!this.updateState(__COMPLETING, __IDLE)) {
                this.ignoreFail();
            }
            pending.complete();
        }
        catch (IOException e) {
            if (DEBUG) {
                LOG.debug("completeWrite exception", (Throwable)e);
            }
            if (this.updateState(__COMPLETING, __IDLE)) {
                pending.fail(e);
            }
            this.fail(pending);
        }
    }

    protected ByteBuffer[] flush(ByteBuffer[] buffers) throws IOException {
        if (this._endPoint.flush(buffers)) {
            return null;
        }
        boolean progress = true;
        while (true) {
            int not_empty;
            for (not_empty = 0; not_empty < buffers.length && BufferUtil.isEmpty((ByteBuffer)buffers[not_empty]); ++not_empty) {
            }
            if (not_empty == buffers.length) {
                return null;
            }
            if (not_empty > 0) {
                buffers = Arrays.copyOfRange(buffers, not_empty, buffers.length);
            }
            if (!progress) break;
            int r = buffers[0].remaining();
            if (this._endPoint.flush(buffers)) {
                return null;
            }
            progress = r != buffers[0].remaining();
        }
        return buffers;
    }

    public boolean onFail(Throwable cause) {
        block4: while (true) {
            State current = this._state.get();
            switch (current.getType()) {
                case IDLE: 
                case FAILED: {
                    if (DEBUG) {
                        LOG.debug("ignored: {} {}", new Object[]{this, cause});
                    }
                    return false;
                }
                case PENDING: {
                    PendingState pending;
                    if (DEBUG) {
                        LOG.debug("failed: {} {}", new Object[]{this, cause});
                    }
                    if (!this.updateState(pending = (PendingState)current, __IDLE)) continue block4;
                    return pending.fail(cause);
                }
            }
            if (DEBUG) {
                LOG.debug("failed: {} {}", new Object[]{this, cause});
            }
            if (this.updateState(current, new FailedState(cause))) break;
        }
        return false;
    }

    public void onClose() {
        if (this._state.get() == __IDLE) {
            return;
        }
        this.onFail(new ClosedChannelException());
    }

    boolean isIdle() {
        return this._state.get().getType() == StateType.IDLE;
    }

    public boolean isInProgress() {
        switch (this._state.get().getType()) {
            case PENDING: 
            case WRITING: 
            case COMPLETING: {
                return true;
            }
        }
        return false;
    }

    public String toString() {
        return String.format("WriteFlusher@%x{%s}", this.hashCode(), this._state.get());
    }

    static {
        __stateTransitions.put(StateType.IDLE, EnumSet.of(StateType.WRITING));
        __stateTransitions.put(StateType.WRITING, EnumSet.of(StateType.IDLE, StateType.PENDING, StateType.FAILED));
        __stateTransitions.put(StateType.PENDING, EnumSet.of(StateType.COMPLETING, StateType.IDLE));
        __stateTransitions.put(StateType.COMPLETING, EnumSet.of(StateType.IDLE, StateType.PENDING, StateType.FAILED));
        __stateTransitions.put(StateType.FAILED, EnumSet.of(StateType.IDLE));
    }

    private class PendingState
    extends State {
        private final Callback _callback;
        private final ByteBuffer[] _buffers;

        private PendingState(ByteBuffer[] buffers, Callback callback) {
            super(StateType.PENDING);
            this._buffers = this.compact(buffers);
            this._callback = callback;
        }

        public ByteBuffer[] getBuffers() {
            return this._buffers;
        }

        protected boolean fail(Throwable cause) {
            if (this._callback != null) {
                this._callback.failed(cause);
                return true;
            }
            return false;
        }

        protected void complete() {
            if (this._callback != null) {
                this._callback.succeeded();
            }
        }

        private ByteBuffer[] compact(ByteBuffer[] buffers) {
            int consumed;
            int length = buffers.length;
            if (length < 2) {
                return buffers;
            }
            for (consumed = 0; consumed < length && BufferUtil.isEmpty((ByteBuffer)buffers[consumed]); ++consumed) {
            }
            if (consumed == 0) {
                return buffers;
            }
            if (consumed == length) {
                return EMPTY_BUFFERS;
            }
            return Arrays.copyOfRange(buffers, consumed, length);
        }
    }

    private static class CompletingState
    extends State {
        private CompletingState() {
            super(StateType.COMPLETING);
        }
    }

    private static class FailedState
    extends State {
        private final Throwable _cause;

        private FailedState(Throwable cause) {
            super(StateType.FAILED);
            this._cause = cause;
        }

        public Throwable getCause() {
            return this._cause;
        }
    }

    private static class WritingState
    extends State {
        private WritingState() {
            super(StateType.WRITING);
        }
    }

    private static class IdleState
    extends State {
        private IdleState() {
            super(StateType.IDLE);
        }
    }

    private static class State {
        private final StateType _type;

        private State(StateType stateType) {
            this._type = stateType;
        }

        public StateType getType() {
            return this._type;
        }

        public String toString() {
            return String.format("%s", new Object[]{this._type});
        }
    }

    private static enum StateType {
        IDLE,
        WRITING,
        PENDING,
        COMPLETING,
        FAILED;

    }
}

