/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.application;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.security.KeyPair;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.eclipse.milo.opcua.stack.core.application.CertificateManager;
import org.eclipse.milo.opcua.stack.core.types.builtin.ByteString;
import org.eclipse.milo.opcua.stack.core.util.DigestUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCertificateManager
implements CertificateManager {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Map<ByteString, KeyPair> privateKeys = Maps.newConcurrentMap();
    private final Map<ByteString, X509Certificate[]> certificates = Maps.newConcurrentMap();

    public DefaultCertificateManager() {
    }

    public DefaultCertificateManager(KeyPair privateKey, X509Certificate certificate) {
        Preconditions.checkNotNull((Object)privateKey, (Object)"privateKey must be non-null");
        Preconditions.checkNotNull((Object)certificate, (Object)"certificate must be non-null");
        this.add(privateKey, certificate);
    }

    public DefaultCertificateManager(KeyPair privateKey, X509Certificate[] certificateChain) {
        Preconditions.checkNotNull((Object)privateKey, (Object)"privateKey must be non-null");
        Preconditions.checkNotNull((Object)certificateChain, (Object)"certificateChain must be non-null");
        this.add(privateKey, certificateChain);
    }

    public DefaultCertificateManager(List<KeyPair> privateKeys, List<X509Certificate> certificates) {
        Preconditions.checkState((privateKeys.size() == certificates.size() ? 1 : 0) != 0, (Object)"privateKeys.size() and certificates.size() must be equal");
        for (int i = 0; i < privateKeys.size(); ++i) {
            KeyPair privateKey = privateKeys.get(i);
            X509Certificate certificate = certificates.get(i);
            this.add(privateKey, certificate);
        }
    }

    public void add(KeyPair privateKey, X509Certificate certificate) {
        this.add(privateKey, new X509Certificate[]{certificate});
    }

    public void add(KeyPair privateKey, X509Certificate[] certificateChain) {
        Preconditions.checkNotNull((Object)privateKey, (Object)"privateKey must be non-null");
        Preconditions.checkNotNull((Object)certificateChain, (Object)"certificateChain must be non-null");
        try {
            X509Certificate certificate = certificateChain[0];
            ByteString thumbprint = ByteString.of(DigestUtil.sha1(certificate.getEncoded()));
            this.privateKeys.put(thumbprint, privateKey);
            this.certificates.put(thumbprint, certificateChain);
        }
        catch (CertificateEncodingException e) {
            this.logger.error("Error getting certificate thumbprint.", (Throwable)e);
        }
    }

    @Override
    public Optional<KeyPair> getKeyPair(ByteString thumbprint) {
        return Optional.ofNullable(this.privateKeys.get(thumbprint));
    }

    @Override
    public Optional<X509Certificate> getCertificate(ByteString thumbprint) {
        X509Certificate[] chain = this.certificates.get(thumbprint);
        if (chain != null && chain.length > 0) {
            return Optional.of(chain[0]);
        }
        return Optional.empty();
    }

    @Override
    public Optional<X509Certificate[]> getCertificateChain(ByteString thumbprint) {
        return Optional.ofNullable(this.certificates.get(thumbprint));
    }

    @Override
    public Set<KeyPair> getKeyPairs() {
        return Sets.newHashSet(this.privateKeys.values());
    }

    @Override
    public Set<X509Certificate> getCertificates() {
        return this.certificates.values().stream().map(a -> a[0]).collect(Collectors.toSet());
    }
}

