/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.application;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.security.cert.CRL;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.application.CertificateValidator;
import org.eclipse.milo.opcua.stack.core.util.CertificateUtil;
import org.eclipse.milo.opcua.stack.core.util.CertificateValidationUtil;
import org.eclipse.milo.opcua.stack.core.util.DigestUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectoryCertificateValidator
implements CertificateValidator,
AutoCloseable {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Set<X509Certificate> trustedCertificates = Sets.newConcurrentHashSet();
    private final Set<X509Certificate> issuerCertificates = Sets.newConcurrentHashSet();
    private final Set<X509CRL> issuerCrls = Sets.newConcurrentHashSet();
    private final WatchService watchService;
    private final Thread watchThread;
    private final File baseDir;
    private final File issuerDir;
    private final File issuerCertsDir;
    private final File issuerCrlsDir;
    private final File trustedDir;
    private final File trustedCertsDir;
    private final File rejectedDir;

    public DirectoryCertificateValidator(File baseDir) throws IOException {
        this.baseDir = baseDir;
        DirectoryCertificateValidator.ensureDirectoryExists(baseDir);
        this.issuerDir = baseDir.toPath().resolve("issuers").toFile();
        DirectoryCertificateValidator.ensureDirectoryExists(this.issuerDir);
        this.issuerCertsDir = this.issuerDir.toPath().resolve("certs").toFile();
        DirectoryCertificateValidator.ensureDirectoryExists(this.issuerCertsDir);
        this.issuerCrlsDir = this.issuerDir.toPath().resolve("crls").toFile();
        DirectoryCertificateValidator.ensureDirectoryExists(this.issuerCrlsDir);
        this.trustedDir = baseDir.toPath().resolve("trusted").toFile();
        DirectoryCertificateValidator.ensureDirectoryExists(this.trustedDir);
        this.trustedCertsDir = this.trustedDir.toPath().resolve("certs").toFile();
        DirectoryCertificateValidator.ensureDirectoryExists(this.trustedCertsDir);
        this.rejectedDir = baseDir.toPath().resolve("rejected").toFile();
        DirectoryCertificateValidator.ensureDirectoryExists(this.rejectedDir);
        this.watchService = FileSystems.getDefault().newWatchService();
        ConcurrentMap watchKeys = Maps.newConcurrentMap();
        watchKeys.put(this.issuerCertsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeIssuerCerts);
        watchKeys.put(this.issuerCrlsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeIssuerCrls);
        watchKeys.put(this.trustedCertsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeTrustedCerts);
        this.watchThread = new Thread(new Watcher(this.watchService, watchKeys));
        this.watchThread.setName("certificate-store-watcher");
        this.watchThread.setDaemon(true);
        this.watchThread.start();
        this.synchronizeIssuerCerts();
        this.synchronizeIssuerCrls();
        this.synchronizeTrustedCerts();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        DirectoryCertificateValidator directoryCertificateValidator = this;
        synchronized (directoryCertificateValidator) {
            this.logger.info("Closing DefaultCertificateStore at {}", (Object)this.baseDir.getAbsolutePath());
            this.watchService.close();
            try {
                this.watchThread.join(5000L);
            }
            catch (InterruptedException e) {
                throw new IOException(e);
            }
            this.issuerCertificates.clear();
            this.issuerCrls.clear();
            this.trustedCertificates.clear();
        }
    }

    @Override
    public synchronized void validate(X509Certificate certificate) throws UaException {
        try {
            CertificateValidationUtil.validateCertificateValidity(certificate);
        }
        catch (UaException e) {
            this.addRejectedCertificate(certificate);
            throw e;
        }
    }

    @Override
    public synchronized void verifyTrustChain(List<X509Certificate> certificateChain) throws UaException {
        try {
            CertificateValidationUtil.verifyTrustChain(certificateChain, this.trustedCertificates, this.issuerCertificates, this.issuerCrls);
        }
        catch (UaException e) {
            X509Certificate certificate = certificateChain.get(0);
            this.addRejectedCertificate(certificate);
            throw e;
        }
    }

    public synchronized void addIssuerCertificate(X509Certificate certificate) {
        this.issuerCertificates.add(certificate);
        this.writeCertificateToDir(certificate, this.issuerCertsDir);
    }

    public synchronized void addTrustedCertificate(X509Certificate certificate) {
        this.trustedCertificates.add(certificate);
        this.writeCertificateToDir(certificate, this.trustedCertsDir);
    }

    public synchronized void addRejectedCertificate(X509Certificate certificate) {
        this.writeCertificateToDir(certificate, this.rejectedDir);
    }

    public synchronized ImmutableSet<X509Certificate> getIssuerCertificates() {
        return ImmutableSet.copyOf(this.issuerCertificates);
    }

    public synchronized ImmutableSet<CRL> getIssuerCrls() {
        return ImmutableSet.copyOf(this.issuerCrls);
    }

    public synchronized ImmutableSet<X509Certificate> getTrustedCertificates() {
        return ImmutableSet.copyOf(this.trustedCertificates);
    }

    public File getBaseDir() {
        return this.baseDir;
    }

    public File getIssuerDir() {
        return this.issuerDir;
    }

    public File getIssuerCertsDir() {
        return this.issuerCertsDir;
    }

    public File getIssuerCrlsDir() {
        return this.issuerCrlsDir;
    }

    public File getTrustedDir() {
        return this.trustedDir;
    }

    public File getTrustedCertsDir() {
        return this.trustedCertsDir;
    }

    public File getRejectedDir() {
        return this.rejectedDir;
    }

    private static String getFilename(X509Certificate certificate) throws Exception {
        String[] ss = certificate.getSubjectX500Principal().getName().split(",");
        String name = ss.length > 0 ? ss[0] : certificate.getSubjectX500Principal().getName();
        String thumbprint = ByteBufUtil.hexDump((ByteBuf)Unpooled.wrappedBuffer((byte[])DigestUtil.sha1(certificate.getEncoded())));
        return String.format("%s [%s].der", thumbprint, URLEncoder.encode(name, "UTF-8"));
    }

    private void writeCertificateToDir(X509Certificate certificate, File dir) {
        try {
            String filename = DirectoryCertificateValidator.getFilename(certificate);
            File f = new File(dir.getAbsolutePath() + File.separator + filename);
            try (FileOutputStream fos = new FileOutputStream(f);){
                fos.write(certificate.getEncoded());
                fos.flush();
            }
            this.logger.info("Wrote certificate entry: {}", (Object)f.getAbsolutePath());
        }
        catch (Exception e) {
            this.logger.error("Error adding rejected certificate entry.", (Throwable)e);
        }
    }

    private synchronized void synchronizeIssuerCerts() {
        File[] files = this.issuerCertsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.issuerCertificates.clear();
        Arrays.stream(files).flatMap(cert -> this.decodeCertificateFile((File)cert).map(Stream::of).orElse(Stream.empty())).forEach(this.issuerCertificates::add);
    }

    private synchronized void synchronizeTrustedCerts() {
        File[] files = this.trustedCertsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.trustedCertificates.clear();
        Arrays.stream(files).flatMap(cert -> this.decodeCertificateFile((File)cert).map(Stream::of).orElse(Stream.empty())).forEach(this.trustedCertificates::add);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<X509Certificate> decodeCertificateFile(File f) {
        try (FileInputStream inputStream = new FileInputStream(f);){
            Optional<X509Certificate> optional = Optional.of(CertificateUtil.decodeCertificate(inputStream));
            return optional;
        }
        catch (Throwable ignored) {
            return Optional.empty();
        }
    }

    private synchronized void synchronizeIssuerCrls() {
        File[] files = this.issuerCrlsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.issuerCrls.clear();
        Arrays.stream(files).flatMap(crl -> this.decodeCrlFile((File)crl).map(Stream::of).orElse(Stream.empty())).flatMap(Collection::stream).forEach(this.issuerCrls::add);
    }

    private Optional<List<X509CRL>> decodeCrlFile(File f) {
        try {
            CertificateFactory factory = CertificateFactory.getInstance("X.509");
            Collection<? extends CRL> crls = factory.generateCRLs(new FileInputStream(f));
            return Optional.of(crls.stream().filter(crl -> crl instanceof X509CRL).map(X509CRL.class::cast).collect(Collectors.toList()));
        }
        catch (FileNotFoundException | CRLException | CertificateException e) {
            return Optional.empty();
        }
    }

    private static void ensureDirectoryExists(File dir) throws IOException {
        if (!dir.exists() && !dir.mkdirs()) {
            throw new IOException("unable to create directory at " + dir.getAbsolutePath());
        }
    }

    private class Watcher
    implements Runnable {
        private final WatchService watchService;
        private final Map<WatchKey, Runnable> watchKeys;

        Watcher(WatchService watchService, Map<WatchKey, Runnable> watchKeys) {
            this.watchService = watchService;
            this.watchKeys = watchKeys;
        }

        @Override
        public void run() {
            while (true) {
                try {
                    WatchKey key;
                    do {
                        boolean synchronize;
                        if (!this.watchKeys.containsKey(key = this.watchService.take()) || !(synchronize = key.pollEvents().stream().anyMatch(e -> e.kind() != StandardWatchEventKinds.OVERFLOW))) continue;
                        this.watchKeys.get(key).run();
                    } while (key.reset());
                    DirectoryCertificateValidator.this.logger.warn("Failed to reset watch key");
                }
                catch (ClosedWatchServiceException e2) {
                    DirectoryCertificateValidator.this.logger.info("Watcher got closed");
                    return;
                }
                catch (InterruptedException e3) {
                    DirectoryCertificateValidator.this.logger.error("Watcher interrupted.", (Throwable)e3);
                    continue;
                }
                break;
            }
        }
    }
}

