/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.channel;

import io.netty.buffer.ByteBuf;
import io.netty.util.ReferenceCountUtil;
import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.channel.ChannelParameters;
import org.eclipse.milo.opcua.stack.core.channel.ChannelSecurity;
import org.eclipse.milo.opcua.stack.core.channel.SecureChannel;
import org.eclipse.milo.opcua.stack.core.channel.headers.AsymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SecureMessageHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SequenceHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.messages.MessageType;
import org.eclipse.milo.opcua.stack.core.security.SecurityAlgorithm;
import org.eclipse.milo.opcua.stack.core.util.BufferUtil;
import org.eclipse.milo.opcua.stack.core.util.LongSequence;
import org.eclipse.milo.opcua.stack.core.util.SignatureUtil;

public final class ChunkEncoder {
    private final AsymmetricEncoder asymmetricEncoder = new AsymmetricEncoder();
    private final SymmetricEncoder symmetricEncoder = new SymmetricEncoder();
    private final LongSequence sequenceNumber = new LongSequence(1L, 0xFFFFFBFFL);
    private final ChannelParameters parameters;

    public ChunkEncoder(ChannelParameters parameters) {
        this.parameters = parameters;
    }

    public void encodeAsymmetric(SecureChannel channel, long requestId, ByteBuf messageBuffer, MessageType messageType, Callback callback) {
        ChunkEncoder.encode(this.asymmetricEncoder, channel, requestId, messageBuffer, messageType, callback);
    }

    public void encodeSymmetric(SecureChannel channel, long requestId, ByteBuf messageBuffer, MessageType messageType, Callback callback) {
        ChunkEncoder.encode(this.symmetricEncoder, channel, requestId, messageBuffer, messageType, callback);
    }

    private static void encode(AbstractEncoder encoder, SecureChannel channel, long requestId, ByteBuf messageBuffer, MessageType messageType, Callback callback) {
        ArrayList<ByteBuf> chunks = new ArrayList<ByteBuf>();
        try {
            encoder.encode(chunks, channel, requestId, messageBuffer, messageType, callback);
        }
        catch (UaException e) {
            callback.onEncodingError(e);
            chunks.forEach(ReferenceCountUtil::safeRelease);
        }
    }

    private class SymmetricEncoder
    extends AbstractEncoder {
        private volatile ChannelSecurity.SecuritySecrets securitySecrets;

        private SymmetricEncoder() {
        }

        @Override
        public void encodeSecurityHeader(SecureChannel channel, ByteBuf buffer) {
            ChannelSecurity channelSecurity = channel.getChannelSecurity();
            long tokenId = channelSecurity != null ? channelSecurity.getCurrentToken().getTokenId().longValue() : 0L;
            SymmetricSecurityHeader.encode(new SymmetricSecurityHeader(tokenId), buffer);
            this.securitySecrets = channelSecurity != null ? channelSecurity.getCurrentKeys() : null;
        }

        @Override
        public byte[] signChunk(SecureChannel channel, ByteBuffer chunkNioBuffer) throws UaException {
            SecurityAlgorithm signatureAlgorithm = channel.getSecurityPolicy().getSymmetricSignatureAlgorithm();
            byte[] signatureKey = channel.getEncryptionKeys(this.securitySecrets).getSignatureKey();
            return SignatureUtil.hmac(signatureAlgorithm, signatureKey, chunkNioBuffer);
        }

        @Override
        public Cipher getAndInitializeCipher(SecureChannel channel) throws UaException {
            try {
                String transformation = channel.getSecurityPolicy().getSymmetricEncryptionAlgorithm().getTransformation();
                ChannelSecurity.SecretKeys secretKeys = channel.getEncryptionKeys(this.securitySecrets);
                SecretKeySpec keySpec = new SecretKeySpec(secretKeys.getEncryptionKey(), "AES");
                IvParameterSpec ivSpec = new IvParameterSpec(secretKeys.getInitializationVector());
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(1, (Key)keySpec, ivSpec);
                assert (cipher.getBlockSize() == channel.getSymmetricBlockSize());
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }

        @Override
        public int getSecurityHeaderSize(SecureChannel channel) {
            return 4;
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getSymmetricBlockSize();
        }

        @Override
        public int getPlainTextBlockSize(SecureChannel channel) {
            return channel.getSymmetricBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getSymmetricSignatureSize();
        }

        @Override
        protected boolean isAsymmetric() {
            return false;
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isSymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isSymmetricSigningEnabled();
        }
    }

    private class AsymmetricEncoder
    extends AbstractEncoder {
        private AsymmetricEncoder() {
        }

        @Override
        public byte[] signChunk(SecureChannel channel, ByteBuffer chunkNioBuffer) throws UaException {
            return SignatureUtil.sign(channel.getSecurityPolicy().getAsymmetricSignatureAlgorithm(), channel.getKeyPair().getPrivate(), chunkNioBuffer);
        }

        @Override
        public Cipher getAndInitializeCipher(SecureChannel channel) throws UaException {
            X509Certificate remoteCertificate = channel.getRemoteCertificate();
            assert (remoteCertificate != null);
            try {
                String transformation = channel.getSecurityPolicy().getAsymmetricEncryptionAlgorithm().getTransformation();
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(1, remoteCertificate.getPublicKey());
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
        }

        @Override
        public void encodeSecurityHeader(SecureChannel channel, ByteBuf buffer) throws UaException {
            AsymmetricSecurityHeader header = new AsymmetricSecurityHeader(channel.getSecurityPolicy().getSecurityPolicyUri(), channel.getLocalCertificateChainBytes(), channel.getRemoteCertificateThumbprint());
            AsymmetricSecurityHeader.encode(header, buffer);
        }

        @Override
        public int getSecurityHeaderSize(SecureChannel channel) throws UaException {
            String securityPolicyUri = channel.getSecurityPolicy().getSecurityPolicyUri();
            byte[] localCertificateChainBytes = channel.getLocalCertificateChainBytes().bytes();
            byte[] remoteCertificateThumbprint = channel.getRemoteCertificateThumbprint().bytes();
            return 12 + securityPolicyUri.length() + (localCertificateChainBytes != null ? localCertificateChainBytes.length : 0) + (remoteCertificateThumbprint != null ? remoteCertificateThumbprint.length : 0);
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getRemoteAsymmetricCipherTextBlockSize();
        }

        @Override
        public int getPlainTextBlockSize(SecureChannel channel) {
            return channel.getRemoteAsymmetricPlainTextBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getLocalAsymmetricSignatureSize();
        }

        @Override
        protected boolean isAsymmetric() {
            return true;
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isAsymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isAsymmetricSigningEnabled();
        }
    }

    private abstract class AbstractEncoder {
        private AbstractEncoder() {
        }

        void encode(List<ByteBuf> chunks, SecureChannel channel, long requestId, ByteBuf messageBuffer, MessageType messageType, Callback callback) throws UaException {
            boolean encrypted = this.isEncryptionEnabled(channel);
            int securityHeaderSize = this.getSecurityHeaderSize(channel);
            int cipherTextBlockSize = this.getCipherTextBlockSize(channel);
            int plainTextBlockSize = this.getPlainTextBlockSize(channel);
            int signatureSize = this.getSignatureSize(channel);
            int maxChunkSize = ChunkEncoder.this.parameters.getLocalSendBufferSize();
            int paddingOverhead = encrypted ? (cipherTextBlockSize > 256 ? 2 : 1) : 0;
            int maxCipherTextSize = maxChunkSize - 12 - securityHeaderSize;
            int maxCipherTextBlocks = maxCipherTextSize / cipherTextBlockSize;
            int maxPlainTextSize = maxCipherTextBlocks * plainTextBlockSize;
            int maxBodySize = maxPlainTextSize - 8 - paddingOverhead - signatureSize;
            assert (maxPlainTextSize + securityHeaderSize + 12 <= maxChunkSize);
            while (messageBuffer.readableBytes() > 0) {
                int plainTextSize;
                int remaining;
                int bodySize = Math.min(messageBuffer.readableBytes(), maxBodySize);
                int paddingSize = encrypted ? ((remaining = (plainTextSize = 8 + bodySize + paddingOverhead + signatureSize) % plainTextBlockSize) > 0 ? plainTextBlockSize - remaining : 0) : 0;
                int plainTextContentSize = 8 + bodySize + signatureSize + paddingSize + paddingOverhead;
                assert (plainTextContentSize % plainTextBlockSize == 0);
                int chunkSize = 12 + securityHeaderSize + plainTextContentSize / plainTextBlockSize * cipherTextBlockSize;
                assert (chunkSize <= maxChunkSize);
                ByteBuf chunkBuffer = BufferUtil.buffer(chunkSize);
                chunks.add(chunkBuffer);
                int remoteMaxChunkCount = ChunkEncoder.this.parameters.getRemoteMaxChunkCount();
                if (remoteMaxChunkCount > 0 && chunks.size() > remoteMaxChunkCount) {
                    throw new UaException(0x80080000L, "remote chunk count exceeded: " + remoteMaxChunkCount);
                }
                SecureMessageHeader messageHeader = new SecureMessageHeader(messageType, messageBuffer.readableBytes() > bodySize ? (char)'C' : 'F', chunkSize, channel.getChannelId());
                SecureMessageHeader.encode(messageHeader, chunkBuffer);
                this.encodeSecurityHeader(channel, chunkBuffer);
                SequenceHeader sequenceHeader = new SequenceHeader(ChunkEncoder.this.sequenceNumber.getAndIncrement(), requestId);
                SequenceHeader.encode(sequenceHeader, chunkBuffer);
                chunkBuffer.writeBytes(messageBuffer, bodySize);
                if (encrypted) {
                    this.writePadding(cipherTextBlockSize, paddingSize, chunkBuffer);
                }
                if (this.isSigningEnabled(channel)) {
                    ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(0, chunkBuffer.writerIndex());
                    byte[] signature = this.signChunk(channel, chunkNioBuffer);
                    chunkBuffer.writeBytes(signature);
                }
                if (encrypted) {
                    chunkBuffer.readerIndex(12 + securityHeaderSize);
                    assert (chunkBuffer.readableBytes() % plainTextBlockSize == 0);
                    try {
                        int blockCount = chunkBuffer.readableBytes() / plainTextBlockSize;
                        ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(chunkBuffer.readerIndex(), blockCount * cipherTextBlockSize);
                        ByteBuf copyBuffer = chunkBuffer.copy();
                        ByteBuffer plainTextNioBuffer = copyBuffer.nioBuffer();
                        Cipher cipher = this.getAndInitializeCipher(channel);
                        if (this.isAsymmetric()) {
                            for (int blockNumber = 0; blockNumber < blockCount; ++blockNumber) {
                                int position = blockNumber * plainTextBlockSize;
                                int limit = (blockNumber + 1) * plainTextBlockSize;
                                plainTextNioBuffer.position(position).limit(limit);
                                int bytesWritten = cipher.doFinal(plainTextNioBuffer, chunkNioBuffer);
                                assert (bytesWritten == cipherTextBlockSize);
                            }
                        } else {
                            cipher.doFinal(plainTextNioBuffer, chunkNioBuffer);
                        }
                        copyBuffer.release();
                    }
                    catch (GeneralSecurityException e) {
                        throw new UaException(2148728832L, (Throwable)e);
                    }
                }
                chunkBuffer.readerIndex(0).writerIndex(chunkSize);
            }
            callback.onMessageEncoded(chunks, requestId);
        }

        private void writePadding(int cipherTextBlockSize, int paddingSize, ByteBuf buffer) {
            if (cipherTextBlockSize > 256) {
                buffer.writeShort(paddingSize);
            } else {
                buffer.writeByte(paddingSize);
            }
            for (int i = 0; i < paddingSize; ++i) {
                buffer.writeByte(paddingSize);
            }
            if (cipherTextBlockSize > 256) {
                int paddingLengthMsb = paddingSize >> 8;
                buffer.writerIndex(buffer.writerIndex() - 1);
                buffer.writeByte(paddingLengthMsb);
            }
        }

        protected abstract byte[] signChunk(SecureChannel var1, ByteBuffer var2) throws UaException;

        protected abstract void encodeSecurityHeader(SecureChannel var1, ByteBuf var2) throws UaException;

        protected abstract Cipher getAndInitializeCipher(SecureChannel var1) throws UaException;

        protected abstract int getSecurityHeaderSize(SecureChannel var1) throws UaException;

        protected abstract int getCipherTextBlockSize(SecureChannel var1);

        protected abstract int getPlainTextBlockSize(SecureChannel var1);

        protected abstract int getSignatureSize(SecureChannel var1);

        protected abstract boolean isAsymmetric();

        protected abstract boolean isEncryptionEnabled(SecureChannel var1);

        protected abstract boolean isSigningEnabled(SecureChannel var1);
    }

    public static interface Callback {
        public void onEncodingError(UaException var1);

        public void onMessageEncoded(List<ByteBuf> var1, long var2);
    }
}

