package org.exoplatform.copyfolder.services;

import liquibase.repackaged.net.sf.jsqlparser.statement.create.procedure.CreateProcedure;
import org.apache.ecs.html.S;
import org.bouncycastle.cert.ocsp.Req;
import org.exoplatform.commons.utils.ListAccess;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.cms.documents.AutoVersionService;
import org.exoplatform.services.cms.mimetype.DMSMimeTypeResolver;
import org.exoplatform.services.jcr.access.AccessControlEntry;
import org.exoplatform.services.jcr.access.AccessControlList;
import org.exoplatform.services.jcr.access.PermissionType;
import org.exoplatform.services.jcr.config.RepositoryConfigurationException;
import org.exoplatform.services.jcr.core.ExtendedNode;
import org.exoplatform.services.jcr.core.ExtendedSession;
import org.exoplatform.services.jcr.core.ManageableRepository;
import org.exoplatform.services.jcr.ext.app.SessionProviderService;
import org.exoplatform.services.jcr.ext.common.SessionProvider;
import org.exoplatform.services.jcr.util.Text;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

import org.exoplatform.services.jcr.RepositoryService;
import org.exoplatform.ecm.webui.utils.Utils;
import org.exoplatform.services.organization.Group;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.security.ConversationState;
import org.exoplatform.services.security.IdentityConstants;
import org.exoplatform.services.wcm.core.NodetypeConstant;
import org.exoplatform.services.wcm.utils.WCMCoreUtils;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

public class CopyFolderService {
  private static final Log LOG = ExoLogger.getLogger(CopyFolderService.class);

  private static final String SOURCE_FOLDER_PARAM = "sourceFolder";
  private static final String TARGET_FOLDER_PARAM = "targetFolder";
  private static final String INDEX_FOLDER_PARAM = "index";
  private static final String INITIAL_LINE = "Folder;Group;Rights";

  private String targetFolder;
  private String                  sourceFolder;
  private String                  indexPath;
  private Map<String, List<Permission>> permissionsIndex = new HashMap<>();


  private RepositoryService repositoryService;

  private OrganizationService organizationService;
  private AutoVersionService autoVersionService;

  public CopyFolderService(InitParams initParams, RepositoryService repositoryService, OrganizationService organizationService, AutoVersionService autoVersionService) {
    if(initParams!=null) {
      if (initParams.getValueParam(SOURCE_FOLDER_PARAM)!=null) {
        sourceFolder=initParams.getValueParam(SOURCE_FOLDER_PARAM).getValue();
      }
      if (initParams.getValueParam(TARGET_FOLDER_PARAM)!=null) {
        targetFolder=initParams.getValueParam(TARGET_FOLDER_PARAM).getValue();
      }
      if (initParams.getValueParam(INDEX_FOLDER_PARAM)!=null) {
        indexPath = initParams.getValueParam(INDEX_FOLDER_PARAM).getValue();
      }
    }
    this.repositoryService = repositoryService;
    this.organizationService = organizationService;
    this.autoVersionService=autoVersionService;
  }

  private void readIndex(String indexPath) {
    LOG.debug("Start Read index");
    this.permissionsIndex = new HashMap<>();
    long startTime = System.currentTimeMillis();
    Path file = Path.of(indexPath);
    if (Files.exists(file)) {
      try (Stream<String> lines = Files.lines(file)) {
        lines.forEach(line -> {
          if (!line.equals(INITIAL_LINE) && !line.equals("")) {
            String[] splitted = line.split(";");
            if (splitted.length!=3) {
              LOG.error("Line {} have not the correct column number.",line);
            } else {
              String folder = splitted[0];
              String group = splitted[1];
              String permission = splitted[2];

              try {
                Permission perm = new Permission(group,AccessRight.valueOf(permission.toUpperCase()));
                LOG.debug("Add permission={} on folder={} in index",perm,folder);
                if (permissionsIndex.containsKey(folder)) {
                  permissionsIndex.get(folder).add(perm);
                } else {
                  List<Permission> perms = new ArrayList<>();
                  perms.add(perm);
                  permissionsIndex.put(folder,perms);
                }
              }catch (IllegalArgumentException e) {
                LOG.error("Line {} contains a non valid permission : {}",line,permission);
              }
            }
          }
        });
      } catch (IOException e) {
        LOG.error("Unable to read index file {}",indexPath);
      }
    }
    LOG.debug("PermissionsIndex : {}, build in {} ms",permissionsIndex.toString(), System.currentTimeMillis() - startTime);
  }

  public void importFromDisk() throws RuntimeException {
    if (targetFolder == null || sourceFolder == null || sourceFolder.isEmpty() || targetFolder.isEmpty()) {
      LOG.error("Unable to start the import from disk. Mandatory parameters are missing : targerFolder={}, sourceFolder={}",
                targetFolder,
                sourceFolder);
      throw new RuntimeException(
          "Mandatory parameters are missing. targetFolder = " + targetFolder + ", sourceFolder = " + sourceFolder);
    }
    long startTime = System.currentTimeMillis();
    LOG.info("Launch importFromDisk, from {} to {}",sourceFolder, targetFolder);
    Result result = new Result();
    readIndex(indexPath);

    try {
      RequestLifeCycle.begin(ExoContainerContext.getCurrentContainer());
      SessionProvider sessionProvider = new SessionProvider(ConversationState.getCurrent());
      Session session = sessionProvider.getSession("collaboration", repositoryService.getDefaultRepository());
      Node targetNode= getOrCreateTargetNode(session, targetFolder);

      Path path = Path.of(sourceFolder);
      if (Files.exists(path)) {
        //count items
        countItems(result, path);
        fillDirectory(targetNode, path, result);
      } else {
        LOG.error("Path source does not exists (sourceFolder={})",sourceFolder);
      }
      session.logout();
    } catch (RepositoryException|RepositoryConfigurationException e) {
      LOG.error("Error when getting eXo Repository",e);
    } finally {
      RequestLifeCycle.end();
    }


    LOG.info("End importFromDisk, execution time = {} ms. {}/{} directories created, {}/{} files created.",
             System.currentTimeMillis() - startTime,
             result.currentDirectory,
             result.nbDirectories,
             result.currentFiles,
             result.nbFiles);
    LOG.info("{} directories in error : {}", result.directoriesInError.size(), result.directoriesInError.toString());
    LOG.info("{} files in error : {}", result.filesInError.size(), result.filesInError.toString());
    LOG.info("Permissions with error : {}", result.permissionsWithError);
  }

  private void countItems(Result result, Path path) {
    LOG.debug("Start Count items");
    long startCount = System.currentTimeMillis();
    try (Stream<Path> stream = Files.walk(path)) {
      stream.forEach(path1 -> {
        LOG.debug("Count path={}",path1);
        (Files.isDirectory(path1) ? result.nbDirectories : result.nbFiles).getAndIncrement();
      });
    } catch (IOException e) {
      LOG.error("Unable to count folders and files.");
    }
    LOG.info("Count done in {} ms, {} directories, {} files", System.currentTimeMillis() - startCount, result.nbDirectories, result.nbFiles);

  }

  private void fillDirectory(Node targetNode, Path path, Result result) {
    try (var files = Files.list(path)) {
      files.forEach(path1 -> {
        String filename = path1.getFileName().toString();
        if (Files.isDirectory(path1)) {
          try {
            Node directoryNode = getOrCreateDirectoryChildNode(targetNode, path1, result);
            fillDirectory(directoryNode, path1, result);
          } catch (Exception e) {
            String targetNodePath = "";
            try {
              targetNode.getSession().refresh(false);
              targetNodePath = targetNode.getPath();
            } catch (RepositoryException re) {
              LOG.error("Unable to refresh session");
            }
            result.directoriesInError.add(path1.toString());
            LOG.error("Unable to create folder {} in node {}.", filename, targetNodePath, e);
          }
        } else {
          try {
            getOrCreateFileChildNode(targetNode, path1,result);
          } catch (Exception e) {
            String targetNodePath = "";
            try {
              targetNode.getSession().refresh(false);
              targetNodePath = targetNode.getPath();
            } catch (RepositoryException re) {
              LOG.error("Unable to refresh session");
            }
            result.filesInError.add(path1.toString());
            LOG.error("Unable to create file {} in node {}.", filename, targetNodePath, e);
          }
        }
        if ((result.currentFiles+ result.currentDirectory) % 100 == 0) {
          LOG.info("Directories {}% ({}/{}), Files {}% ({}/{}).",
                   (result.currentDirectory * 100) / result.nbDirectories.intValue(),
                   result.currentDirectory,
                   result.nbDirectories,
                   (result.currentFiles * 100) / result.nbFiles.intValue(),
                   result.currentFiles,
                   result.nbFiles);
        }
      });
    } catch (IOException e) {
      LOG.error("Unable to read sourceFolder {}",sourceFolder,e);
    }

  }

  private Node getOrCreateFileChildNode(Node targetNode, Path path, Result result) throws Exception {
    String filename = path.getFileName().toString();
    String cleanedFileName = Text.escapeIllegalJcrChars(org.exoplatform.services.cms.impl.Utils.cleanString(filename));
    if (targetNode.hasNode(cleanedFileName)) {
      Node file = targetNode.getNode(cleanedFileName);
      Instant lastModifiedInExo = file.getProperty("exo:lastModifiedDate").getDate().getTime().toInstant();
      BasicFileAttributes attr = Files.readAttributes(path, BasicFileAttributes.class);
      Instant lastModifiedInFS = attr.lastModifiedTime().toInstant();

      if (lastModifiedInExo.isBefore(lastModifiedInFS)) {
        LOG.debug("File {}/{} exists, and is more recent in filesystem, we need to reload, lastModifiedInExo={}, lastModifiedInFS={}",targetNode.getPath(),cleanedFileName, lastModifiedInExo, lastModifiedInFS);
        autoVersionService.autoVersion(file);
        Node jcrContent = file.hasNode("jcr:content")?file.getNode("jcr:content"):file.addNode("jcr:content","nt:resource");
        jcrContent.setProperty("jcr:lastModified", new GregorianCalendar());
        jcrContent.setProperty("jcr:data", Files.newInputStream(path));
        DMSMimeTypeResolver mimeTypeResolver = DMSMimeTypeResolver.getInstance();
        String mimetype = mimeTypeResolver.getMimeType(path.getFileName().toString());
        jcrContent.setProperty("jcr:mimeType", mimetype);
        file.save();
      } else {

        //File is more recent in eXo, do nothing
        LOG.debug("File {}/{} exists, and is more recent in eXo, do nothing, lastModifiedInExo={}, lastModifiedInFS={}",targetNode.getPath(),cleanedFileName, lastModifiedInExo, lastModifiedInFS);
      }
      result.currentFiles++;
      return targetNode.getNode(cleanedFileName);
    } else {
      long startTime = System.currentTimeMillis();
      Node addedNode = targetNode.addNode(cleanedFileName, "nt:file");
      if(!addedNode.isNodeType(NodetypeConstant.MIX_REFERENCEABLE)) {
        addedNode.addMixin(NodetypeConstant.MIX_REFERENCEABLE);
      }

      if(!addedNode.isNodeType(NodetypeConstant.MIX_COMMENTABLE))
        addedNode.addMixin(NodetypeConstant.MIX_COMMENTABLE);

      if(!addedNode.isNodeType(NodetypeConstant.MIX_VOTABLE))
        addedNode.addMixin(NodetypeConstant.MIX_VOTABLE);

      if(!addedNode.isNodeType(NodetypeConstant.MIX_I18N))
        addedNode.addMixin(NodetypeConstant.MIX_I18N);

      if(!addedNode.hasProperty(NodetypeConstant.EXO_TITLE)) {
        addedNode.setProperty(NodetypeConstant.EXO_TITLE, filename);
      }
      Node jcrContent = addedNode.addNode("jcr:content","nt:resource");

      BasicFileAttributes attr = Files.readAttributes(path, BasicFileAttributes.class);
      GregorianCalendar lastModifiedDate = new GregorianCalendar();
      lastModifiedDate.setTimeInMillis(attr.lastModifiedTime().toMillis());
      jcrContent.setProperty("jcr:lastModified", lastModifiedDate);
      jcrContent.setProperty("jcr:data", Files.newInputStream(path));
      DMSMimeTypeResolver mimeTypeResolver = DMSMimeTypeResolver.getInstance();
      String mimetype = mimeTypeResolver.getMimeType(path.getFileName().toString());
      jcrContent.setProperty("jcr:mimeType", mimetype);
      targetNode.getSession().save();
      autoVersionService.autoVersion(addedNode);


      result.currentFiles++;
      LOG.debug("File {}/{} not exists. Create it in {} ms.",targetNode.getPath(),cleanedFileName, System.currentTimeMillis() - startTime);
      return addedNode;
    }
  }

  private Node getOrCreateDirectoryChildNode(Node targetNode, Path path, Result result) throws Exception {
    String filename = path.getFileName().toString();
    String cleanedFileName = Text.escapeIllegalJcrChars(org.exoplatform.services.cms.impl.Utils.cleanString(filename));
    if (targetNode.hasNode(cleanedFileName)) {
      LOG.debug("Folder {}/{} exists.",targetNode.getPath(),cleanedFileName);
      Node node = targetNode.getNode(cleanedFileName);
      updatePermissions(node, path, result);
      result.currentDirectory++;
      return node;
    } else {
      long startTime = System.currentTimeMillis();
      Node addedNode = targetNode.addNode(cleanedFileName, "nt:folder");
      // Set title
      if (!addedNode.hasProperty(Utils.EXO_TITLE)) {
        addedNode.addMixin(Utils.EXO_RSS_ENABLE);
      }
      addedNode.setProperty(Utils.EXO_TITLE, filename);

      targetNode.save();
      updatePermissions(addedNode, path, result);
      result.currentDirectory++;
      LOG.debug("Folder {}/{} not exists. Create it in ms.",targetNode.getPath(),cleanedFileName, System.currentTimeMillis() - startTime);
      return addedNode;
    }
  }

  private void updatePermissions(Node addedNode, Path path, Result result) {
    String pathInIndex = path.toString().replace(sourceFolder,"");
    ExtendedNode extendedNode = (ExtendedNode) addedNode;
    if (permissionsIndex.containsKey(pathInIndex)) {
      long startTime = System.currentTimeMillis();
      List<Permission> permissions = permissionsIndex.get(pathInIndex);
      try {
        if (extendedNode.canAddMixin("exo:privilegeable")) {
          extendedNode.addMixin("exo:privilegeable");
        }
        for (AccessControlEntry acl : extendedNode.getACL().getPermissionEntries()) {
          if (!acl.getIdentity().equals("*:/platform/administrators")) {
            extendedNode.removePermission(acl.getIdentity(),acl.getPermission());
          }
        }
        permissions.stream().forEach(permission -> {
          try {
            ListAccess<Group> groups = organizationService.getGroupHandler().findGroupsByKeyword(permission.groupName);

            int groupWithExactName = 0;
            Group foundedGroup = null;
            Group[] matchingGroups = groups.load(0, groups.getSize());
            for (int i=0;i<groups.getSize();i++) {
              if (matchingGroups[i].getGroupName().equals(permission.groupName)) {
                groupWithExactName++;
                foundedGroup=matchingGroups[i];
              }
            }

            if (groupWithExactName == 0) {
              LOG.error("Group with name {} do not exists. Ignore permission.", permission.groupName);
              addInPermisionWithError(result.permissionsWithError, permission, pathInIndex);
            } else if (groupWithExactName > 1) {
              LOG.error("Group with name {} return more than one group. Ignore permission.", permission.groupName);
              addInPermisionWithError(result.permissionsWithError, permission, pathInIndex);
            } else {
              if (permission.right == AccessRight.READ) {
                String[] permsArray = new String[1];
                permsArray[0] = PermissionType.READ;
                extendedNode.setPermission("*:"+foundedGroup.getId(), permsArray);
              } else {
                extendedNode.setPermission("*:"+foundedGroup.getId(), PermissionType.ALL);
              }
            }
          } catch (Exception e) {
            LOG.error("Error when getting group {}", permission.groupName, e);
            addInPermisionWithError(result.permissionsWithError, permission, pathInIndex);
          }

        });
        extendedNode.save();
        LOG.debug("Permissions {} added on folder {} in {} ms", permissions, pathInIndex, System.currentTimeMillis() - startTime);
      } catch (RepositoryException e) {
        try {
          LOG.error("Unable to add permission mixin on node {}", addedNode.getPath(),e);
        } catch (RepositoryException ex) {
          LOG.error("Unable read exo node {}", pathInIndex,ex);
        }
        if (permissionsIndex.containsKey(pathInIndex)) {
          result.permissionsWithError.put(pathInIndex, permissionsIndex.get(pathInIndex));
        }

      }
    } else {
      LOG.debug("No permission exists for path {}", pathInIndex);
    }
  }

  private void addInPermisionWithError(Map<String, List<Permission>> permissionsWithError,
                         Permission permission, String path) {
    if (permissionsWithError.containsKey(path)) {
      permissionsWithError.get(path).add(permission);
    } else {
      List<Permission> permissionList = new ArrayList<>();
      permissionList.add(permission);
      permissionsWithError.put(path, permissionList);
    }
  }

  private Node getOrCreateTargetNode(Session session, String targetFolder) throws RuntimeException {
    try {
      Node targetNode = (Node) session.getItem(targetFolder);
      LOG.debug("Target Node {} is present in eXo, get it.", targetNode.getPath());
      return targetNode;
    } catch (PathNotFoundException pne) {
      LOG.debug("Target Node {} is not present in eXo, create it.", targetFolder);
    } catch (RepositoryException e) {
      LOG.error("Unable to get targetNode {}",targetFolder,e);
      throw new RuntimeException("Unable to get targetNode "+targetFolder,e);
    }

    try {
      Node rootNode = session.getRootNode();
      String[] paths = targetFolder.split("/");
      for (String path : paths) {
        //ignore empty path name
        if (path.equals("")) {
          continue;
        }
        try {
          if (!rootNode.hasNode(path)) {
            Node addedNode = rootNode.addNode(path, "nt:folder");
            // Set title
            if (!addedNode.hasProperty(Utils.EXO_TITLE)) {
              addedNode.addMixin(Utils.EXO_RSS_ENABLE);
            }
            addedNode.setProperty(Utils.EXO_TITLE, path);
            rootNode.save();
            rootNode = addedNode;
            LOG.debug("Node {} created.", rootNode.getPath());
          } else {
            rootNode = rootNode.getNode(path);
            LOG.debug("Node {} already exists.", rootNode.getPath());
          }
        } catch (RepositoryException r) {
          LOG.error("Unable to create folder {} in path {} in eXo", path, targetFolder, r);
          throw new RuntimeException("Unable to create folder "+path+" in path "+targetFolder+" in eXo", r);
        }
      }
      return rootNode;
    } catch (RepositoryException r) {
      LOG.error("Unable read rootNode", r);
      throw new RuntimeException("Unable read rootNode", r);
    }
  }

  public class Result {
    //start at -1 to not count the root physical folder
    public AtomicInteger nbDirectories = new AtomicInteger(-1);
    public AtomicInteger nbFiles = new AtomicInteger(0);
    public int currentDirectory=0;
    public int currentFiles=0;
    public List<String> filesInError = new ArrayList<>();
    public List<String> directoriesInError = new ArrayList<>();
    public Map<String, List<Permission>> permissionsWithError = new HashMap<>();
  }

  private class Permission {
    String groupName;
    AccessRight right;

    public Permission(String group, AccessRight permission) {
      this.groupName = group;
      this.right = permission;
    }

    public String toString() {
      return "["+groupName+","+right.toString()+"]";
    }
  }
  public enum AccessRight {
    READ,
    WRITE
  }

}
