/*
 * Copyright (C) 2003-2015 eXo Platform SAS.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.exoplatform.cs.integration.notification;

import org.exoplatform.commons.api.notification.NotificationContext;
import org.exoplatform.commons.api.notification.NotificationMessageUtils;
import org.exoplatform.commons.api.notification.annotation.TemplateConfig;
import org.exoplatform.commons.api.notification.annotation.TemplateConfigs;
import org.exoplatform.commons.api.notification.channel.template.AbstractTemplateBuilder;
import org.exoplatform.commons.api.notification.channel.template.TemplateProvider;
import org.exoplatform.commons.api.notification.model.MessageInfo;
import org.exoplatform.commons.api.notification.model.NotificationInfo;
import org.exoplatform.commons.api.notification.model.PluginKey;
import org.exoplatform.commons.api.notification.service.template.TemplateContext;
import org.exoplatform.commons.notification.template.TemplateUtils;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.commons.utils.HTMLEntityEncoder;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.cs.service.util.CSUtils;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.identity.model.Profile;
import org.exoplatform.social.core.identity.provider.OrganizationIdentityProvider;
import org.exoplatform.social.core.manager.IdentityManager;
import org.exoplatform.social.core.service.LinkProvider;
import org.exoplatform.social.notification.LinkProviderUtils;
import org.exoplatform.webui.utils.TimeConvertUtils;
import org.gatein.common.text.EntityEncoder;

import java.io.Writer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

@TemplateConfigs(
   templates = {
       @TemplateConfig( pluginId=CSReminderPlugin.ID, template="war:/notification/templates/web/CSReminderPlugin.gtmpl"),
       @TemplateConfig( pluginId=CSStatusChangedNotificationPlugin.ID, template="war:/notification/templates/web/CSStatusChangedNotificationPlugin.gtmpl"),
       @TemplateConfig( pluginId=CSTicketAssignedNotificationPlugin.ID, template="war:/notification/templates/web/CSTicketAssignedNotificationPlugin.gtmpl"),
           @TemplateConfig( pluginId=CSTicketAssignedToMeNotificationPlugin.ID, template="war:/notification/templates/web/CSTicketAssignedToMeNotificationPlugin.gtmpl"),
           @TemplateConfig( pluginId=CSTicketClosedNotificationPlugin.ID, template="war:/notification/templates/web/CSTicketClosedNotificationPlugin.gtmpl"),
           @TemplateConfig( pluginId=CSSubscriptionExpiredPlugin.ID, template="war:/notification/templates/web/CSSubscriptionExpiredPlugin.gtmpl"),
           @TemplateConfig( pluginId=CSSubscriptionReminderPlugin.ID, template="war:/notification/templates/web/CSSubscriptionReminderPlugin.gtmpl"),
           @TemplateConfig( pluginId=CSPrivatePostPlugin.ID, template="war:/notification/templates/web/CSPrivatePostPlugin.gtmpl"),
           @TemplateConfig( pluginId=CSSLAReminderPlugin.ID, template="war:/notification/templates/web/CSSLAReminderPlugin.gtmpl")
   }
)


public class WebTemplateProvider extends TemplateProvider {

  protected DateFormat formatter = new SimpleDateFormat("E MMM dd HH:mm:ss Z yyyy");
  protected static Log log = ExoLogger.getLogger(MailTemplateProvider.class);
  
  public WebTemplateProvider(InitParams initParams) {
    super(initParams);

    this.templateBuilders.put(PluginKey.key(CSReminderPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSStatusChangedNotificationPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSTicketAssignedNotificationPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSTicketAssignedToMeNotificationPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSTicketClosedNotificationPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSSubscriptionExpiredPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSSubscriptionReminderPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSPrivatePostPlugin.ID), new TemplateBuilder());
    this.templateBuilders.put(PluginKey.key(CSSLAReminderPlugin.ID), new TemplateBuilder());

  }

  private class TemplateBuilder extends AbstractTemplateBuilder {
    @Override
    protected MessageInfo makeMessage(NotificationContext ctx) {
      NotificationInfo notification = ctx.getNotificationInfo();
      String pluginId = notification.getKey().getId();      

      String language = getLanguage(notification);
      TemplateContext templateContext = TemplateContext.newChannelInstance(getChannelKey(), pluginId, language);
      
      String creator = notification.getValueOwnerParameter(NotificationUtils.CREATOR);
      String ticketUrl = notification.getValueOwnerParameter(NotificationUtils.TICKET_URL);
      String userName = notification.getValueOwnerParameter(NotificationUtils.USER_NAME);
      String ticketId = notification.getValueOwnerParameter(NotificationUtils.TICKET_ID);
      String ticketTitle = notification.getValueOwnerParameter(NotificationUtils.TICKET_TITLE);
      String ticketStatus = notification.getValueOwnerParameter(NotificationUtils.TICKET_STATUS);
      String assignee = notification.getValueOwnerParameter(NotificationUtils.ASSIGNEE);
      String oldValue = notification.getValueOwnerParameter(NotificationUtils.OLD_VALUE);
      String period = notification.getValueOwnerParameter(NotificationUtils.PERIOD);
      String spaceId = notification.getValueOwnerParameter(NotificationUtils.SPACE_ID);
      String spaceName = notification.getValueOwnerParameter(NotificationUtils.SPACE_NAME);
      String spaceUrl = notification.getValueOwnerParameter(NotificationUtils.SPACE_URL);
      String endDate = notification.getValueOwnerParameter(NotificationUtils.END_DATE);
      String post = notification.getValueOwnerParameter(NotificationUtils.POST);
      String days = notification.getValueOwnerParameter(NotificationUtils.DAYS);
      String hours = notification.getValueOwnerParameter(NotificationUtils.HOURS);
      String minutes = notification.getValueOwnerParameter(NotificationUtils.MINUTES);

      EntityEncoder encoder = HTMLEntityEncoder.getInstance();
      IdentityManager identityManager = CommonsUtils.getService(IdentityManager.class);
      Identity identity = identityManager.getOrCreateIdentity(OrganizationIdentityProvider.NAME, creator, true);
      Profile profile = identity.getProfile();
      templateContext.put("USER", encoder.encode(profile.getFullName().toString()));
      templateContext.put("AVATAR", profile.getAvatarUrl() != null ? profile.getAvatarUrl() : LinkProvider.PROFILE_DEFAULT_AVATAR_URL);
      templateContext.put("PROFILE_URL", LinkProviderUtils.getRedirectUrl("user", identity.getRemoteId()));
      //
      if(ticketUrl!=null) {
        templateContext.put("TICKET_URL", ticketUrl);
      }

      if(userName!=null) {
        Identity id=identityManager.getOrCreateIdentity(OrganizationIdentityProvider.NAME, userName, false);
        templateContext.put("USER_NAME", id.getProfile().getFullName());
      }


      if(ticketId!=null) {
        templateContext.put("TICKET_ID", ticketId);
      }

      if(ticketTitle!=null) {
        templateContext.put("TICKET_TITLE", ticketTitle);
      }

      if(ticketStatus!=null) {
        templateContext.put("TICKET_STATUS", ticketStatus);
      }

      if(assignee!=null) {
        Identity id=identityManager.getOrCreateIdentity(OrganizationIdentityProvider.NAME, assignee, false);
        templateContext.put("ASSIGNEE", id.getProfile().getFullName());
      }


      if(oldValue!=null) {
        templateContext.put("OLD_VALUE", oldValue);
      }

      if(period!=null) {
        templateContext.put("PERIOD", period);
      }


      if(spaceId!=null) {
        templateContext.put("SPACE_ID", spaceId);
      }

      if(spaceName!=null) {
        templateContext.put("SPACE_NAME", spaceName);
      }

      if(spaceUrl!=null) {
        templateContext.put("SPACE_URL", spaceUrl);
      }


      if (endDate != null && !endDate.equals("")) {
        templateContext.put("END_DATE", CSUtils.formatDate(endDate, CSUtils.getUserTimezone(notification.getTo())));
      }


      if (post != null) {
        templateContext.put("POST", post);
      }


      if (hours != null) {
        templateContext.put("HOURS", hours);
      }


      if (days != null) {
        templateContext.put("DAYS", days);
      }


      if (minutes != null) {
        templateContext.put("MINUTES", minutes);
      }


      templateContext.put("READ", Boolean.valueOf(notification.getValueOwnerParameter(NotificationMessageUtils.READ_PORPERTY.getKey())) ? "read" : "unread");
      templateContext.put("NOTIFICATION_ID", notification.getId());
      Calendar lastModified = Calendar.getInstance();
      lastModified.setTimeInMillis(notification.getLastModifiedDate());
      templateContext.put("LAST_UPDATED_TIME", TimeConvertUtils.convertXTimeAgoByTimeServer(lastModified.getTime(),"EE, dd yyyy", new Locale(language), TimeConvertUtils.YEAR));

      //
      String body = TemplateUtils.processGroovy(templateContext);
      //binding the exception throws by processing template
      ctx.setException(templateContext.getException());
      MessageInfo messageInfo = new MessageInfo();
      return messageInfo.body(body).end();
    }

    @Override
    protected boolean makeDigest(NotificationContext ctx, Writer writer) {
      return false;
    }

  };


}
