/*
 * Copyright (C) 2003-2014 eXo Platform SAS.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.exoplatform.ps.reporting;

import org.exoplatform.ps.rest.ProjectRestService;
import com.ibm.icu.text.SimpleDateFormat;
import org.exoplatform.commons.api.settings.SettingService;
import org.exoplatform.commons.api.settings.SettingValue;
import org.exoplatform.commons.api.settings.data.Context;
import org.exoplatform.commons.api.settings.data.Scope;
import org.exoplatform.services.jcr.RepositoryService;
import org.exoplatform.services.jcr.core.ManageableRepository;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.rest.resource.ResourceContainer;
import org.exoplatform.services.security.ConversationState;
import org.exoplatform.services.security.Identity;
import org.exoplatform.social.service.rest.Util;

import javax.jcr.Node;
import javax.jcr.Session;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Response;
import java.util.Date;

@Path("/bbreportingservice")
public class ReportingService implements ResourceContainer {

    private static final String SOCIAL_WORKSPACE = "social";
    private static final String TOTAL_VISITS = "exo:totalVisits";
    private static final String LAST_VISIT = "exo:lastVisit";
    private static final String TOTAL_VIEWS = "exo:totalViews";
    private static final String TOTAL_DOWNLOADS = "exo:totalDownloads";
    private static final Log LOG = ExoLogger.getLogger(ProjectRestService.class.getName());
    private static final String SUBS_GROUP = "/organization/subcontractors";
    private RepositoryService repositoryService;
    private SettingService settingService;
    private static final String BASE_URL = "baseUrl";


    public ReportingService(RepositoryService repositoryService,SettingService settingService) {
        this.repositoryService = repositoryService;
        this.settingService = settingService;
    }

    @POST
    @Path("planRoomReporting/{projectName}")
    public Response planRoomReporting(@PathParam("projectName") String projectName) {
        try {
            if (settingService.get(Context.GLOBAL, Scope.GLOBAL, BASE_URL) == null) {
                settingService.set(Context.GLOBAL, Scope.GLOBAL, BASE_URL, SettingValue.create(Util.getBaseUrl()));
            }
            Identity currentUserIdentity = ConversationState.getCurrent().getIdentity();
            if (currentUserIdentity.isMemberOf(SUBS_GROUP)) {
                ManageableRepository manageableRepository = repositoryService.getCurrentRepository();
                Session session = manageableRepository.getSystemSession(SOCIAL_WORKSPACE);
                Node socialProductionNode = (Node) session.getItem("/production");
                if (!socialProductionNode.hasNode("reporting")) {
                    socialProductionNode.addNode("reporting", "exo:reporting");
                    session.save();
                }
                Node reportingNode = socialProductionNode.getNode("reporting");
                if (!reportingNode.hasNode(projectName)) {
                    reportingNode.addNode(projectName, "exo:projectReporting");
                    session.save();
                }
                Node projectReportingNode = reportingNode.getNode(projectName);
                if (!projectReportingNode.hasNode(currentUserIdentity.getUserId())) {
                    projectReportingNode.addNode(currentUserIdentity.getUserId(), "exo:userReporting");
                    session.save();
                }
                Node userReportingNode = projectReportingNode.getNode(currentUserIdentity.getUserId());
                if (!userReportingNode.hasProperty(TOTAL_VISITS)) {
                    userReportingNode.setProperty(TOTAL_VISITS, 1);
                } else {
                    int totalVisitValue = Integer.parseInt(userReportingNode.getProperty(TOTAL_VISITS).getString()) + 1;
                    userReportingNode.setProperty(TOTAL_VISITS, totalVisitValue);
                }
                session.save();
                SimpleDateFormat formater = new SimpleDateFormat("MM/dd/yy");
                userReportingNode.setProperty(LAST_VISIT, formater.format(new Date()));
                session.save();
            }
            return Response.ok("VisitsReported").build();
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity("An internal error has occured").build();
        }
    }

    public void docViewsReporting(Node node) {
        Identity currentUserIdentity = ConversationState.getCurrent().getIdentity();
        try {
            if (currentUserIdentity.isMemberOf(SUBS_GROUP)) {
                docReporting(node, currentUserIdentity.getUserId(), TOTAL_VIEWS);
            }
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
    }

    public void docDownloadsReporting(Node node) {
        Identity currentUserIdentity = ConversationState.getCurrent().getIdentity();
        try {
            if (currentUserIdentity.isMemberOf(SUBS_GROUP)) {
                docReporting(node, currentUserIdentity.getUserId(), TOTAL_DOWNLOADS);
            }
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
        }
    }

    private void docReporting(Node node, String user, String operation) throws Exception {
        ManageableRepository manageableRepository = repositoryService.getCurrentRepository();
        Session session = manageableRepository.getSystemSession(SOCIAL_WORKSPACE);
        Node socialProductionNode = (Node) session.getItem("/production");
        if (!socialProductionNode.hasNode("reporting")) {
            socialProductionNode.addNode("reporting", "exo:reporting");
            session.save();
        }
        Node reportingNode = socialProductionNode.getNode("reporting");
        String project = node.getPath().split("/Groups/spaces/")[1].split("/Documents")[0];
        if (!reportingNode.hasNode(project)) {
            reportingNode.addNode(project, "exo:projectReporting");
            session.save();
        }
        Node projectReportingNode = reportingNode.getNode(project);
        if (!projectReportingNode.hasNode(user)) {
            projectReportingNode.addNode(user, "exo:userReporting");
            session.save();
        }
        Node userReportingNode = projectReportingNode.getNode(user);
        if (!userReportingNode.hasNode(node.getUUID())) {
            userReportingNode.addNode(node.getUUID(), "exo:docReporting");
            session.save();
        }
        Node docReportingNode = userReportingNode.getNode(node.getUUID());
        if (!docReportingNode.hasProperty(operation)) {
            docReportingNode.setProperty(operation, 1);
        } else {
            int operationValue = Integer.parseInt(docReportingNode.getProperty(operation).getString()) + 1;
            docReportingNode.setProperty(operation, operationValue);
        }
        session.save();
    }
}