/**
 * Skype for Bunsiness provider module for Web Conferencing. This script will be used to add a provider to Web Conferencing module and then
 * handle calls for portal user/groups.
 */
(function($, webConferencing) {
	"use strict";
	
	var globalWebConferencing = typeof eXo != "undefined" && eXo && eXo.webConferencing ? eXo.webConferencing : null;
	
	// Use webConferencing from global eXo namespace (for non AMD uses)
	if (!webConferencing && globalWebConferencing) {
		webConferencing = globalWebConferencing;
	}

	if (webConferencing) {

		// Start with default logger, later in configure() we'll get it for the provider.
		// We know it's mssfb here, but mark with asterisk as not yet configured.
		var log = webConferencing.getLog("mssfb");
		//log.trace("> Loading at " + location.origin + location.pathname);
		
		function CallContainer($container) {
			var onMssfbCallShow = [];
			var onMssfbCallHide = [];
			
			var conversation = null;
			var callId = null;
			var callerId = null;
			var attached = true;
			var shown = false;
			
			var self = this;
			self.$container = $container;
			self.$element = null;
			self.element = null;
			
			var initElement = function() {
				self.$container.find("#mssfb-call-conversation").remove();
				var $convo = $("<div id='mssfb-call-conversation'></div>");
				self.$container.append($convo);
				self.$element = $convo;
				self.element = self.$element.get(0);
			};
			
			var callHandlers = function(handrels) {
				for (var i=0; i<handrels.length; i++) {
					handrels[i](self.$element);
				}
			};
			
			this.init = function() {
				initElement();
				conversation = null;
				callId = null;
				callerId = null;
				attached = true;
				return this;
			};
			
			this.getConversation = function() {
				return conversation;
			};
			
			this.getCallerId = function() {
				return callerId;
			};
			
			this.getCallId = function() {
				return callId;
			};
			
			this.setConversation = function(convoInstance, id, caller) {
				conversation = convoInstance;
				callId = id;
				callerId = caller;
			};
			
			this.setCallId = function(id) {
				callId = id;
			};
			
			this.isAttached = function() {
				return attached;
			};
			
			this.attached = function() {
				attached = true;
			};
			
			this.detached = function() {
				attached = false;
			};
			
			this.isVisible = function() {
				return shown;
			};
			
			this.show = function() {
				callHandlers(onMssfbCallShow);
				shown = true;
				return this;
			};
			
			this.hide = function() {
				callHandlers(onMssfbCallHide);
				shown = false;
				return this;
			};
			
			this.onShow = function(handler) {
				if (handler) {
					onMssfbCallShow.push(handler);
				}
				return this;
			};
			
			this.onHide = function(handler) {
				if (handler) {
					onMssfbCallHide.push(handler);
				}
				return this;
			};
		}
		
		function SfBProvider() {
			var EMAIL_PATTERN = /\S+@\S+[\.]*\S+/;
			
			var TOKEN_STORE = "mssfb_login_token";
			
			var MSONLINE_LOGIN_PREFIX = "https://login.microsoftonline.com";
			
			var self = this;
			var settings, currentKey;
			var apiInstance, appInstance, uiApiInstance, uiAppInstance;
			
			var localConvos = {};
			
			var isMac = navigator.userAgent.toUpperCase().indexOf("MAC") >= 0;
			var isWindows = navigator.userAgent.toUpperCase().indexOf("WINDOWS") >= 0;
			
			var isClosed = false;
			var onClosePage = function(conversation, app) {
				if (!isClosed && conversation && app) {
					// TODO In WebKIT browsers this stuff doesn't work and outgoing call still may run for the remote party if
					// simply close the window. It is caused by the browser implementation, in WebKIT it doesn't let asynchronous
					// requests to work on this stage, but if user will click to stay(!) on the page, then this code will be
					// proceeded and the call ended.
					// In IE and FF (but not for video/audio in FF currently, Apr 26 2017), this code will be
					// executed by the browser and call rejected or stopped.
					conversation.leave().then(function() {
						isClosed = true;
						log.trace("<< conversation leaved");
						app.conversationsManager.conversations.remove(conversation);
					});
					//conversation.selfParticipant.reset();
					// TODO what a right approach to leave the convo? 
					// conversation.videoService.stop();
					// conversation.audioService.stop();
					// conversation.chatService.stop();
					// uiApp.signInManager.signOut().then(function() {
					// log.trace("<<<< Skype signed out");
					// }, function(error) {
					// log.trace("<<<< Error signing out Skype:" + error);
					// });
					return "You canceled the call";
				}
			}
			
			var authRedirectLink = function(apiClientId, redirectUri, resource) {
				var loginUri = MSONLINE_LOGIN_PREFIX + "/common/oauth2/authorize?response_type=token&client_id="
					+ apiClientId
					+ "&redirect_uri="
					+ encodeURIComponent(redirectUri)
					+ "&resource="
					+ encodeURIComponent(resource);
				return loginUri;
			};
			
			var authRedirectWindow = function(apiClientId, redirectUri, resource) {
				var loginUri = authRedirectLink(apiClientId, redirectUri, resource);
				log.trace("MSSFB login/call: " + loginUri);
				var theWindow = webConferencing.showCallPopup(loginUri, "Skype for Business Login");
				return theWindow;
			};
			
			var notifyCaller = function(destWindow, title, message) {
				if (destWindow.notifyUser) {
					destWindow.notifyUser({
						title : title, 
						text : message							        
					});
				} else {
					setTimeout($.proxy(notifyCaller, this), 250, destWindow, title, message);
				}
			};
			
			var sipId = function(mssfbId) {
				if (mssfbId && !mssfbId.startsWith("sip:")) {
					mssfbId = "sip:" + mssfbId;
				}
				return mssfbId;
			};
			
			var tokenHashInfo = function(hline) {
				var tokenEnd = hline.indexOf("&token_type");
				if (tokenEnd > 0) {
					var start = tokenEnd - 7;
					if (hline.length - start > 0) {
						return hline.substring(start);
					} else {
						log.warn(">> tokenHashInfo: unexpected hash line format: " + hline);
					}
				}
				return hline; 
			};
			this.tokenHashInfo = tokenHashInfo;
			
			var removeLocalToken = function() {
				log.trace("Remove login token");
				if (typeof Storage != "undefined") {
			    // Code for localStorage/sessionStorage.
					localStorage.removeItem(TOKEN_STORE);
				} else {
				  // No Web Storage support.
					if (eXo && eXo.webConferencing && eXo.webConferencing.mssfb) {
						delete eXo.webConferencing.mssfb.__localToken;
					} else {
						log.warn("Removing access token: local storage not supported.");
					}
				}
			};
			var saveLocalToken = function(token) {
				log.trace("Using new login token");
				if (typeof Storage != "undefined") {
			    // Code for localStorage/sessionStorage.
					try {
						localStorage.setItem(TOKEN_STORE, JSON.stringify(token));
					} catch(err) {
						log.error("Error saving access token", err);
					}
				} else {
				  // Sorry! No Web Storage support.
					// TODO save it in session storage or server-side?
					if (eXo && eXo.webConferencing && eXo.webConferencing.mssfb) {
						eXo.webConferencing.mssfb.__localToken = token;
						log.warn("Access token saved only in this window, not local storage!");
					} else {
						log.warn("Saving access token: local storage not supported.");
					}
				}
			};
			this.saveToken = function(token) {
				saveLocalToken(token);
			};
			var currentToken = function() {
				var token;
				if (typeof Storage != "undefined") {
					var savedToken = localStorage.getItem(TOKEN_STORE);
					if (savedToken) {
						// TODO in Session Storage we can find "swx-sessionid" it appears when SDK logged in, 
						// this record can be used as a flag of active session
						// TODO there are also several records in Local Storage
						try {
							token = JSON.parse(savedToken);
						} catch(e) {
							log.warn("Login (saved) parsing error: " + e, e);
							localStorage.removeItem(TOKEN_STORE);
						}		
					} else {
						log.trace("Login (saved) token not found");
					}
				} else if (eXo && eXo.webConferencing && eXo.webConferencing.mssfb) {
					token = eXo.webConferencing.mssfb.__localToken;
				}
				if (token) {
					// cut a gap in 20sec for actual request processing
					var expiresIn = token.created + token.expires_in - 20;
					var now = new Date().getTime()/1000;
					if (now >= expiresIn) {
						// token expired
						log.debug("Login (saved) token expired");
						token = null;
						localStorage.removeItem(TOKEN_STORE);
					} else {
						// else, we can use current token
						//log.trace(">> currentToken: " + tokenHashInfo(token.hash_line));
						log.debug("Using existing login token");
					}
				}
				return token;
			};
			
			var callUri = function(callId, context, hashLine) {
				//var currentSpaceId = webConferencing.getCurrentSpaceId();
				//var currentRoomTitle = webConferencing.getCurrentRoomTitle();
				var q;
				if (context.isSpace) {
					q = "?space=" + encodeURIComponent(context.spaceId);
				} else if (context.isRoom) {
					q = "?room=" + encodeURIComponent(context.roomTitle);
				} else {
					q = "";
				}
				var uri = webConferencing.getBaseUrl() + "/portal/skype/call/" + callId + q;
				if (hashLine) {
					uri += hashLine;
				}
				return uri;
			};
			
			var openCallWindow = function(callId, context) {
				var token = currentToken();
				// TODO check if such window isn't already open by this app
				if (token) {
					// use existing token
					var uri = callUri(callId, context, token.hash_line);
					return webConferencing.showCallPopup(uri, "Skype For Business Call");
				} else {
					// open SfB login window with redirect to the call URI
					var uri = callUri(callId, context);
					return authRedirectWindow(settings.apiClientId, uri, "https://webdir.online.lync.com");
				}
			};
			
			var getSDKErrorData = function(sdkErr) {
				var r = sdkErr ? (sdkErr.rsp ? sdkErr.rsp : (sdkErr.req ? sdkErr.req : null)) : null;
				return r && r.data ? r.data : null;
			};
			
			var handleErrorData = function(callId, title, errData) {
				var errMsg;
				if (errData.code == "Gone" && errData.subcode == "TooManyApplications") {
					errMsg = "Too many applications. " + errData.message;	
				} else {
					errMsg = "[" + errData.code + "] " + errData.message;
				}
				log.showError("Call failure: " + callId, 
							"[" + errData.code + " " + (errData.subcode ? "(" + errData.subcode + ")] " : "] ") + errData.message, 
							title, errMsg);		
			};
			
			var callUpdate = function() {
				// empty by default, may be overridden by chat, space etc.
			};
			
			var hasJoinedCall = function(callRef) {
				// may be overridden by chat, space etc.
				return false;
			};
			
			var getLocalCall = function(callRef) {
				// may be overridden by chat, space etc.
				return localConvos[callRef];
			};
			
			var saveLocalCall = function(state) {
				// may be overridden by chat, space etc.
				var prev = localConvos[state.callId];
				localConvos[state.callId] = state;
				localConvos[state.peer.id] = state;
				if (state.peer.chatRoom) {
					localConvos[state.peer.chatRoom] = state;					
				}
				return prev;
			};
			
			var removeLocalCall = function(callId) {
				// may be overridden by chat, space etc.
				var state = localConvos[callId];
				if (state) {
					delete localConvos[callId];	
					delete localConvos[state.peer.id];
				}
				return state;
			};
			
			this.getType = function() {
				if (settings) {
					return settings.type;
				}
			};
			
			this.getSupportedTypes = function() {
				if (settings) {
					return settings.supportedTypes;
				}
			};

			this.getTitle = function() {
				if (settings) {
					return settings.title;
				}
			};

			this.getCallTitle = function() {
				return "Call"; // TODO i18n
			};
			
			this.getJoinTitle = function() {
				return "Join"; // TODO i18n
			};

			this.getApiClientId = function() {
				if (settings) {
					return settings.apiClientId;
				}
			};

			this.configure = function(skypeEnv) {
				settings = skypeEnv;
			};

			this.isConfigured = function() {
				return settings != null;
			};

			this.application = function(redirectUri, getOAuthToken) {
				var initializer = $.Deferred();
				if (settings) {
					if (apiInstance && appInstance) {
						log.trace("Use existing Skype instance");
						initializer.resolve(apiInstance, appInstance);
					} else {
						var user = webConferencing.getUser();
						var sessionId = user.id + "_session" + Math.floor((Math.random() * 1000000) + 1);
						log.debug("Skype sessionId: " + sessionId + " version: " + settings.version + ", API clientId: " + settings.apiClientId);
						Skype.initialize({
							"version" : settings.version,
							"apiKey" : settings.apiKey,
							"correlationIds" : {
								"sessionId" : sessionId
							// Necessary for troubleshooting requests, should be unique per session
							}
						}, function(api) {
							var app = new api.application();
							// SignIn SfB Online: the SDK will get its own access token
							if (!redirectUri) {
								redirectUri = redirectUri ? redirectUri : settings.redirectUri; 
							}
							var args = {
								"client_id" : settings.apiClientId,
								"origins" : settings.origins,
								"cors" : true,
								"version" : settings.version
							// Necessary for troubleshooting requests; identifies your application in our telemetry
							};
							if (getOAuthToken && typeof getOAuthToken == "function") {
								args.get_oauth_token = getOAuthToken;
							} else {
								args.redirect_uri = redirectUri;
							}
							app.signInManager.signIn(args).then(function(res) {
								log.debug("Skype signed in as " + app.personsAndGroupsManager.mePerson.displayName());
								// ensure local and remote users are of the same account in MS
								var exoUserSFB = webConferencing.imAccount(user, "mssfb");
								var mssfbUserId = app.personsAndGroupsManager.mePerson.id(); // sip:email...
								if (exoUserSFB && exoUserSFB.id != mssfbUserId) {
									// bad, we cannot proceed - need ask local user login with his account in MS
									log.debug("Skype current user and local eXo user have different Microsoft IDs: " 
												+ mssfbUserId + " vs " + exoUserSFB.id + ". Need login Skype under " + exoUserSFB.id);
									// Reject with a message to user
									initializer.reject("Skype signed in under different account: " + mssfbUserId + ". Please login as " + exoUserSFB.id);
								} else {
									// else, we don't care here if it is SfB user on eXo side
									apiInstance = api;
									appInstance = app;
									initializer.resolve(api, app);
								}
							}, function(err) {
								initializer.reject(err);
							});

							// whenever client.state changes, display its value
							app.signInManager.state.changed(function(state) {
								// TODO update user state and UI in PLF page
								log.trace("Skype user state change: " + JSON.stringify(state));
							});
						}, function(err) {
							initializer.reject(err);
						});
					}
				} else {
					initializer.reject("Skype settings not found");
				}
				return initializer.promise();
			};

			this.uiApplication = function(redirectUri, user) {
				var initializer = $.Deferred();
				if (settings) {
					if (uiApiInstance && uiAppInstance) {
						log.trace("Use existing SkypeCC instance");
						initializer.resolve(uiApiInstance, uiAppInstance);
					} else {
						if (!user) {
							user = webConferencing.getUser();
						}
						var sessionId = user.id + "_uisession" + Math.floor((Math.random() * 1000000) + 1);
						log.debug("SkypeCC sessionId: " + sessionId + " version: " + settings.version + ", API clientId: " + settings.apiClientId);
						Skype.initialize({
							"version" : settings.version,
							"apiKey" : settings.apiKeyCC,
							"correlationIds" : {
								"sessionId" : sessionId
							// Necessary for troubleshooting requests, should be unique per session
							}
						}, function(api) {
							var app = api.UIApplicationInstance;
							// SignIn SfB Online: the SDK will get its own access token
							if (!redirectUri) {
								redirectUri = redirectUri ? redirectUri : settings.redirectUri; 
							}
							app.signInManager.signIn({
								"client_id" : settings.apiClientId,
								"origins" : settings.origins,
								"cors" : true,
								"redirect_uri" : redirectUri,
								"version" : settings.version
							// Necessary for troubleshooting requests; identifies your application in our telemetry
							}).then(function() {
								log.debug("SkypeCC signed in as " + app.personsAndGroupsManager.mePerson.displayName());
								// ensure local and remote users are of the same account in MS
								var exoUserSFB = webConferencing.imAccount(user, "mssfb");
								var mssfbUserId = app.personsAndGroupsManager.mePerson.id(); // sip:email...
								if (exoUserSFB && exoUserSFB.id != mssfbUserId) {
									// bad, we cannot proceed - need ask local user login with his account in MS
									log.debug("SkypeCC current user and local eXo user have different Microsoft IDs: " 
												+ mssfbUserId + " vs " + exoUserSFB.id + ". Need login SkypeCC under " + exoUserSFB.id);
									// Reject with a message to user
									initializer.reject("Skype signed in under different account: " + mssfbUserId + ". Please login as " + exoUserSFB.id);
								} else {
									// else, we don't care here if it is SfB user on eXo side
									uiApiInstance = api;
									uiAppInstance = app;
									initializer.resolve(api, app);
								}
							}, function(err) {
								initializer.reject(err);
							});

							// whenever client.state changes, display its value
							app.signInManager.state.changed(function(state) {
								log.trace("SkypeCC user state change: " + JSON.stringify(state));
							});
						}, function(err) {
							initializer.reject(err);
						});
					}
				} else {
					initializer.reject("Skype settings not found");
				}
				return initializer.promise();
			};

			//////////////// Call handlers //////////////////
			
			this.newCallContainer = function($container) {
				return new CallContainer($container);
			};
			
			// Check if Skype web plugin installed
			var checkPlugin = function(app) {
				var process = $.Deferred();
				var cap = app.devicesManager.mediaCapabilities;
				if (cap.isBrowserMediaSupported() || cap.isPluginInstalled()) {
					// everything OK
					process.resolve(false);
				} else {
					cap.installedVersion.get().finally(function() {
						function pluginLink() {
							var link;
							if (isWindows) {
								link = cap.pluginDownloadLinks("msi");
								if (!link) {
									link = cap.pluginDownloadLinks("exe");
								}
							} else if (isMac) {
								link = cap.pluginDownloadLinks("pkg");
								if (!link) {
									link = cap.pluginDownloadLinks("dmg");
								}
							}
							if (!link) {
								link = "https://support.skype.com/en/faq/FA12316/what-is-the-skype-web-plugin-and-how-do-i-install-it";
							}
							return link;
						}
						var p = cap.isPluginInstalled;
						var r = p.reason;
						if (p()) {
							if (r == "CompatiblePluginInstalled") {
						     // optional:
						     // tell user which version is detected: cap.installedVersion()
						     // recommend user to upgrade plugin to latest
						     // using cap.pluginDownloadLinks('msi') etc.
								var plink = pluginLink();
								log.warn("Skype web plugin version can be upgraded to " + plink);
								webConferencing.showWarn("Skype web plugin upgrade", "You are using obsolete version of Skype plugin. <a " +
										"href='" + plink + "' target='_blank' class='pnotifyTextLink'>Latest version</a> can be installed.");	
								process.resolve(true); // true - means user action recommended
						  } else {
						  	process.resolve(false);
						  }
						} else {
							var installedVersion;
							var stateText = "not found";
							if (r == "ObsoletePluginInstalled") {
								// tell user which version is detected: cap.installedVersion()
								installedVersion = cap.installedVersion();
								stateText = "obsolete";
							} else if (r == "NoPluginInstalled") {
						    // tell user no plugin is detected
								stateText = "not installed";
							}
							// recommend user to upgrade plugin to latest
							// using cap.pluginDownloadLinks('msi') etc.
							var versionInfo = installedVersion ? ". Current installed version is " + installedVersion + "." : "";
							log.error("Skype web plugin " + stateText + versionInfo);
							webConferencing.showWarn("Skype web plugin " + stateText, "You need install supported Skype plugin <a " +
										"href='" + pluginLink() + "' target='_blank' class='pnotifyTextLink'>version</a>" + versionInfo);
							process.reject();
						}
					});
				}
				return process.promise();
			};
			var loginUri = webConferencing.getBaseUrl() + "/portal/skype/call/login";
			var loginWindow = function() {
				return authRedirectWindow(settings.apiClientId, loginUri, "https://webdir.online.lync.com");
			};
			var loginIframe;
			var loginTokenUpdater;
			var loginTokenHandler = function(token, user) {
				delete self.loginToken;
				var process = $.Deferred();
				try {
					if (loginTokenUpdater) {
						clearTimeout(loginTokenUpdater);
					}
					var prevHash = location.hash;
					if (prevHash && prevHash.indexOf("access_token") >= 0) {
						log.debug(">> loginTokenHandler WARN loading on page with access_token in the hash: " + prevHash);
					}
					location.hash = token.hash_line;
					var appInitializer = self.uiApplication(loginUri, user);
					appInitializer.done(function(api, app) {
						// TODO save token in the server-side for late use?
					  log.debug("Login OK, app created OK");
					  // Save the token hash in local storage for later use
					  location.hash = prevHash;
					  token.userId = app.personsAndGroupsManager.mePerson.id();
					  saveLocalToken(token);
					  process.resolve(api, app);
					  // Remove warnings if any
					  // TODO check exactly that the SfB authorized 
						$(".mssfbLoginButton").remove();
					  // And care about token renewal
						// TODO another approach to rely on the SDK calls to login page, but this doesn't work in IE
						var updateTime = (token.expires_in > 3660 ? 3600 : token.expires_in - 60)  * 1000;
						loginTokenUpdater = setTimeout(function() {
							loginTokenUpdater = null;
							loginIframe().done(function() {
								log.debug("Updated login token to: " + token.created);
							}).fail(function(err) {
								log.warn("Failed to update login token", err);
							});
						}, updateTime);
					});
					appInitializer.fail(function(err) {
						log.error("Login error", err);
						location.hash = prevHash;
						process.reject("Login error");
					});
					// it is possible that MS auth will fail silently and we will not receive any status or callback from their SDK
					// an usecase, logout from MS account, but keep saved login token in this script and try use it, as a result
					// we'll see an iframe crated by the SDK, it will contain an error about not found tenant and error 400, but error
					// callback will not be executed in the SDK signIn and the state will stay on "SigningIn".
					// Thus we'll wait a reasonable time for login and if appInitializer will stay pending - reject it.
					setTimeout(function() {
						if (appInitializer.state() == "pending") {
							location.hash = prevHash;
							// Force getting of a new token
							localStorage.removeItem(TOKEN_STORE);
							process.reject("Application timeout. Please retry.");
						}
					},30000);
				} catch(e) {
					process.reject(e);
				}
				return process.promise();
			};
			loginIframe = function() {
				var process = $.Deferred();
				// FYI if user not authorized (in cookies etc) this iframe will fail due to 'X-Frame-Options' to 'deny' set by MS
				// TODO may be another URL found to do this?
				var iframeUri = authRedirectLink(settings.apiClientId, loginUri, "https://webdir.online.lync.com");
				var $iframe = $("<iframe src='" + iframeUri + "' height='0' width='0' style='display: none;'></iframe>");
				self.loginToken = function(token) {
					var callback = loginTokenHandler(token);
					callback.done(function(api, app) {
						process.resolve(api, app);
					});
					callback.fail(function(err) {
						process.reject(err);
					});
					callback.always(function() {
						$iframe.remove();
					});
					return callback.promise();
				};
				setTimeout(function() {
					// check if $iframe not stays on MS server: it will mean we need login user explicitly
					try {
						var iframe = $iframe.get(0);
						var doc = (iframe.contentWindow || iframe.contentDocument);
						// if accessed ok, then it's eXo server URL, not MS login
						if (doc) {
							if (doc.document) doc = doc.document;
							var checkUri = doc.URL;
							var aci = checkUri.indexOf("#access_token");
							if (aci > 0) {
								checkUri = checkUri.substring(0, aci);
							}
							log.debug("Login iframe check DONE: " + checkUri);							
						} else {
							throw "Document not accessible";
						}
					} catch (e) {
						// it's an error, like DOMException for
						$iframe.remove();
						delete self.loginToken;
						log.error("Login iframe check FAILED", e);
						process.reject("User not logged in");
					}
				}, 3000);
				$(document.body).append($iframe);
				return process.promise();
			};
			
			var getCallId = function(c) {
				if (c.isGroupConversation()) {
					var curi = c.uri();
					if (curi) {
						return "g/" + curi;
					} else if (c.state() != "Created") {
						log.error("Group conversation state not 'Created' but without URI, state was: " + c.state());
					}
					return null;
				} else {
					var state = c.state();
					var id = "p/";
					var creatorId = c.creator.id();
					var myId = c.selfParticipant.person.id();
					var p1 = c.participants(0);
					if (p1) {
						var p1Id = p1.person.id();
						if (creatorId != p1Id) {
							id += creatorId + ";" + p1Id;	
						} else if (creatorId != myId) {
							id += creatorId + ";" + myId;
						} else {
							id += creatorId;
						}
					} else {
						id += myId;
					}
					return id;
				}
			};
			this.getCallId = getCallId;
			var logging = {};
			var logConversation = function(c, key, index) {
				var key = key ? key : (c.id ? c.id() : null);
				if (key) { 
					if (logging[key]) {
						// already tracked
						log.debug("Log (already) CONVERSATION " + getCallId(c) + " state:" + c.state() + (key ? " key:" + key : "") + (index ? " index:" + index : ""));
						return;
					} else {
						logging[key] = true;
					}
				}
				log.debug("Log CONVERSATION " + getCallId(c) + (key ? " key:" + key : "") + (index ? " index:" + index : ""));
				c.state.changed(function(newValue, reason, oldValue) {
					log.debug("CONVERSATION " + getCallId(c) + (key ? " (" + key + ")" : "") + " state: " + oldValue + "->" + newValue + " reason:" + reason);
				});					
			};
			this.logConversation = logConversation;
			var isModalityUnsupported = function(error) {
				// TODO CommandDisabled also will appear for state of previously disconnected convo
				// AssertionFailed happens in Chrome when Video cannot be rendered
				if (error.code == "CommandDisabled" || error.code == "AssertionFailed"
						|| (error.code == "InvitationFailed" && error.reason.subcode == "UnsupportedMediaType")
						// it's a case when group conference re-created and incoming happen, this error occurs but doesn't prevent the work
						|| (error.code == "InvalidState" && error.actual == "Connecting" && error.expected == "Notified")) {
					return true;
				}
				return false;
			};
			this.isModalityUnsupported = isModalityUnsupported;
			var handleError = function(callId, error, postOp) {
				var title = "Error starting call";
				if (error) {
					// TODO {"code":"CommandDisabled"}? 
					// {"code":"AssertionFailed"}? happens in Chrome when Video cannot be rendered
					var handled = false;
					var isError = true;
					var isOutdated = false;
					if (error.code) { 
						if (error.code == "Canceled") {
							// Do nothing
							handled = true;
							isError = false;
						} else if (error.code == "InvitationFailed" && error.reason && "DestinationNotFound" == error.reason.subcode) {
							// It's a case of outdated conference call, need re-establish it
							// {"code":"InvitationFailed","reason":{"code":"NotFound","subcode":"DestinationNotFound","message":"The person or meeting doesn't exist."}}
							handled = true;
							isError = false;
							isOutdated = true;
						}
					}
					if (!handled) {
						if (error.reason && error.reason.subcode && error.reason.message) {
							log.showError("Call failure: " + callId, "[" + error.reason.subcode + "] " + error.reason.message, title, error.reason.message);
						} else {
							var errData = getSDKErrorData(error);
							if (errData) {
								handleErrorData(callId, title, errData);
							} else {
								log.showError("Call failure: " + callId, title + ". " + error, title, error);
							}
						}
					}
				}
				if (postOp) {
					postOp(isError, isOutdated);
				}
			};
			
			var activeParticipants = function(conversation) {
				var process = $.Deferred();
				conversation.participants.get().then(function(ulist) {
					var active = [];
					//for (var i=0; cpi<conversation.participantsCount(); i++) {
					for (var i=0; i<ulist.length; i++) {
						//if (conversation.participants(i).state.get() != "Disconnected") {
						if (ulist[i].state.get() != "Disconnected") {
							active.push(ulist[i]);
						};
					}
					process.resolve(active);
				}).catch(function(err) {
					process.reject(err);
				});				
				return process.promise();
			};
			
			var canJoin = function(localCall) {
				return localCall.state == "started" || localCall.state == "canceled" || localCall.state == "leaved";
			};
			
			var outgoingCallHandler = function(api, app, container, currentUser, target, userIds, participants, localCall) {
				var process = $.Deferred();
				container.init();
				checkPlugin(app).done(function() {
					var peerId, peerType, peerRoom;
					var options = {
								modalities : [ "Chat" ]
					};
					if (localCall && localCall.conversation) {
						// it's existing conversation, it was running in this app/page
						options.conversation = localCall.conversation;
					} else {
						if (target.group) {
							var addParticipant = function(conversation, pid) {
								try {
									var remoteParty = conversation.createParticipant(pid);
									conversation.participants.add(remoteParty);
								} catch(e) {
									log.warn("Failed to create a group participant " + pid, e);
									// TODO notify user
								}
							};
							if (target.callId) {
								// reuse existing group convo: cut 'g/' from the call id - it's a SIP URI
								var conversation = app.conversationsManager.getConversationByUri(target.callId.substring(2));
								if (conversation) {
									log.trace("Reuse group conversation " + target.callId + " " + conversation.state());
									options.conversation = conversation;
									// We add new parts from given participants to the existing convo
									// TODO Should we remove not existing locally?
									var remoteParts = [];
									for (var cpi=0; cpi<conversation.participantsCount(); cpi++) {
										remoteParts.push(conversation.participants(cpi).person.id());
									}
									nextLocal: for (var lpi=0; lpi<participants.length; lpi++) {
										for (var rpi=0; rpi<remoteParts.length; rpi++) {
											var lpid = participants[lpi];
											if (lpid) {
												if (lpid == remoteParts[rpi]) {
													continue nextLocal;
												}
												// add a new party
												addParticipant(conversation, lpid);
											}
										}
									}
								} else {
									log.warn("Group conversation not found " + target.callId + " for call '" + target.title + "'");
									options.participants = participants;
								}
							} else {
								var conversation = app.conversationsManager.createConversation();
								for (var i=0; i<participants.length; i++) {
									var pid = participants[i];
									addParticipant(conversation, pid);
								}
								options.conversation = conversation;
							}
						} else {
							options.participants = participants;
						}
					}
					api.renderConversation(container.element, options).then(function(conversation) {
						var callId = getCallId(conversation);
						log.info("Outgoing call '" + target.title + "': " + callId);
						logConversation(conversation);
						// TODO in case of video error, but audio or chat success - show a hint message to an user and auto-hide it
						var beforeunloadListener = function(e) {
							var msg = onClosePage(conversation, app);
							if (msg) {
								e.returnValue = msg; // Gecko, Trident, Chrome 34+
								return msg; // Gecko, WebKit, Chrome <34
							}
						};
						var unloadListener = function(e) {
							onClosePage(conversation, app);
						};
						var ownerType, ownerId;
						if (target.group) {
							ownerType = peerType = target.type;
							ownerId = peerId = target.id;
						} else {
							ownerType = peerType = "user";
							ownerId = currentUser.id;
							peerId = userIds[1]; // first user is an owner (caller), second one is a remote party
						}
						var callStateUpdate = function(state) {
							callUpdate({
								state : state,
								callId : callId,
								peer : {
									id : peerId,
									type : peerType,
									chatRoom : webConferencing.getChat().currentRoomId()
								},
								conversation : conversation,
								saved : localCall ? true : false
							});
						};
						var added = false;
						var addCall = function() {
							if (!added && callId != "g/adhoc") {
								log.trace(">>> Adding " + callId + " > " + new Date().getTime());
								added = true;
								var callInfo = {
									owner : ownerId,
									ownerType : ownerType,  
									provider : self.getType(),
									participants : userIds.join(";") // eXo user ids here
								};
								var topic = conversation.topic();
								if (topic) {
									callInfo.title = topic; 
								}
								callStateUpdate("started");
								return webConferencing.addCall(callId, callInfo).done(function(call) {
									log.info("Created call " + callId + " parts:" + conversation.participantsCount());
								}).fail(function(err) {
									added = false;
									log.error("Failed to create a call: " + callId, err);
								});									
							}
						};
						var startedCall = function() {
							callStateUpdate("started");
							return webConferencing.updateCall(callId, "started").done(function(call) {
								log.info("Call started: " + callId + " parts:" + conversation.participantsCount());
							}).fail(function(err) {
								log.error("Failed to start a call: " + callId, err);
							});
						};
						var deleted = false;
						var deleteCall = function() {
							if (!deleted) {
								deleted = true;
								log.trace("Deleting call: " + callId);
								callStateUpdate("stopped");
								return webConferencing.deleteCall(callId).done(function() {
									log.info("Deleted call: " + callId + " parts:" + conversation.participantsCount());
								}).fail(function(err) {
									deleted = false;
									log.error("Failed to delete a call: " + callId, err);
								});								
							}
						};
						var joined = false;
						var joinedGroupCall = function() {
							if (!joined) {
								joined = true;
								callStateUpdate("joined");
								webConferencing.updateUserCall(callId, "joined").fail(function(err) {
									log.error("Failed to join group call: " + callId, err);
								});
							}
						};
						var leavedGroupCall = function() {
							joined = false; // do this sooner
							callStateUpdate("leaved");
							webConferencing.updateUserCall(callId, "leaved").fail(function(err) {
								log.error("Failed to leave group call: " + callId, err);
							});
						};
						var started = false;
						var startingCall = function(stateName) {
							if (!started) {
								started = true;
								// Get call ID again from the conversation to keep it fresh (as may change from adhoc to real ID)
								callId = getCallId(conversation);
								log.trace(stateName + " outgoing " + callId);
								container.setCallId(callId);
								process.resolve(callId, conversation);
								if (conversation.isGroupConversation()) {
									if (target.callId) {
										if (localCall && canJoin(localCall)) {
											joinedGroupCall();
										} else {
											startedCall();
										}
									} else {
										addCall();
									}
									// In group call, we want produce started/stopped status on audio/video start/stop
									/*conversation.selfParticipant.audio.state.changed(function listener(newValue, reason, oldValue) {
										log.trace(">>> AUDIO state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
													+ " CONVERSATION state: " + conversation.state());
										if (newValue === "Disconnected") {
											log.trace("<<< AUDIO disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
											if (oldValue === "Connected" || oldValue === "Connecting") {
												conversation.selfParticipant.audio.state.changed.off(listener);
												if (conversation.participantsCount() <= 0) {
													updateCall("stopped");											
												}
											}
										}
									});
									conversation.selfParticipant.video.state.changed(function listener(newValue, reason, oldValue) {
										log.trace(">>> VIDEO state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
													+ " CONVERSATION state: " + conversation.state());
										if (newValue === "Disconnected") {
											log.trace("<<< VIDEO disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
											if (oldValue === "Connected" || oldValue === "Connecting") {
												conversation.selfParticipant.video.state.changed.off(listener);
												if (conversation.participantsCount() <= 0) {
													updateCall("stopped");											
												}
											}
										}
									});*/
								} // P2P call will be saved before the start, see below
								conversation.state.once("Disconnected", function() {
									log.debug("Disconnected outgoing call: " + callId);
									if (conversation.isGroupConversation()) {
										leavedGroupCall();
										/*activeParticipants(conversation).done(function(list) {
											if (list.length <= 0) {
												updateCall("stopped");
												log.trace("<<< Stopped outgoing conference " + callId);
											} else  {
												log.trace("<<< Still active outgoing conference " + callId + " parts: " + list.length);
											}
										});*/
										/*conversation.participantsCount.get().then(function(res) {
											if (res <= 0) {
												updateCall("stopped");
												log.trace("<<< Stopped outgoing conference " + callId);
											} else  {
												log.trace("<<< Still running outgoing conference " + callId + " parts: " + res);
											}
										});*/
									} else {
										deleteCall();
									}
									window.removeEventListener("beforeunload", beforeunloadListener);
									window.removeEventListener("unload", unloadListener);
									//container.hide();
									if (process.state() == "pending") {
										process.reject("disconnected");
									}
								});								
							}
						};
						if (target.group && conversation.selfParticipant.person.id() == conversation.creator.id()) {
							// set topic only if a creator and of a group call, 
							// otherwise it's R/O property for other parts and for p2p we don't want set any topic now
							try {
								conversation.topic(target.title);
							} catch(e) {
								log.warn("Failed to set conversation topic: " + target.title, e);
							}
						}
						if (!localCall) {
							conversation.participants.added(function(part) {
								log.info("Participant added " + part.person.displayName() + "(" + part.person.id() + ") to call: " + callId);
							});
						}
						container.setConversation(conversation, callId, target.id);
						container.show();
						//process.notify(callId, conversation); // inform call is starting to the invoker code
						if (conversation.isGroupConversation()) {
							// FYI This doesn't work for group calls, at this stage a newly created convo will not have an URI, 
							// thus the call ID will be like g/adhoc, not what other parts will see in added one
							/*conversation.state.once("Conferencing", function() {
								startingCall("Conferencing");
							});*/							
							/* At this state the convo ID will be adhoc, need do on conversation service start
							 conversation.state.once("Conferenced", function() {
								startingCall("Conferenced");
							});*/
							/*conversation.participants.added(function(person) {
						    // Another participant has accepted an invitation and joined the conversation
								log.trace(">>> Added participant to outgoing Conference " + callId + " " + JSON.stringify(person));
							});*/
						} else {
							// XXX For P2P, save in eXo sooner, to let this info be ready when the convo will be fired as 'added' at the other side
							//startingCall(conversation.state());
							addCall();
							conversation.state.once("Connecting", function() {
								startingCall("Connecting");
							});
						}
						conversation.videoService.start().then(function() {
							log.debug("Outgoing video STARTED ");
							startingCall("Started");
						}, function(videoError) {
							// error starting videoService, cancel (by this user) also will go here
							log.warn("Failed to start outgoing video for: " + callId, videoError);
							var finisWithError = function(error) {
								handleError(callId, error, function(isError, isOutdated) {
									// For a case when Disconnected will not happen
									window.removeEventListener("beforeunload", beforeunloadListener);
									window.removeEventListener("unload", unloadListener);
									if (isOutdated) {
										app.conversationsManager.conversations.remove(conversation);
										// delete this call records in eXo service
										target.callId = null;
										deleteCall().always(function() {
											// Wait a bit for things completion
											setTimeout(function() {
												outgoingCallHandler(api, app, container, currentUser, target, userIds, participants, null).done(function(callId, newConvo) {
													process.resolve(callId, newConvo);
												}).fail(function(err) {
													process.reject(err);
												});											
											}, 250);
										});
									} else {
										if (process.state() == "pending") {
											process.reject(isError ? error : "canceled");
										}
										if (conversation.isGroupConversation()) {
											leavedGroupCall();
										} else {
											deleteCall();
										}
									}
								});
							};
							if (isModalityUnsupported(videoError)) {
								// ok, try audio
								conversation.audioService.start().then(function() {
									log.debug("Outgoing audio STARTED ");
									startingCall("Started");
								}, function(audioError) {
									log.warn("Failed to start outgoing audio for: " + callId, audioError);
									if (isModalityUnsupported(audioError)) {
										// well, it will be chat (it should work everywhere)
										conversation.chatService.start().then(function() {
											log.debug("Outgoing chat STARTED ");
										}, function(chatError) {
											log.warn("Failed to start a chat for outgoing call: " + callId, chatError);
											// we deal with original error
											finisWithError(videoError);
										});
									} else {
										finisWithError(videoError);
									}
								});
							} else {
								finisWithError(videoError);
							}
						});
						window.addEventListener("beforeunload", beforeunloadListener);
						window.addEventListener("unload", unloadListener);
					}, function(err) {
						// error rendering Conversation Control
						container.hide();
						process.reject(err);
						var title = "Conversation rendering error";
						var msg = "";
						if (err.name && err.message) {
							msg = err.name + " " + err.message;
						} else {
							msg = webConferencing.errorText(err);
						}
						log.showError(title, err, title, msg);
					});					
				}).fail(function() {
					container.$element.append($("<div><div class='pluginError'>Please install Skype web plugin.</div></div>"));
					process.reject("plugin error"); // problem already logged in checkPlugin()
				});
				return process.promise();
			};
			this.startOutgoingCall = function(api, app, container, details, localCall) {
				var currentUser = webConferencing.getUser();
				return outgoingCallHandler(api, app, container, currentUser, details.target, details.users, details.participants, localCall);
			};
			
			var makeCallPopover = function(title, message) {
				// TODO show an info popover in bottom right corner of the screen as specified in CALLEE_01
				log.trace(">> makeCallPopover '" + title + "' message:" + message);
				var process = $.Deferred();
				var $call = $("div.uiOutgoingCall");
				if ($call.length > 0) {
					try {
						$call.dialog("destroy");
					} catch(e) {
						log.warn("makeCallPopover: error destroing previous dialog ", e);
					}
					$call.remove();
				}
				$call = $("<div class='uiOutgoingCall' title='" + title + " call'></div>");
				$call.append($("<p><span class='ui-icon messageIcon' style='float:left; margin:12px 12px 20px 0;'></span>"
					//+ "<a target='_self' href='" + callerLink + "'><img src='" + callerAvatar + "'></a>"
					+ "<div class='messageText'>" + message + "</div></p>"));
				$(document.body).append($call);
				$call.dialog({
		      resizable: false,
		      height: "auto",
		      width: 400,
		      modal: false,
		      buttons: {
		        "Call": function() {
		        	process.resolve("confirmed");
		        	$call.dialog( "close" );
		        },
		        "Cancel": function() {
		        	process.reject("canceled");
		        	$call.dialog( "close" );
		        }
		      }
				});
				return process.promise();
			};
			
			var stopCallPopover = function(title, message) {
				log.trace(">> stopCallPopover '" + title + "' message:" + message);
				var process = $.Deferred();
				var $call = $("div.uiStopCall");
				if ($call.length > 0) {
					try {
						$call.dialog("destroy");
					} catch(e) {
						log.warn("stopCallPopover: error destroing previous dialog ", e);
					}
					$call.remove();
				}
				$call = $("<div class='uiStopCall' title='" + title + " call'></div>");
				$call.append($("<p><span class='ui-icon warningIcon' style='float:left; margin:12px 12px 20px 0;'></span>"
					//+ "<a target='_self' href='" + callerLink + "'><img src='" + callerAvatar + "'></a>"
					+ "<div class='messageText'>" + message + "</div></p>"));
				$(document.body).append($call);
				$call.dialog({
		      resizable: false,
		      height: "auto",
		      width: 400,
		      modal: false,
		      buttons: {
		        "Call": function() {
		        	process.resolve("confirmed");
		        	$call.dialog( "close" );
		        },
		        "Cancel": function() {
		        	process.reject("canceled");
		        	$call.dialog( "close" );
		        }
		      }
				});
				return process.promise();
			};
			
			var showWrongUsers = function(wrongUsers, callWindow) {
				if (wrongUsers.length > 0) {
					var userNames = "";
					for (var i=0; i<wrongUsers.length; i++) {
						if (i > 0) {
							userNames += ", ";
						}
						var wu = wrongUsers[i];
						userNames += wu.firstName + " " + wu.lastName;
					}
					var s, have, who; 
					if (wrongUsers.length > 1) {
						s = "s";
						have = "have";
						who = "They were";
					} else {
						s = "";
						have = "has";
						who = "It was";
					}
					var title = "Wrong Skype account" + s;
					var message = "Following user" + s + " " + have + " wrong business account: " + 
						userNames + ". " + who + " not added to the call.";
					log.warn(title + ". " + message);
					if (callWindow) {
						// inform the caller in call window
						$(callWindow).on("load", function() {
							notifyCaller(callWindow, title, message);
						});
					} else {
						// inform on this page
						webConferencing.showWarn(title, message);
					}
				}
			};
			
			var targetDetails = function(currentUserSip, target) {
				var participants = []; // for embedded call
				//var ims = []; // for call on a new page
				var users = [];
				var wrongUsers = [];
				var addParticipant = function(user) {
					//var uskype = webConferencing.imAccount(u, "skype");
					var ubusiness = webConferencing.imAccount(user, "mssfb");
					if (ubusiness) {
						if (EMAIL_PATTERN.test(ubusiness.id)) {
							if (ubusiness.id != currentUserSip) {
								participants.push(ubusiness.id);
								//ims.push(encodeURIComponent(ubusiness.id));
							}
							users.push(user.id);
						} else {
							wrongUsers.push(ubusiness);
						}
					} // else, skip this user
				};
				if (target.group) {
					for ( var uname in target.members) {
						if (target.members.hasOwnProperty(uname)) {
							var u = target.members[uname];
							addParticipant(u);
						}
					}								
				} else {
					users.push(webConferencing.getUser().id);
					addParticipant(target);
				}
				return { 
					target : target, 
					users : users,
					wrongUsers : wrongUsers, 
					participants : participants
				}
			};
			this.readTargetDetails = targetDetails;
			
			this.callButton = function(context) {
				var button = $.Deferred();
				if (settings && context && context.currentUser) {
					// TODO temporarily we don't support calling regular Skype users
					//context.currentUserSkype = webConferencing.imAccount(context.currentUser, "skype");
					var currentUserSFB = webConferencing.imAccount(context.currentUser, "mssfb");
					if (currentUserSFB) {
						context.details().done(function(target) {
							var details = targetDetails(currentUserSFB.id, target);
							var targetId = webConferencing.contextId(context);
							if (details.participants.length > 0) {
								context.currentUserSFB = currentUserSFB;
								var fullTitle = self.getTitle() + " " + self.getCallTitle();
								var localCall = getLocalCall(targetId);
								var title = localCall && canJoin(localCall) ? self.getJoinTitle() : self.getCallTitle();
								var disabledClass = hasJoinedCall(targetId) ? "callDisabled" : "";
								var $button = $("<a title='" + fullTitle + "' href='javascript:void(0);' class='mssfbCallAction "
											+ disabledClass + "'>"
											+ "<i class='uiIconMssfbCall uiIconLightGray'></i>"
											+ "<span class='callTitle'>" + title + "</span></a>");
								setTimeout(function() {
									if (!$button.hasClass("btn")) {
										// in dropdown show longer description
										$button.find(".callTitle").text(fullTitle);
									}
								}, 1000);
								$button.click(function() {
									if (!hasJoinedCall(targetId)) {
										var token = currentToken();
										var container = $("#mssfb-call-container").data("callcontainer");
										if (container) {
											// Call will run inside current page
											// We need SDK API/app instance - try reuse saved locally SfB token
											// TODO get rid of this method
											var callDetails = function(callback) {
												// TODO if we want here get in account users added to a group while page with this button was open, 
												// then need re-request details from the server
												callback(context.currentUser, details.target, details.users, details.wrongUsers, details.participants);
												//
												/*context.details().done(function(target) {
													var details = targetDetails(currentUserSFB.id, target);
													if (details.participants.length > 0) {
														callback(context.currentUser, details.target, details.users, details.wrongUsers, details.participants);
													} else {
														webConferencing.showWarn("Cannot start a call", "No " + self.getTitle() + " users found.");
													}
												}).fail(function(err) {
													webConferencing.showWarn("Error starting a call", err.message);
												});*/
											};
											var embeddedCall = function(api, app) {
												callDetails(function(currentUser, target, users, wrongUsers, participants) {
													if (participants.length > 0) {
														var localCall;
														if (target.callId) {
															localCall = getLocalCall(target.callId);
														}
														if (container.isVisible()) {
															var currentConvo = container.getConversation();
															var message = "Do you want " 
																	+ (currentConvo.isGroupConversation() ? "leave '" + currentConvo.topic() + "' call" : "stop call with " + currentConvo.participants(0).person.displayName())
																	+ " and start " 
																	+ (target.group ? "'" + target.title + "' call" : " call to " + target.title) + "?";
															stopCallPopover("Start a new " + self.getTitle() + " call&#63;", message).done(function() {
																container.getConversation().leave().then(function() {
																	log.trace("<<< Current conversation leaved " + container.getCallId() + " for " + (localCall ? "saved" : "new") + " outgoing");
																});
																outgoingCallHandler(api, app, container, currentUser, target, users, participants, localCall);
																showWrongUsers(wrongUsers);
															}).fail(function() {
																log.trace("User don't want stop existing call and call to " + target.title);
															});
														} else {
															outgoingCallHandler(api, app, container, currentUser, target, users, participants, localCall);
															showWrongUsers(wrongUsers);													
														}
													} else {
														log.warn("Cannot start a call: no " + self.getTitle() + " users found for " + targetId);
														webConferencing.showWarn("Cannot start a call", "No " + self.getTitle() + " users found.");
													}										
												});
											};
											if (token && uiApiInstance && uiAppInstance) {
												log.info("Automatic login done.");
												embeddedCall(uiApiInstance, uiAppInstance);
											} else {
												// we need try SfB login window in hidden iframe (if user already logged in AD, then it will work)
												// FYI this iframe will fail due to 'X-Frame-Options' to 'deny' set by MS
												// TODO may be another URL found to do this? - see incoming call handler for code
												// need login user explicitly (in a popup)
												log.warn("Automatic login failed: login token not found or expired");
												var login = loginWindow();
												self.loginToken = function(token) {
													var callback = loginTokenHandler(token);
													callback.done(function(api, app) {
														log.info("User login done.");
														embeddedCall(api, app);
													});
													callback.fail(function(err) {
														log.showError("User login failed", err, self.getTitle() + " error", "Unable sign in your " 
																	+ self.getTitle() + " account. " + webConferencing.errorText(err));
													});
													return callback.promise();
												};
												if (!login) {
													log.error("User login failed due to blocked call popup");
												}
											}
										} else {
											// Call will run in a new page using call reference and call title,
											// Call reference format here: 'target_type'/'target_id', the call page will request the target details in REST service
											var callId;
											if (context.userId) {
												callId = "user/" + context.userId;
											} else if (context.spaceId) {
												callId = "space/" + context.spaceId;
											} else if (context.roomId) {
												callId = "chat_room/" + context.roomId;
											} else {
												log.error("Unsupported call context " + context);
											}
											if (callId) {
												var callWindow = openCallWindow(callId, context);
												callWindow.document.title = fullTitle + "...";
											}
										}								
									}
								});
								button.resolve($button);
							} else {
								var msg = "No " + self.getTitle() + " users found for " + targetId;
								log.debug(msg);
								button.reject(msg);
							}
						}).fail(function(err) {
							if (err && err.code == "NOT_FOUND_ERROR") {
								button.reject(err.message);
							} else {
								var msg = "Error getting context details";
								log.error(msg, err);
								button.reject(msg, err);
							}
						});
					} else {
						var msg = "Not " + self.getTitle() + " user " + context.currentUser.id;
						log.debug(msg);
						button.reject(msg);
					}
				} else {
					var msg = "Not configured or empty context for " + self.getTitle();
					log.error(msg);
					button.reject(msg);
				}
				return button.promise();
			};
				
			var acceptCallPopover = function(callerLink, callerAvatar, callerMessage, withRing) {
				// TODO show an info popover in bottom right corner of the screen as specified in CALLEE_01
				log.trace(">> acceptCallPopover '" + callerMessage + "' caler:" + callerLink + " avatar:" + callerAvatar);
				var process = $.Deferred();
				var $call = $("div.uiIncomingCall");
				if ($call.length > 0) {
					try {
						$call.dialog("destroy");
					} catch(e) {
						log.warn("acceptCallPopover: error destroing previous dialog ", e);
					}
					$call.remove();
				}
				$call = $("<div class='uiIncomingCall' title='" + self.getTitle() + " call'></div>");
				$call.append($("<div class='messageAuthor'><a target='_blank' href='" + callerLink + "' class='avatarMedium'>"
					+ "<img src='" + callerAvatar + "'></a></div>"
					+ "<div class='messageBody'><div class='messageText'>" + callerMessage + "</div></div>"));
				// eXo UX guides way
				//$call.append($("<ul class='singleMessage popupMessage'><li><span class='messageAuthor'>"
				//		+ "<a class='avatarMedium'><img src='" + callerAvatar + "'></a></span>"
				//		+ "<span class='messageText'>" + callerMessage + "</span></li></ul>"));
				$(document.body).append($call);
				$call.dialog({
					resizable: false,
					height: "auto",
					width: 400,
					modal: false,
					buttons: {
					  "Answer": function() {
					  	process.resolve("accepted");
					  	$call.dialog("close");
					  },
					  "Decline": function() {
					  	process.reject("declined");
					  	$call.dialog("close");
					  }
					} 
				});
				$call.on("dialogclose", function( event, ui ) {
					if (process.state() == "pending") {
						process.reject("closed");
					}
				});
				process.notify($call);
				if (withRing) { 
					var $ring = $("<audio controls loop autoplay style='display: none;'>"
								+ "<source src='https://latest-swx.cdn.skype.com/assets/v/0.0.300/audio/m4a/call-outgoing-p1.m4a' type='audio/mpeg'>"  
								+ "Your browser does not support the audio element.</audio>");
					$(document.body).append($ring);
					var player = $ring.get(0);
					//player.pause();
					//player.currentTime = 0;
					// TODO this doesn't work on mobile - requires user gesture
					/*setTimeout(function () {      
						player.play();
					}, 250);*/
					process.always(function() {
						player.pause();
						$ring.remove();
					});
				}
				return process.promise();
			};
			
			var incomingCallHandler = function(api, app, container) {
				var $callPopup;
				var closeCallPopup = function(callId) {
					if ($callPopup && $callPopup.callId && $callPopup.callId == callId) {
						if ($callPopup.is(":visible")) {
							$callPopup.dialog("close");
						}
					}
				};
				var handled = {};
				// This function handles a single conversation object, added in SDK or notified via eXo user update
				var handleIncoming = function(conversation, saved) {
					var callId = getCallId(conversation);
					log.info(conversation.state() + " call: " + callId);
					var accept;
					var callerMessage, callerLink, callerAvatar, callerId, callerType, callerRoom;
					var callStateUpdate = function(state, modality, error) {
						var res = { 
							saved : saved,
							callId : callId,
							peer : {
								id : callerId,
								type : callerType,
								chatRoom : callerRoom
							},
							conversation : conversation
						};
						if (modality) {
							res.modality = modality;
						}
						if (error) {
							res.error = error;
						} else {
							res.state = state;
						}
						callUpdate(res);
					};
					// As conversation may be notified several times via eXo user update, we want set its listeners
					// only once (to avoid multiplication of events)
					// TODO callId may change in case of escalation P2P to group call
					if (!handled[callId]) {
						handled[callId] = true;
						var beforeunloadListener = function(e) {
							var msg = onClosePage(conversation, app);
							if (msg) {
								e.returnValue = msg; // Gecko, Trident, Chrome 34+
								return msg; // Gecko, WebKit, Chrome <34
							}
						};
						var unloadListener = function(e) {
							onClosePage(conversation, app);
						};
						if (conversation.isGroupConversation()) {
							callerId = conversation.topic(); // this not true, need space/room title
							callerMessage = callerId; 
							callerLink = ""; // /portal/g/:spaces:marketing_team/marketing_team
							callerAvatar = ""; // /rest/social/identity/space/marketing_team/avatar
						} else {
							// remote participant it's who is calling to
							var caller = conversation.participants(0).person;
							callerId = caller.displayName();
							callerMessage = caller.displayName() + " is calling.";
							callerLink = ""; // /portal/intranet/profile/patrice
							callerAvatar = caller.avatarUrl(); // /rest/social/identity/organization/patrice/avatar
						}
						var joined = false;
						var joinedGroupCall = function() {
							if (!joined) {
								joined = true;
								webConferencing.updateUserCall(callId, "joined").fail(function(err) {
									log.error("Failed to join group call: " + callId, err);
								});
							}
						};
						var leavedGroupCall = function() {
							joined = false; // do this sooner
							webConferencing.updateUserCall(callId, "leaved").fail(function(err) {
								log.error("Failed to leave group call: " + callId, err);
							});
						};
						// TODO add events after the start/join
						if (conversation.isGroupConversation()) {
							conversation.state.when("Conferencing", function() {
								log.debug("Conferencing incoming: " + callId);
							});
							conversation.state.when("Conferenced", function() {
								log.info("Conferenced incoming: " + callId + " parts: " + conversation.participantsCount());
								window.addEventListener("beforeunload", beforeunloadListener);
								window.addEventListener("unload", unloadListener);
							});
							/*conversation.participants.added(function(person) {
						    // Another participant has accepted an invitation and joined the conversation
								log.trace(">>> Added participant to incoming Conference " + callId + " " + JSON.stringify(person));
							});*/
						} else {
							conversation.state.when("Connecting", function() {
								log.debug("Connecting incoming: " + callId);
								window.addEventListener("beforeunload", beforeunloadListener);
								window.addEventListener("unload", unloadListener);
							});
							conversation.state.when("Connected", function() {
								log.info(">>> Connected incoming: " + callId);
							});
							// If person added and it was a P2P, we have call escalation to a conference, need update callId
							// This will happen if add user inside the SfB call UI, an external user etc.
							conversation.participants.added(function(part) {
								//var oldCallId = callId;
								var callId = getCallId(conversation);
								log.debug("Participant added " + part.person.id() + " to " + callId);
							});
						}
						conversation.state.when("Disconnected", function() {
							log.info("Disconnected incoming: " + callId);
							//accept = null; // TODO - see always/timeout below. Release the acceptor for a next call within this convo
							if (conversation.isGroupConversation()) {
								callStateUpdate("leaved");
								leavedGroupCall();
								/*activeParticipants(conversation).done(function(list) {
									if (list.length <= 0) {
										webConferencing.updateCall(callId, "stopped").done(function(call) {
											log.trace("<<< Stopped incoming conference " + callId);
										}).fail(function(err) {
											log.trace("<<< ERROR updating to stopped " + callId + ": " + JSON.stringify(err));
										});
									} else  {
										log.trace("<<< Still active incoming conference " + callId + " parts: " + list.length);
									}
								});*/
								/*conversation.participantsCount.get().then(function(res) {
									if (res <= 0) {
										webConferencing.updateCall(callId, "stopped").done(function(call) {
											log.trace("<<< Stopped " + callId + " parts:" + conversation.participantsCount() + " > " + new Date().getTime());
										}).fail(function(err) {
											log.trace("<<< ERROR updating to stopped " + callId + ": " + JSON.stringify(err));
										});
									}
								});*/
							} else {
								callStateUpdate("stopped");
							}
							window.removeEventListener("beforeunload", beforeunloadListener);
							window.removeEventListener("unload", unloadListener);
						});
					}
					
					// This method may be invoked several times by the above SDK conversation's services accept callback
					// be we want ask an user once per a call, thus we will reset this deferred in seconds.
					// The 'accept' variable used by single (!) conversation, but acceptCall() can be invoked several times
					// by eXo user update event
					var acceptCall = function() {
						if (!accept) {
							accept = $.Deferred();
							accept.always(function() {
								// wait a bit for latency of video/audio notification in Skype SDK and release it for next attempts
								setTimeout(function() {
									accept = null;
								}, 2500);
							});
							var showCallPopover = function() {
								if (container.isVisible()) {
									callerMessage += " Your current call will be stopped if you answer this call.";
								}
								var popover = acceptCallPopover(callerLink, callerAvatar, callerMessage, /*saved &&*/ conversation.isGroupConversation());
								popover.progress(function($call) {
									$callPopup = $call;
									$callPopup.callId = callId;
									conversation.state.changed(function listener(newValue, reason, oldValue) {
										// convo may be already in Disconnected state for saved calls
										if (newValue === "Disconnected" && (oldValue === "Incoming" || oldValue === "Connected" || oldValue === "Connecting" || oldValue === "Conferencing" || oldValue === "Conferenced")) {
											conversation.state.changed.off(listener);
											if ($call.is(":visible")) {
												$call.dialog("close");
											}
											closeCallPopup(callId);
										}
									});
								}); 
								popover.done(function(msg) {
									log.info("User " + msg + " call: " + callId);
									if (container) {
										// TODO support several calls at the same time
										// If call container already running a conversation, we leave it and start this one 
										if (container.isVisible()) { // && callId != container.getCallId()
											container.getConversation().leave().then(function() {
												log.info("Current conversation leaved " + container.getCallId() + " for accepted incoming");
											});
										}
										callStateUpdate("accepted");
										container.init();
										var options = {
											conversation : conversation,
											modalities : [ "Chat" ]
										};
										api.renderConversation(container.element, options).then(function(conversation) {
											log.info("Incoming call '" + conversation.topic() + "': " + callId);
											container.setConversation(conversation, callId, callerId);
											container.show();
											if (saved && conversation.isGroupConversation()) { // was saved
												// This should work for saved (reused) group conference calls, 
												// P2P and new incoming conference will not work this way
												// TODO two state.changed below for debug info only
												conversation.selfParticipant.audio.state.changed(function listener(newValue, reason, oldValue) {
													log.debug("AUDIO state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
																+ " CONVERSATION state: " + conversation.state());
													/*if (newValue === "Disconnected") {
														log.trace("<<< AUDIO disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
														if (oldValue === "Connected" || oldValue === "Connecting") {
															conversation.selfParticipant.audio.state.changed.off(listener);
															if (conversation.participantsCount() <= 0) {
																updateCall("stopped");											
															}
														}
													}*/
												});
												conversation.selfParticipant.video.state.changed(function listener(newValue, reason, oldValue) {
													log.debug("VIDEO state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
																+ " CONVERSATION state: " + conversation.state());
													/*if (newValue === "Disconnected") {
														log.trace("<<< VIDEO disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
														if (oldValue === "Connected" || oldValue === "Connecting") {
															conversation.selfParticipant.video.state.changed.off(listener);
															if (conversation.participantsCount() <= 0) {
																updateCall("stopped");											
															}
														}
													}*/
												});
												// When conversation was accepted previously, we just start modality on it
												conversation.videoService.start().then(function() {
													log.debug("Incoming (saved) VIDEO STARTED");
													accept.resolve("video");
												}, function(videoError) {
													// error starting videoService, cancel (by this user) also will go here
													log.warn("Failed to start incoming (saved) VIDEO for: " + callId, videoError);
													if (isModalityUnsupported(videoError)) {
														// ok, try audio
														conversation.audioService.start().then(function() {
															log.debug("Incoming (saved) AUDIO STARTED");
															accept.resolve("audio");
														}, function(audioError) {
															log.warn("Failed to start incoming (saved) AUDIO for: " + callId, audioError);
															if (isModalityUnsupported(audioError)) {
																// well, it will be chat (it should work everywhere)
																conversation.chatService.start().then(function() {
																	log.debug("Incoming (saved) CHAT STARTED ");
																	accept.resolve("chat");
																}, function(chatError) {
																	log.warn("Failed to start CHAT for incoming (saved): " + callId, chatError);
																	// we deal with original error
																	accept.reject(videoError);
																});
															} else {
																accept.reject(videoError);
															}
														});
													} else {
														accept.reject(videoError);
													}
												});
											} else {
												// TODO check this working in FF where no video/audio supported
												conversation.selfParticipant.video.state.changed(function listener(newValue, reason, oldValue) {
													// 'Notified' indicates that there is an incoming call
													log.debug("VIDEO state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
																+ " CONVERSATION state: " + conversation.state());
													if (newValue === "Notified") {
														// TODO in case of video error, but audio or chat success - show a hint message to an user and auto-hide it
														log.trace("Incoming VIDEO ACCEPTING Notified");
														conversation.videoService.accept().then(function() {
															log.debug("Incoming VIDEO ACCEPTED");
															accept.resolve("video");
														}, function(videoError) {
															// error starting videoService, cancel (by this user) also will go here
															log.warn("Failed to accept video for: " + callId, videoError);
															if (!isModalityUnsupported(videoError)) {
																accept.reject(videoError);
															}
														});
													} else if (newValue === "Disconnected") {
														log.debug("VIDEO disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
														if (oldValue === "Connected" || oldValue === "Connecting") {
															conversation.selfParticipant.video.state.changed.off(listener);
															if (reason && typeof reason === "string" && reason.indexOf("PluginUninited") >= 0) {
																log.error("Skype plugin not initialized");
																webConferencing.showError("Skype Plugin Not Initialized", 
																			"Please install <a href='https://support.skype.com/en/faq/FA12316/what-is-the-skype-web-plugin-and-how-do-i-install-it'>Skype web plugin</a> to make calls.");
															}
														}
													}
												});
												conversation.selfParticipant.audio.state.changed(function listener(newValue, reason, oldValue) {
													log.debug("AUDIO state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
																+ " CONVERSATION state: " + conversation.state());
													if (newValue === "Notified") {
														log.trace("Incoming AUDIO ACCEPTING Notified");
														conversation.audioService.accept().then(function() {
															log.debug("Incoming AUDIO ACCEPTED");
															accept.resolve("audio");
														}, function(audioError) {
															log.warb("Error accepting audio for: " + callId, audioError);
															if (!isModalityUnsupported(audioError)) {
																accept.reject(audioError);
															}
														});
													} else if (newValue === "Disconnected") {
														log.debug("AUDIO disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
														if (oldValue === "Connected" || oldValue === "Connecting") {
															conversation.selfParticipant.audio.state.changed.off(listener);
															if (reason && typeof reason === "string" && reason.indexOf("PluginUninited") >= 0) {
																log.error("Skype plugin not initialized");
																webConferencing.showError("Skype Plugin Not Initialized", 
																			"Please install <a href='https://support.skype.com/en/faq/FA12316/what-is-the-skype-web-plugin-and-how-do-i-install-it'>Skype web plugin</a> to make calls.");
															}
														}
													}
												});
												conversation.selfParticipant.chat.state.changed(function listener(newValue, reason, oldValue) {
													log.debug("CHAT state changed " + callId + ": " + oldValue + "->" + newValue + " reason:" + reason
																+ " CONVERSATION state: " + conversation.state());
													if (newValue === "Notified") {
														conversation.chatService.accept().then(function() {
															log.trace("Incoming CHAT ACCEPTED");
															accept.resolve("chat");
														}, function(chatError) {
															log.warn("Error accepting chat for: " + callId, chatError);
															if (!isModalityUnsupported(chatError)) {
																accept.reject(chatError);
															}
														});
													} else if (newValue === "Disconnected") {
														log.debug("CHAT disconnected for call " + callId + " CONVERSATION state: " + conversation.state());
														if (oldValue === "Connected" || oldValue === "Connecting") {
															conversation.selfParticipant.chat.state.changed.off(listener);
														}
													}
												});
											}
										}, function(err) {
											// error rendering Conversation Control
											var title = "Conversation rendering error";
											var msg = "";
											if (err.name && err.message) {
												msg = err.name + " " + err.message;
											} else {
												msg = webConferencing.errorText(err);
											}
											log.showError(title, err, title, msg);
											container.hide();
											accept.reject("conversation error");
										});
									} else {
										var msg = "UI container not found";
										log.error(msg);
										accept.reject(msg);
									}
								});
								popover.fail(function(err) {
									log.info("User " + err + " call: " + callId);
									accept.reject(err);
								});
								popover.always(function() {
									if (accept && accept.state() == "pending") {
										var thisAccept = accept;
										setTimeout(function() {
											if (thisAccept.state() == "pending") {
												log.trace(">>> accept STILL pending - reject it");
												// if in 15sec no one conversation service was accepted, we reject it
												try {
													thisAccept.reject("timeout");															
												} catch(e) {
													// was null due to below timer?
												}
											}
										}, 15000);	
									}
								});
							};
							if (hasJoinedCall(callId)) { // container.isVisible() && callId == container.getCallId()
								// User already running this call (this may happen if remote initiator leaved and joined again)
								accept.resolve();
							} else {
								webConferencing.getCall(callId).done(function(call) {
									log.trace(">>> Got registered " + callId);
									callerId = call.owner.id;
									callerLink = call.owner.profileLink;
									callerAvatar = call.owner.avatarLink;
									callerType = call.owner.type;
									if (call.owner.group) {
										callerMessage = call.title + " conference call.";
									} // otherwise, we already have a right message (see above)
									if (callerType == "space") {
										// Find room ID for space calls in Chat
										webConferencing.getChat().getRoom(callerId, "space-name").done(function(room) {
											if (room) {
												callerRoom = room.user;
											} else {
												log.warn("Chat room not found for space " + callerId);
												callerRoom = null;
											}
											callStateUpdate("incoming");
										}).fail(function(err) {
											log.error("Failed to request Chat room: " + callerId + " for " + callId, err);
										});
									} else {
										// we assume: else if (callerType == "chat_room" || callerType == "user") 
										callerRoom = callerId;
										callStateUpdate("incoming");
									}
									showCallPopover();
								}).fail(function(err) {
									if (err) {
										if (err.code == "NOT_FOUND_ERROR") {
											// Call not registered, we treat it as a call started outside web conferencing
											log.warn("Incoming call not found: " + callId + ", treat it as an external one", err);
											showCallPopover();
										}	else {
											log.error("Failed to get call info for: " + callId, err);
											accept.reject(webConferencing.errorText(err));
										}
									} else {
										log.error("Error getting call info for: " + callId);
										accept.reject("call info error");
									}
								});								
							}
							accept.done(function(modality) {
								if (conversation.isGroupConversation()) {
									callStateUpdate("joined", modality);
									joinedGroupCall();
								} else {
									callStateUpdate("started", modality);									
								}
							});
							accept.fail(function(err) {
								if (err == "declined" || err == "closed" || err == "timeout") {
									callStateUpdate("canceled");
								} else if (err == "closed") {
									// TODO User closed the accept popup: not accepted not declined, was not on place? 
									// We don't update the state at all, is it correct?
								} else {
									callStateUpdate("error", null, err);
								}
							});
						}
						return accept.promise();
					};
					
					// TODO rework the logic to handle added and saved differently: add convo listeners for saved once only
					if (saved) {
						acceptCall();
					} else {
						conversation.videoService.accept.enabled.when(true, function() {
							log.debug("videoService ACCEPT: " + callId);
							acceptCall().fail(function(err) {
								conversation.videoService.reject();
								log.debug("videoService REJECTED " + callId, err);
							});
						});
						conversation.audioService.accept.enabled.when(true, function() {
							log.debug("audioService ACCEPT: " + callId);
							acceptCall().fail(function(err) {
								conversation.audioService.reject();
								log.debug("audioService REJECTED " + callId, err);
							});
						});
						conversation.chatService.accept.enabled.when(true, function() {
							log.debug("chatService ACCEPT: " + callId);
							// TODO chat needs specific handling
						});
						// TODO audioPhoneService accept?
					}
				};
				checkPlugin(app).done(function() {
					// We want handle both "Incoming" added by remote part and "Created" added here as outgoing and later re-used as "Incoming"
					app.conversationsManager.conversations.removed(function(conversation, key, index) {
						var callId = getCallId(conversation);
						removeLocalCall(callId);
						delete handled[callId];
						delete logging[key];
					});
					app.conversationsManager.conversations.added(function(conversation, key, index) {
						var state = conversation.state();
						log.debug(state + " (added) call: " + getCallId(conversation));
						logConversation(conversation, key, index);
						if (state == "Incoming") {
							handleIncoming(conversation);
						} else if (state == "Created") {
							// Late it may become Incoming
							conversation.state.once("Incoming", function() {
								// Update call ID as it will change after first Connecting
								var callId = getCallId(conversation);
								if (!hasJoinedCall(callId)) { //  (container.isVisible() && callId != container.getCallId())
									log.debug("Created (added) incoming call: " + callId);
									setTimeout(function() {
										// We let some time to Incoming listener below in user updates polling
										var localCall = getLocalCall(callId);
										handleIncoming(conversation, localCall ? localCall.saved : false);										
									}, 250);
								} else {
									log.debug("Created (added) incoming already handled: " + callId);
								}
							});
						}
					});
					// We also handle re-established group conferences (while it is not outdated by Skype services)
					webConferencing.getUserGroupCalls().done(function(list) {
						var checkCallJoined = function(call) {
							if (call.state == "started" && call.participants) {
								for (var i=0; i<call.participants.length; i++) {
									var part = call.participants[i];
									if (part.id == webConferencing.getUser().id && part.state == "joined") {
										// Need leave this user from the call, if it's last user there the call will be stopped, 
										// if not - then user still able to join it manually
										webConferencing.updateUserCall(call.id, "leaved").fail(function(err) {
											log.error("Failed to leave group call: " + call.id, err);
										});
										return true;			
									}
								}
							}
							return false;
						};
						var callStarted = function(call, peer) {
							callUpdate({
								state : "started",
								callId : call.id,
								peer : peer,
								saved : true
							});
							if (checkCallJoined(call)) {
								callUpdate({
									state : "leaved",
									callId : call.id,
									peer : peer,
									saved : false
								});
							}
						};
						for (var i=0; i<list.length; i++) {
							var callState = list[i];
							if (callState.state == "started") {
								// Mark it started in Chat
								webConferencing.getCall(callState.id).done(function(call) {
									if (call.providerType == self.getType()) {
										if (call.owner.type == "space") {
											// Find room ID for space calls in Chat
											webConferencing.getChat().getRoom(call.owner.id, "space-name").done(function(room) {
												callStarted(call, {
													id : call.owner.id,
													type : call.owner.type,
													chatRoom : room ? room.user : null
												});
											}).fail(function(err) {
												log.error("Failed to request Chat room: " + call.owner.id + " for: " + callId, err);
											});
										} else {
											// we assume: else if (callerType == "chat_room" || callerType == "user")
											callStarted(call, {
												id : call.owner.id,
												type : call.owner.type,
												chatRoom : call.owner.id
											});
										}
									} // don't care about calls of other providers
								}).fail(function(err) {
									log.error("Failed to get call info: " + callId, err);
								});
							}
						}
						//log.trace(">>> User's registered group calls: " + JSON.stringify(list));
						var userId = webConferencing.getUser().id;
						webConferencing.onUserUpdate(userId, function(update) {
							if (update.providerType == self.getType()) {
								if (update.eventType == "call_state") {
									log.trace("User call state updated: " + JSON.stringify(update));
									// Ignore remote P2P calls start, SDK will fire them via added conversation
									if (update.callState == "started" && update.owner.type != "user") {
										log.trace("Incoming call: " + update.callId);
										var conversation = app.conversationsManager.getConversationByUri(update.callId.substring(2));
										if (conversation) {
											logConversation(conversation);
											var state = conversation.state();
											var needHandle = true;
											var localCall = getLocalCall(update.callId); 
											if (localCall && localCall.state == "started") {
												needHandle = false;
											}
											if (needHandle) {
												// Handle incoming call
												// Created - for restored saved, Conferenced - not happens, but if will, it has the same meaning
												// Disconnected - for previously conferenced on this page
												if (state == "Disconnected" || state == "Conferenced") {
													log.debug("Incoming (saved) existing: " + update.callId);
													handleIncoming(conversation, true);
												} else if (state == "Created") {
													log.debug("Incoming (saved) created: " + update.callId);
													// Created (from above getConversationByUri()) may quickly become Incoming by SDK logic for new group calls,
													// thus let the SDK work and if not yet incoming, proceed with it
													var handle = setTimeout(function() {
														if (!localCall || localCall.state != "started") {
															handleIncoming(conversation, true);
														}
													}, 5000);
													conversation.state.once("Incoming", function() {
														// This listener should work before Incoming handled in added handler above
														// if it become Incoming - cancel the delayed handler, it will be worked in 'added' logic there
														clearTimeout(handle);
													});
												} else if (state == "Incoming") {
													log.debug("User call " + update.callId + " state Incoming skipped, will be handled in added listener");
												} else {
													log.warn("User call " + update.callId + " not active (" + state + ")");
												}
											}
										} else {
											log.warn("User call " + update.callId + " not found in conversation manager");
										}
									} else if (update.callState == "stopped") {
										log.info("Call stopped remotelly: " + update.callId);
										// Hide accept popover for this call
										closeCallPopup(update.callId);
										var localCall = getLocalCall(update.callId);
										if (localCall && canJoin(localCall)) {
											// We leave if something running
											if (localCall.conversation) {
												localCall.conversation.leave().then(function() {
													log.info("Current conversation stopped and leaved " + container.getCallId());
												});
											}
											// And save the call as stopped with firing UI updates
											// Nov 28 2017, code below moved in this if-block from being after it
											// thus it will work only for calls not yet stopped locally
											var callStopped = function(callerRoom) {
												callUpdate({
													state : "stopped",
													callId : update.callId,
													peer : {
														id : update.owner.id,
														type : update.owner.type,
														chatRoom : callerRoom
													},
													saved : true
												});
											};
											if (update.owner.type == "space") {
												// Find room ID for space calls in Chat
												webConferencing.getChat().getRoom(update.owner.id, "space-name").done(function(room) {
													callStopped(room ? room.user : null);
												}).fail(function(err) {
													log.error("Failed to request Chat room: " + update.owner.id + " for: " + update.callId, err);
												});
											} else {
												// we assume: else if (callerType == "chat_room" || callerType == "user") 
												callStopped(update.owner.id);
											}
										}
									}
								} else if (update.eventType == "call_joined") {
									log.info("User call joined: " + update.callId);
									if ($callPopup && $callPopup.callId && $callPopup.callId == update.callId && userId == update.part.id) {
										if ($callPopup.is(":visible")) {
											$callPopup.dialog("close");
										}
									}
								} else if (update.eventType == "call_leaved") {
									// TODO not used
								} else if (update.eventType == "retry") {
									log.trace("<<< Retry for user updates");
								} else {
									log.warn("Unexpected user update: " + JSON.stringify(update));
								}
							} // it's other provider type
						}, function(err) {
							log.error("Failed to listen on user updates", err);
						});
					}).fail(function(err) {
						log.error("Failed to get user group calls", err);
					});
				}).fail(function() {
					if (container) {
						container.init();
						container.$element.append($("<div><div class='pluginError'>Please install Skype web plugin.</div></div>"));
					} // else, user already warned in checkPlugin()
				});
			};
			
			var alignContainerClose = function($container) {
				var pos = $container.position();
				$container.find(".callHoverBar .callClose").position({
					left : pos.left - 17,
					top : pos.top
				});
			};
			
			var alignWindowMiniCallContainer = function($container) {
				var aw = $(document).width();
				var ah = $(document).height();
				var w, h, top, left;
				if (aw > 760) {
					w = 340;
				  h = 180;
				} else if (aw > 360) {
					w = 164;
				  h = 82;
				} else {
					w = 96;
				  h = 48;
				}
				left = (aw/2)-(w/2);
			  top = 0;
				
				$container.height(h);
				$container.width(w);
				$container.offset({
					left: left,
					top: top
				});
				//$container.find("#chatInputContainer").width("100%");
				alignContainerClose($container);
			};
			var alignChatsCallContainer = function($chats, $container) {
				var chatsPos = $chats.offset();
				$container.height($chats.height());
				$container.width($chats.width());
				$container.offset(chatsPos);
				$container.find("#chatInputContainer").width("100%");
				alignContainerClose($container);
			};
			$(window).resize(function() {
			  // align Call container to #chats div dimentions
				var $chats = $("#chat-application #chats");
				var $container = $("#mssfb-call-container");
				var container = $container.data("callcontainer");
				if ($chats.length > 0 && $container.length > 0 && $container.is(":visible")) {
					if (container.isAttached()) {
						alignChatsCallContainer($chats, $container);
					} else {
						alignWindowMiniCallContainer($container);
					}
				}
			});
			
			var showSettingsLogin = function(mssfbId) {
				var process = $.Deferred();
				var $settings = $("div.uiMssfbSettings");
				$settings.remove();
				$settings = $("<div class='uiMssfbSettings' title='" + self.getTitle() + " settings'></div>");
				$(document.body).append($settings);
				$settings.append($("<p><span class='ui-icon messageIcon ui-icon-gear' style='float:left; margin:12px 12px 20px 0;'></span>" +
					"<div class='messageText'>Login in to your " + self.getTitle() + " account.</div></p>"));
				$settings.dialog({
				  resizable: false,
				  height: "auto",
				  width: 400,
				  modal: true,
				  buttons: {
						"Login": function() {
							loginWindow();
							// this method will be available at eXo.webConferencing.mssfb on a call page
							self.loginToken = function(token) {
								var user;
								if (mssfbId) {
									user = webConferencing.getUser();
									var exoUserSFB = webConferencing.imAccount(user, "mssfb");
						  		if (exoUserSFB) {
						  			exoUserSFB.id = mssfbId;
						  		}		        			
								}
								var callback = loginTokenHandler(token, user);
								callback.done(function() {
									process.resolve();
								}); 
								callback.fail(function(err) {
									$settings.find(".messageIcon").removeClass("ui-icon-gear").addClass("ui-icon-alert");
									$settings.find(".messageText").html("Error! " + err);
									$settings.dialog({
										resizable: false,
							      height: "auto",
							      width: 400,
							      modal: true,
							      buttons: {
							        Ok: function() {
							        	$settings.dialog( "close" );
							        }
							      }
							    });
									process.reject(err);
								});
								callback.always(function() {
									$settings.dialog( "close" );									
								});
								return callback.promise();
							};
						},
						"Cancel": function() {
							process.reject();
							$settings.dialog( "close" );
						}
				  }
				});
				$settings.on("dialogclose", function( event, ui ) {
					if (process.state() == "pending") {
						process.reject("closed");
					}
				});
				return process.promise();
			};
			
			var showSettingsLogout = function() {
				var process = $.Deferred();
				var $settings = $("div.uiMssfbSettings");
				$settings.remove();
				$settings = $("<div class='uiMssfbSettings' title='" + self.getTitle() + " settings'></div>");
				$(document.body).append($settings);
				$settings.append($("<p><span class='ui-icon messageIcon ui-icon-gear' style='float:left; margin:12px 12px 20px 0;'></span>" +
					"<div class='messageText'>Forget your " + self.getTitle() 
					+ " access token. A new token can be aquired later by doing login in user profile or by making an outgoing call.</div></p>"));
				$settings.dialog({
					resizable: false,
					height: "auto",
					width: 400,
					modal: true,
					buttons: {
					  "Logout": function() {
					  	removeLocalToken();
					  	process.resolve();
							$settings.dialog( "close" );
					  },
					  "Cancel": function() {
					  	process.reject();
					  	$settings.dialog( "close" );
					  }
					}
				});
				$settings.on("dialogclose", function( event, ui ) {
					if (process.state() == "pending") {
						process.reject("closed");
					}
				});
				return process.promise();
			};
			
			var createLoginButton = function(handler) {
				// Login button
				var $button = $("<div class='actionIcon mssfbLoginButton parentPosition'></div>");
				var $link = $("<a class='mssfbLoginLink'><i class='uiIconColorWarning'></i></a>");
				$button.append($link);
				// Login click handler
				if (handler) {
					$link.click(handler);
				}
				// Popover on the button
				var $popoverContainer = $("<div class='gotPosition' style='position: relative; display:block'></div>");
				$popoverContainer.append("<div class='popover bottom popupOverContent' style='display: none;'>"
							+ "<span class='arrow'></span>"
							+ "<div class='popover-content'>" + self.getTitle() 
								+ " authorization required. Click this icon to open sign-in window.</div>"
							+ "</div>");
				$button.append($popoverContainer);
				var $popover = $popoverContainer.find(".popupOverContent:first");
				$link.mouseenter(function() {
					if (!$popover.is(":visible")) {
						$popover.show("fade", 300);
					}
				});
				$link.mouseleave(function() {
					if ($popover.is(":visible")) {
						$popover.hide("fade", 300);
					}
				});
				setTimeout(function() {
					$popover.show("fade", 1500);
				}, 2000);
				// hide popover in time
				setTimeout(function() {
					$popover.hide("fade", 1500);
				}, 15000);
				$("body").on("click", function() {
					if ($popover.is(":visible")) {
						$popover.hide("fade", 300);
					}
				});
				return $button;
			};
			
			var createLogoutButton = function(handler) {
				var $button = $("<div class='actionIcon mssfbLogoutButton parentPosition'></div>");
				var $link = $("<a class='mssfbLogoutLink' data-placement='bottom' rel='tooltip' title='' data-original-title='Settings'><i class='uiIconSettings uiIconLightGray'></i></a>");
				$button.append($link);
				$link.tooltip();
	    	if (handler) {
	    		$link.click(handler);
	    	}
	    	return $button;
			};
			
			var createCallAccountPopover = function($button) {
				if ($button.find(".gotPosition>.popover").length == 0) {
					var $a = $button.find(".actionIcon>a");
					var $link = $a.length > 0 ? $a : $button;
					// Popover on the button
					var $popoverContainer = $("<div class='gotPosition' style='position: relative; display:block'></div>");
					$popoverContainer.append("<div class='popover bottom popupOverContent' style='display: none;'>"
								+ "<span class='arrow'></span>"
								+ "<div class='popover-content'>This account will be used for " + self.getTitle() + " calls.</div>"
								+ "</div>");
					$button.append($popoverContainer);
					var $popover = $popoverContainer.find(".popupOverContent:first");
					$link.mouseenter(function() {
						if (!$popover.is(":visible")) {
							$popover.show("fade", 300);
						}
					});
					$link.mouseleave(function() {
						if ($popover.is(":visible")) {
							$popover.hide("fade", 300);
						}
					});
					setTimeout(function() {
						$popover.show("fade", 1500);
					}, 2000);
					// hide popover in time
					setTimeout(function() {
						$popover.hide("fade", 1500);
					}, 10000);
					$("body").on("click", function() {
						if ($popover.is(":visible")) {
							$popover.hide("fade", 300);
						}
					});
					return $button;
				}
			};
			
			this.init = function(context) {
				//log.trace("Init at " + location.origin + location.pathname);
				var process = $.Deferred();
				if (window.location.pathname.startsWith("/portal/") && window.location.pathname.indexOf("/edit-profile/") > 0) {
					// in user profile edit page 
					var $form = $("#UIEditUserProfileForm");
					if ($form.length > 0) {
						var $ims = $form.find("#ims");
						var removeToken = false;
						var addControlButton = function($imRow, createFunc) {
							$imRow.find(".mssfbControl").remove();
							var $button = createFunc();
							$button.addClass("mssfbControl");
							$imRow.find(".selectInput+.actionIcon").after($button);
							return $button;
						};
						var controlHandler = function() {
							var $imRow = $(this).parents(".controls-row:first");
							var mssfbId = sipId($imRow.find(".selectInput").val());
							var token = currentToken();
							if (token && token.userId == mssfbId) {
								// User can forget the token
								showSettingsLogout().done(function() {
									// show warn next to SfB IM field
									showLoginButton($imRow);
								});
							} else {
								// User needs login with his (new) SfM account
								showSettingsLogin(mssfbId).done(function() {
									// show settings (logout for now) icon next to SfB IM field
									showLogoutButton($imRow);									
								});
							}
						};
						var showLoginButton = function($imRow) {
							return addControlButton($imRow, function() {
								return createLoginButton(controlHandler);
							});
						};
						var showLogoutButton = function($imRow) {
							var $button = addControlButton($imRow, function() {
								return createLogoutButton(controlHandler);
							});
							if ($ims.find(".mssfbControl:visible").length > 1) {
								createCallAccountPopover($button);
							}
							return $button; 
						};
						var controlInit = function($imRow, mssfbId) {
							var token = currentToken();
							if (token && token.userId == mssfbId) {
								return showLogoutButton($imRow);
							} else {
								return showLoginButton($imRow);
							}
						};
						var imChangeHandler = function($imRow) {
							// FYI $imRow should be a single row, otherwise this method will work wrong!
							if ($imRow.data("mssfbinit")) {
								return true;
							} else {
								$imRow.data("mssfbinit", true);
								var $imType = $imRow.find("select.selectbox");
								var $im = $imRow.find(".selectInput");
								var $control = $imRow.find(".mssfbControl");
								var isMssfbSelected = function() {
									return $imType.find(":selected").val() == "mssfb";
								};
								var isMssfb = isMssfbSelected();
								var initControl = function() {
									if (isMssfb) {
										var im = $im.val();
								    if (EMAIL_PATTERN.test(im)) {
								    	removeToken = false;
								    	// Already looks as an email, we can show login Icon on the right if it is not already there
								    	if ($control.length == 0) {
								    		$control = controlInit($imRow, sipId(im));
								    	} else {
								    		$control.show();
								    	}
								    } else {
								    	// TODO mark this filed invalid
								    	removeToken = true;
								    	$control.hide();
								    }
									} else {
							    	// else, not SfB IM
										removeToken = true;
							    	$control.hide();
							    }
								};
								$imType.change(function() {
									isMssfb = isMssfbSelected();
									initControl();
								});
								$im.on("input", function() { 
									initControl(); 
								});
								if (isMssfb) {
									removeToken = false;
								}
								return isMssfb;
							}
						};
						var initForm = function() {
							var hasMssfb = false;
							$ims.find(".controls-row").each(function() {
								// Add change handler only to first SfB IM
								var $imRow = $(this);
								if (!hasMssfb && imChangeHandler($imRow)) {
									hasMssfb = true;
									var mssfbId = sipId($imRow.find(".selectInput").val());
									controlInit($imRow, mssfbId);
								} else {
									// XXX we don't support multiple SfB for calls - remove settings control
									$imRow.find(".mssfbControl").remove();
								}
							});
						};
						initForm();
						$form.find(".uiAction .btn-save").click(function() {
							if (removeToken) {
								removeLocalToken();
								removeToken = false;
							}							
						});
						// We need also handle added IM rows by later Add New (+) actions 
						setTimeout(function() {
							var MutationObserver = window.MutationObserver || window.WebKitMutationObserver || window.MozMutationObserver;
							var observer = new MutationObserver(function(mutations) {
								// will it fired twice on each update?
								var needUpdate = false;
								nextChange : for (var i=0; i<mutations.length; i++) {
									var m = mutations[i];
									if (m.type == "childList") {
										//var $node = $(m.target);
										//log.trace(">> observed: " + $node.attr("class"));
										for (var i = 0; i < m.addedNodes.length; i++) {
										  var $node = $(m.addedNodes[i]);
										  if ($node.hasClass("controls-row")) {
												needUpdate = true;
												break nextChange;
											}
										}
									}
								} 
								if (needUpdate) {
									initForm();
								}
							});
							observer.observe($ims.get(0), {
								subtree : false,
								childList : true,
								attributes : false,
								characterData : false
							});
						}, 1000);
					} else {
						log.warn("Cannot init IM settings: UIEditUserProfileForm not found");
					}
				} else if (webConferencing.getChat().isApplication()) {
					// we are in the Chat, chatApplication is a global on chat app page
					var $chat = $("#chat-application");
					if ($chat.length > 0) {
						var exoUserSFB = webConferencing.imAccount(context.currentUser, "mssfb");
						if (exoUserSFB) {
							// user has SfB account, we need a call container on Chat page
							var $chats = $chat.find("#chats");
							if ($chats.length > 0) {
								var $container = $("#mssfb-call-container");
								if ($container.length == 0) {
									$container = $("<div id='mssfb-call-container' style='display: none;'></div>");
									var $closeHover = $("<div class='callHoverBar'><a class='actionHover callClose pull-right' href='#' data-placement='bottom' rel='tooltip' data-original-title='Close'>"
											+ "<i class='uiIconClose uiIconLightGray pull-right'></i></a></div>");
									$container.append($closeHover);
									alignContainerClose($container);
									$closeHover.click(function() {
										// leave conversation here
										var container = $container.data("callcontainer");
										if (container && container.getConversation()) {
											// TODO ask user for confirmation of leaving the call
											var c = container.getConversation();
											var cstate = c.state();
											if (cstate == "Connecting" || cstate == "Connected") {
												c.leave().then(function() {
													log.debug("Leaved call: " + getCallId(c));
												}, function(err) {
													log.error("Failed to leave call: " + getCallId(c), err);
												});
											} else if (cstate == "Conferencing" || cstate == "Conferenced") {
												var pcount = c.participantsCount();
												c.leave().then(function() {
													log.debug("Leaved conference call: " + getCallId(c) + " parts: " + pcount + "->" + c.participantsCount());
												}, function(err) {
													log.error("Failed to leave conference call: " + getCallId(c), err);
												});
												// TODO it is a better way to close the call?
												/*c.activeModalities.video.when(true, function() {
											    c.videoService.stop();
												});
												c.activeModalities.audio.when(true, function() {
											    c.audioService.stop();
												});
												c.activeModalities.chat.when(true, function() {
											    c.chatService.stop();
												});*/										
											}
										}
										container.hide();
									});
									var chatsZ = $chats.css("z-index");
									if (typeof chatsZ == "number") {
										chatsZ++;
									} else {
										chatsZ = 99;
									}
									$container.css("z-index", chatsZ);
									$(document.body).append($container);
								} else {
									$container.hide();
								}
								var container = new CallContainer($container);
								container.resized = false;
								container.onShow(function() {
									if (!container.resized) {
										container.resized = true;
										alignChatsCallContainer($chats, $container);										
									}
									$container.show();
									setTimeout(function() {
										$(window).resize();
									}, 1500);
								});
								container.onHide(function() {
									$container.hide();
								});
								$container.data("callcontainer", container);
								// try login in SfB SDK using saved token
								var initializer = $.Deferred();
								var token = currentToken();
								if (token) {
									// we reuse saved token on the Chat page
									var callback = loginTokenHandler(token);
									callback.done(function(api, app) {
										initializer.resolve(api, app);
									});
									callback.fail(function(err) {
										initializer.reject(err);
									});
								} else {
									// we need try SfB login window in hidden iframe (if user already logged in AD, then it will work)
								  loginIframe().done(function(api, app) {
								  	initializer.resolve(api, app);
								  }).fail(function(err) {
								  	initializer.reject(err);
								  });
								}
								var $users = $chat.find("#chat-users");
								var joinedCall = null;
								var moveCallContainer = function(forceDetach) {
									// TODO is ti possible to get rid the timer and rely on actual DOM update for the chat?
									setTimeout(function() {
										if ($container.is(":visible")) {
											var roomId = chatApplication.targetUser;
											if (roomId && joinedCall) {
												var roomTitle = chatApplication.targetFullname;
												if (container.isAttached()) {
													if (forceDetach || roomId != joinedCall.peer.chatRoom) {
														// move call container to the window top-center with smaller size
														//log.trace(">>> room not of the call < " + roomTitle);
														alignWindowMiniCallContainer($container);
														container.detached();
													} // else, do nothing
												} else {
													if (roomId == joinedCall.peer.chatRoom) {
														// move call container to the current room chats size
														//log.trace(">>> room of the call > " + roomTitle);
														alignChatsCallContainer($chats, $container);
														container.attached();
													} // else, do nothing
												}
											}
										}
									}, 1200);
								};
								// If user later will click another room during the call, we move the call container to a window top
								// if user will get back to the active call room, we will resize the container to that call chats
								$users.click(function() {
									moveCallContainer(false);
								});
								$chat.find(".uiRightContainerArea #room-detail>#back").click(function() {
									moveCallContainer(true);
								});
								// Init update procedure common for the whole provider
								var activeRooms = {};
								// Mark timeout should be longer of used one below in callUpdate() 'accepted' 
								var markRoom = function(id, markFunc) {
									if (id) {
										activeRooms[id] = markFunc;
										// XXX Do twice: sooner and after possible refreshes caused by callUpdate() 'accepted' 
										setTimeout(markFunc, 750);
										setTimeout(markFunc, 2750);										
									}
								};
								var unmarkRoom = function(roomId) {
									if (roomId) {
										delete activeRooms[roomId];
										setTimeout(function() {
											var $room = $users.find("#users-online-" + roomId);
											$room.removeClass("activeCall");
											$room.removeClass("incomingCall");
										}, 750);										
									}
								};
								var markRoomActive = function(roomId) {
									markRoom(roomId, function() {
										var $room = $users.find("#users-online-" + roomId);
										if (!$room.hasClass("activeCall")) {
											$room.addClass("activeCall");
										}
										$room.removeClass("startingCall");
									});
								};
								var markRoomIncoming = function(roomId) {
									markRoom(roomId, function() {
										// mark Chat room icon red blinking (aka incoming)
										var $room = $users.find("#users-online-" + roomId);
										if (!$room.hasClass("activeCall")) {
											$room.addClass("activeCall");
										}
										if (!$room.hasClass("startingCall")) {
											$room.addClass("startingCall");
										};
									});
								};
								setTimeout(function() {
									var MutationObserver = window.MutationObserver || window.WebKitMutationObserver || window.MozMutationObserver;
									var observer = new MutationObserver(function(mutations) {
										// will it fired twice on each update?
										for (var id in activeRooms) {
											if (activeRooms.hasOwnProperty(id)) {
												var roomFunc = activeRooms[id];
												markRoom(id, roomFunc);
											}
										}
									});
									observer.observe($users.get(0), {
										subtree : true,
										childList : true,
										attributes : false,
										characterData : false
									});
								}, 2500);
								hasJoinedCall = function(callRef) {
									return joinedCall ? callRef == joinedCall.peer.id || callRef == joinedCall.peer.chatRoom : false;
								};
								callUpdate = function(info) {
									log.trace(">>> callUpdate: [" + info.state + "] " + (info.saved ? "[saved]" : "") + " " + info.callId + " peer: " + JSON.stringify(info.peer));
									if (info.state) {
										// 'started' and 'stopped' can be remote (saved) events for group calls, 
										// when an one initiated and when last part leaved respectively 
										// 'incoming' only local, it is first for an incoming call 
										// 'accepted' only local, it is second event for incoming when user accepts it, otherwise 'canceled' will be next
										// 'started' is a first event for P2P outgoing and a next for P2P incoming
										// 'joined' is a first event for group outgoing and a next for group incoming
										// 'leaved' when user leaves a group call (it still may run)
										// 'stopped' for finished call, locally (P2P or group) or remotely (group only) initiated
										// 'canceled' when user declined the call or timeout happened waiting the dialog
										var isGroup = info.callId.startsWith("g/");
										var $callButton = $chat.find("#room-detail .mssfbCallAction");
										if (info.state == "incoming") {
											// Do need change Chat UI for this event?
											markRoomIncoming(info.peer.chatRoom);
										} else if (info.state == "accepted") {
											saveLocalCall(info);
											// if call container was running a convo, it is leaving right here (see accepted logic)
											joinedCall = info; 
											if (info.peer.chatRoom != chatApplication.targetUser) {
												// switch to the call room in the Chat
												var selector = ".users-online .room-link[user-data='" + info.peer.chatRoom + "']";
												var $rooms = $users.find(selector);
												if ($rooms.length == 0) {
													// try find in the history/offline
													$chat.find(".btn-history:not(.active)").click();
													setTimeout(function() {
														$chat.find(".btn-offline:not(.active)").click();
														setTimeout(function() {
															$users.find(selector).first().click();
														}, 1000);
													}, 1000);
													//$chat.find(".btn-history, .btn-offline").filter(":not(.active)").click();
													// TODO do we want hide opened offline/history rooms?
												} else {
													$rooms.first().click();
												}												
											}
											markRoomActive(info.peer.chatRoom);
										} else if (info.state == "started") {
											saveLocalCall(info);
											if (chatApplication.targetUser == info.peer.chatRoom) {
												// this call room is current open in Chat
												if (info.conversation) { // !info.saved && - for repeated it will be true
													// it's started outgoing call
													joinedCall = info;
													//if ($callButton.data("callid") == info.callId) {
													if (!$callButton.hasClass("callDisabled")) {
														$callButton.addClass("callDisabled");
													}
												} else if (isGroup) {
													$callButton.find(".callTitle").text(self.getJoinTitle());
												}
											}
											// set user status: do-not-disturb, remember current status
											/*chatNotification.getStatus(chatApplication.username, function(status) {
												chatApplication.setStatusDoNotDisturb();
												userStatus = status;
											});*/
											markRoomActive(info.peer.chatRoom);
										} else if (info.state == "stopped") {
											saveLocalCall(info);
											if (joinedCall && joinedCall.callId == info.callId) {
												joinedCall = null;
											}
											if (chatApplication.targetUser == info.peer.chatRoom) {
												if (isGroup) {
													// this (group) call room is current open in Chat
													$callButton.find(".callTitle").text(self.getCallTitle());
												}
												$callButton.removeClass("callDisabled");
											}
											/*if (userStatus) {
												// set user status to a previous one
												chatApplication.setStatus(userStatus);
												userStatus = null;
											}*/
											unmarkRoom(info.peer.chatRoom);
										} else if (info.state == "joined") {
											saveLocalCall(info);
											joinedCall = info;
											if (chatApplication.targetUser == info.peer.chatRoom) {
												if (!$callButton.hasClass("callDisabled")) {
													$callButton.addClass("callDisabled");
												}
											}
										} else if (info.state == "leaved") {
											saveLocalCall(info);
											var hasLeaved = false;
											if (joinedCall && joinedCall.callId == info.callId) {
												joinedCall = null;
												hasLeaved = true;
											}
											if (chatApplication.targetUser == info.peer.chatRoom) {
												$callButton.removeClass("callDisabled");
												if (isGroup && hasLeaved) {
													$callButton.find(".callTitle").text(self.getJoinTitle());
												}
											}
										} else if (info.state == "canceled") {
											saveLocalCall(info);
											if (chatApplication.targetUser == info.peer.chatRoom) {
												if (isGroup) {
													$callButton.find(".callTitle").text(self.getJoinTitle());													
												} // else, for P2P we don't do anything
											}
											markRoomActive(info.peer.chatRoom);
										} else {
											log.warn("Unexpected call update status: " + JSON.stringify(info));
										}
									} else if (info.error) {
										log.error("Failed to update call: " + info.error);
									}
								};
								initializer.done(function(api, app) {
									log.debug("Automatic login done.");
									incomingCallHandler(api, app, container);
								});
								initializer.fail(function(err) {
									// need login user explicitly (in a popup)
									log.warn("Automatic login failed", err);
									var $roomDetail = $chat.find("#room-detail");
									var userLogin = function() {
										loginWindow();
										self.loginToken = function(token) {
											var callback = loginTokenHandler(token);
											callback.done(function(api, app) {
												log.info("User login done.");
												incomingCallHandler(api, app, container);
											});
											callback.fail(function(err) {
												log.showError("User login failed", err, self.getTitle() + " error", "Unable sign in your " + self.getTitle() 
															+ " account. " + webConferencing.errorText(err));
											});
											return callback.promise();
										}
									};
									// show warn icon near the Call Button, icon with click and details popover
									var MAX_WAIT = 24; // 6sec = 24 * 250ms
									var attemtCount = 0;
									var addLoginWarn = function(wait) {
										var $wrapper = $roomDetail.find(".callButtonContainerWrapper");
										if ($wrapper.length == 0 && wait) {
											if (attemtCount < MAX_WAIT) {
												// wait a bit and try again
												attemtCount++;
												setTimeout(function() {
													addLoginWarn(attemtCount < MAX_WAIT);
												}, 250);										
											} else {
												// TODO this code never works (see wait flag above)
												var loginLinkId = "SfB_login_" + Math.floor((Math.random() * 1000) + 1);
												webConferencing.showWarn("Authorize in " + self.getTitle(), "Please <a id='" + loginLinkId + "' href='#' class='pnotifyTextLink'>authorize</a> in your " +
														self.getTitle() + " account to make you available.", function(pnotify) {
													$(pnotify.container).find("#" + loginLinkId).click(function() {
														userLogin();
													});
												});
											}
										} else if (!$wrapper.data("callloginwarned")) {
											$wrapper.data("callloginwarned", true);
											// add the link/icon with warning and login popup
											// this warning will be removed by loginTokenHandler() on done
											$chat.find(".mssfbLoginButton").remove();
											var $button = createLoginButton(userLogin);
											if ($roomDetail.is(":visible")) {
												// Show in room info near the call button or team dropdown
												$button.addClass("pull-right");
												if ($wrapper.length > 0) {
												 $wrapper.after($button);
												} else {
												 $roomDetail.find("#chat-team-button-dropdown").after($button);
												}	
											} else {
												// Show in .selectUserStatus after the user name
												$button.addClass("pull-left");
												$chat.find(".no-user-selection .selectUserStatus+label:first").after($button);												
											}
										}
									};
									addLoginWarn(true);
								});
							} else {
								log.warn("Cannot init Chat for calls container: #chats element not found");
							} 
						} // else, current user has not SfB - nothing to initialize
					} else {
						log.warn("Chat application element not found.");
					}
				} else if (false && window.location.pathname.startsWith("/portal/")) {
					// TODO temporarily not used logic
					// we somewhere in the portal, try login using saved token
					if (typeof Storage !== "undefined") {
						var savedToken = localStorage.getItem("mssfb_login_token");
						if (savedToken) {
							try {
								var token = JSON.parse(savedToken);
								var appInitializer = self.application(loginUri, function(resource) {
									return token.token_type + " " + token.access_token;
								});
								appInitializer.done(function(api, app) {
									// TODO re-save token?
								  log.info("Login OK (saved), app created OK, token: " + location.hash);
								});
								appInitializer.fail(function(err) {
									log.error("Login (saved) error", err);
									localStorage.removeItem(TOKEN_STORE);
								});
							} catch(e) {
								log.error("Login (saved) parsing error: " + e, e);
							}
						} else {
							log.warn("Login (saved) not possible: access token not found in local storage.");
						}
					} else {
					  // Sorry! No Web Storage support..
						log.warn("Cannot read access token: local storage not supported.");
					}
				} // else, it's also may be a call page - do we need something here?
				process.resolve();
				return process.promise();
			};
		}

		var provider = new SfBProvider();

		// Add SfB provider into webConferencing object of global eXo namespace (for non AMD uses)
		if (globalWebConferencing) {
			globalWebConferencing.mssfb = provider;
		} else {
			log.warn("eXo.webConferencing not defined");
		}
		
		$(function() {
			try {
				// XXX workaround to load CSS until gatein-resources.xml's portlet-skin will be able to load after the Enterprise skin
				webConferencing.loadStyle("/skype/skin/mssfb.css");
				webConferencing.loadStyle("/skype/skin/mssfb-call.css");
				if (eXo.env && eXo.env.client && eXo.env.client.skin && eXo.env.client.skin == "Enterprise") {
					webConferencing.loadStyle("/skype/skin/skype-mssfb-enterprise.css");	
				}
			} catch(e) {
				log.error("Error loading styles (for SfB).", e);
			}
		});

		log.trace("< Loaded at " + location.origin + location.pathname);
		
		return provider;
	} else {
		window.console && window.console.log("WARN: webConferencing not given and eXo.webConferencing not defined. Skype provider registration skipped.");
	}
})($, typeof webConferencing != "undefined" ? webConferencing : null );
