/*
 * Decompiled with CFR 0.152.
 */
package org.exoplatform.addon.ethereum.wallet.service;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.net.URI;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.addon.ethereum.wallet.model.GlobalSettings;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.web3j.protocol.Web3j;
import org.web3j.protocol.Web3jService;
import org.web3j.protocol.core.methods.response.EthBlock;
import org.web3j.protocol.core.methods.response.EthGetTransactionReceipt;
import org.web3j.protocol.core.methods.response.EthTransaction;
import org.web3j.protocol.core.methods.response.Transaction;
import org.web3j.protocol.core.methods.response.TransactionReceipt;
import org.web3j.protocol.websocket.WebSocketClient;
import org.web3j.protocol.websocket.WebSocketListener;
import org.web3j.protocol.websocket.WebSocketService;

public class EthereumClientConnector {
    private static final String ERROR_CLOSING_WEB_SOCKET_MESSAGE = "Error closing web socket";
    private static final Log LOG = ExoLogger.getLogger(EthereumClientConnector.class);
    private Web3j web3j = null;
    private WebSocketClient webSocketClient = null;
    private WebSocketService web3jService = null;
    private GlobalSettings globalSettings = null;
    private ScheduledExecutorService connectionVerifierExecutor = null;
    private int connectionInterruptionCount = -1;
    private boolean connectionInProgress = false;
    private boolean serviceStopping = false;

    public EthereumClientConnector() {
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("Ethereum-websocket-connector-%d").build();
        this.connectionVerifierExecutor = Executors.newSingleThreadScheduledExecutor(namedThreadFactory);
    }

    public void start(GlobalSettings storedSettings) {
        this.globalSettings = storedSettings;
        this.connectionVerifierExecutor.scheduleWithFixedDelay(() -> {
            try {
                if (!this.initWeb3Connection()) {
                    return;
                }
            }
            catch (Throwable e) {
                LOG.warn("Error while checking connection status to Etherreum Websocket endpoint: {}", new Object[]{e.getMessage()});
                this.closeConnection();
                return;
            }
        }, 5L, 10L, TimeUnit.SECONDS);
    }

    public void stop() {
        this.serviceStopping = true;
        this.connectionVerifierExecutor.shutdown();
        this.closeConnection();
    }

    public Transaction getTransaction(String transactionHash) throws InterruptedException {
        EthTransaction ethTransaction;
        this.waitConnection();
        try {
            ethTransaction = (EthTransaction)this.web3j.ethGetTransactionByHash(transactionHash).send();
        }
        catch (IOException e) {
            LOG.info("Connection interrupted while getting Transaction '{}' information. Reattempt until getting it. Reason: {}", new Object[]{transactionHash, e.getMessage()});
            return this.getTransaction(transactionHash);
        }
        if (ethTransaction != null) {
            return (Transaction)ethTransaction.getResult();
        }
        return null;
    }

    public EthBlock.Block getBlock(String blockHash) throws InterruptedException {
        EthBlock ethBlock;
        this.waitConnection();
        try {
            ethBlock = (EthBlock)this.web3j.ethGetBlockByHash(blockHash, false).send();
        }
        catch (IOException e) {
            LOG.info("Connection interrupted while getting Block '{}' information. Reattempt until getting it. Reason: {}", new Object[]{blockHash, e.getMessage()});
            return this.getBlock(blockHash);
        }
        if (ethBlock != null && ethBlock.getResult() != null) {
            return (EthBlock.Block)ethBlock.getResult();
        }
        return null;
    }

    public TransactionReceipt getTransactionReceipt(String transactionHash) throws InterruptedException {
        EthGetTransactionReceipt ethGetTransactionReceipt;
        this.waitConnection();
        try {
            ethGetTransactionReceipt = (EthGetTransactionReceipt)this.web3j.ethGetTransactionReceipt(transactionHash).send();
        }
        catch (IOException e) {
            LOG.info("Connection interrupted while getting Transaction receipt '{}' information. Reattempt until getting it. Reason: {}", new Object[]{transactionHash, e.getMessage()});
            return this.getTransactionReceipt(transactionHash);
        }
        if (ethGetTransactionReceipt != null) {
            return (TransactionReceipt)ethGetTransactionReceipt.getResult();
        }
        return null;
    }

    public void changeSettings(GlobalSettings newGlobalSettings) {
        if (newGlobalSettings == null) {
            throw new IllegalArgumentException("GlobalSettings argument is mandatory");
        }
        if (this.globalSettings == null) {
            return;
        }
        GlobalSettings oldGlobalSettings = this.globalSettings;
        this.globalSettings = newGlobalSettings;
        if (StringUtils.isBlank((CharSequence)newGlobalSettings.getWebsocketProviderURL()) || !StringUtils.equals((CharSequence)newGlobalSettings.getWebsocketProviderURL(), (CharSequence)oldGlobalSettings.getWebsocketProviderURL())) {
            this.closeConnection();
        }
    }

    public boolean isConnected() {
        return this.web3j != null && this.web3jService != null && this.webSocketClient != null && this.webSocketClient.isOpen();
    }

    public int getConnectionInterruptionCount() {
        return this.connectionInterruptionCount;
    }

    private String getWebsocketProviderURL() {
        return this.globalSettings == null ? null : this.globalSettings.getWebsocketProviderURL();
    }

    private void waitConnection() throws InterruptedException {
        if (this.serviceStopping) {
            throw new IllegalStateException("Server is stopping, thus not Web3 request should be emitted");
        }
        while (!this.isConnected()) {
            LOG.info((Object)"Wait until Websocket connection to blockchain is established to retrieve information");
            Thread.sleep(5000L);
        }
    }

    private void closeConnection() {
        if (this.web3j != null) {
            block8: {
                try {
                    this.web3j.shutdown();
                }
                catch (Throwable e) {
                    LOG.warn("Error closing old web3j connection: {}", new Object[]{e.getMessage()});
                    if (this.web3jService != null && this.webSocketClient != null && this.webSocketClient.isOpen()) {
                        try {
                            this.web3jService.close();
                        }
                        catch (Throwable e1) {
                            LOG.warn("Error closing old websocket connection: {}", new Object[]{e1.getMessage()});
                        }
                    }
                    if (this.webSocketClient == null || !this.webSocketClient.isOpen()) break block8;
                    try {
                        this.webSocketClient.close();
                    }
                    catch (Throwable e1) {
                        LOG.warn("Error closing old websocket connection: {}", new Object[]{e1.getMessage()});
                    }
                }
            }
            this.web3j = null;
            this.web3jService = null;
            this.webSocketClient = null;
        }
    }

    private boolean initWeb3Connection() {
        if (this.connectionInProgress) {
            LOG.info((Object)"Web3 connection initialization in progress, skip transaction processing until it's initialized");
            return false;
        }
        if (this.serviceStopping) {
            LOG.info((Object)"Stopping server, thus no new connection is attempted again");
            return false;
        }
        String websocketProviderURL = this.getWebsocketProviderURL();
        if (StringUtils.isBlank((CharSequence)websocketProviderURL)) {
            LOG.info((Object)"No configured URL for Ethereum Websocket connection");
            this.closeConnection();
            return false;
        }
        if (!websocketProviderURL.startsWith("ws:") && !websocketProviderURL.startsWith("wss:")) {
            LOG.warn((Object)("Bad format for configured URL " + websocketProviderURL + " for Ethereum Websocket connection"));
            this.closeConnection();
            return false;
        }
        if (this.isConnected()) {
            return false;
        }
        this.connectionInProgress = true;
        try {
            ++this.connectionInterruptionCount;
            this.closeConnection();
            LOG.info("Connecting to Ethereum network endpoint {}", new Object[]{this.getWebsocketProviderURL()});
            this.web3jService = this.testConnection();
            this.web3j = Web3j.build((Web3jService)this.web3jService);
            LOG.info("Connection established to Ethereum network endpoint {}", new Object[]{this.getWebsocketProviderURL()});
        }
        finally {
            this.connectionInProgress = false;
        }
        return true;
    }

    private WebSocketService testConnection() {
        try {
            this.webSocketClient = new WebSocketClient(new URI(this.getWebsocketProviderURL()));
            this.webSocketClient.setConnectionLostTimeout(10);
            this.web3jService = new WebSocketService(this.webSocketClient, true);
            this.webSocketClient.setListener(new WebSocketListener(){

                public void onMessage(String message) throws IOException {
                    LOG.debug((Object)"A new message is received in testConnection method");
                }

                public void onError(Exception e) {
                    LOG.warn((Object)EthereumClientConnector.this.getConnectionFailedMessage());
                }

                public void onClose() {
                    LOG.debug((Object)"Websocket connection closed for testConnection method");
                }
            });
            this.web3jService.connect();
            Thread.sleep(10000L);
            if (!this.webSocketClient.isOpen()) {
                this.closeConnectionAndThrowError(null);
            }
        }
        catch (Throwable e) {
            this.closeConnectionAndThrowError(e);
        }
        return this.web3jService;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeConnectionAndThrowError(Throwable e) {
        if (this.web3j != null) {
            try {
                this.web3j.shutdown();
            }
            catch (Exception e1) {
                LOG.debug((Object)ERROR_CLOSING_WEB_SOCKET_MESSAGE, (Throwable)e1);
            }
            finally {
                this.web3j = null;
                this.webSocketClient = null;
                this.web3jService = null;
            }
        }
        if (this.web3jService != null) {
            try {
                this.web3jService.close();
            }
            catch (Exception e1) {
                LOG.debug((Object)ERROR_CLOSING_WEB_SOCKET_MESSAGE, (Throwable)e1);
            }
            finally {
                this.web3jService = null;
                this.webSocketClient = null;
            }
        }
        if (this.webSocketClient != null && this.webSocketClient.isOpen()) {
            try {
                this.webSocketClient.close();
            }
            catch (Exception e1) {
                LOG.debug((Object)ERROR_CLOSING_WEB_SOCKET_MESSAGE, (Throwable)e1);
            }
            finally {
                this.web3j = null;
                this.webSocketClient = null;
                this.web3jService = null;
            }
        }
        if (e == null) {
            throw new IllegalStateException(this.getConnectionFailedMessage());
        }
        throw new IllegalStateException(this.getConnectionFailedMessage(), e);
    }

    private String getConnectionFailedMessage() {
        return "Connection failed to " + this.getWebsocketProviderURL();
    }
}

