/*
 * Copyright (C) 2022 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.connector;

import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.exoplatform.commons.api.notification.NotificationContext;
import org.exoplatform.commons.api.notification.model.ArgumentLiteral;
import org.exoplatform.commons.api.notification.model.PluginKey;
import org.exoplatform.commons.notification.impl.NotificationContextImpl;
import org.exoplatform.container.component.BaseComponentPlugin;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.PropertiesParam;

/**
 * Is extended by all MalwareDetectionItem connectors, and allows to build configuration needed by a list of connectors that is used for Malware detection.
 * 
 */
public abstract class MalwareDetectionItemConnector extends BaseComponentPlugin {
  
  public static final String INFECTED_ITEM_NAME = "infectedItemName";
  public static final ArgumentLiteral<String> INFECTED_ITEM_NAME_ARGUMENT = new ArgumentLiteral<String>(String.class, INFECTED_ITEM_NAME);
  public static final String INFECTED_ITEM_LAST_MODIFIER = "infectedItemLastModifier";
  public static final ArgumentLiteral<String> INFECTED_ITEM_LAST_MODIFIER_ARGUMENT = new ArgumentLiteral<String>(String.class, INFECTED_ITEM_LAST_MODIFIER);
  public static final String INFECTED_ITEM_ID = "infectedItemId";
  public static final String INFECTED_ITEM_MODIFICATION_DATE = "infectedItemModificationDate";
  
  private String type;
  private boolean enable = true;
  
  protected static final String MALWARE_DETECTION_FEATURE = "malware-detection";
  private static final String MALWARE_DETECTION_PLUGIN_ID = "MalwareDetectionPlugin";
  private static final String SLASH_SEPARATOR = "/";
  private static final String BACK_SLASH_SEPARATOR = "\\";
  
  
  /**
   * Initializes a malware detection item connector. The constructor is default that connectors must implement.
   * @param initParams The parameters which are used for initializing the malware detection item connector from configuration.
   * @LevelAPI Experimental
   */
  public MalwareDetectionItemConnector(InitParams initParams) {
    PropertiesParam param = initParams.getPropertiesParam("constructor.params");
    this.type = param.getProperty("type");
    if (StringUtils.isNotBlank(param.getProperty("enable"))) {
      this.setEnable(Boolean.parseBoolean(param.getProperty("enable")));
    }
  }
  
  /**
   * @return the type
   */
  public String getType() {
    return type;
  }

  /**
   * @param type the type to set
   */
  public void setType(String type) {
    this.type = type;
  }

  /**
   * @return the enable
   */
  public boolean isEnable() {
    return enable;
  }

  /**
   * @param enable the enable to set
   */
  public void setEnable(boolean enable) {
    this.enable = enable;
  }
  
  public void processInfectedItem(String infectedItemPath) {
    for (Map<String, String> infectedItem : getInfectedItems(infectedItemPath)) {
      sendInfectedItemNotification(infectedItem);
      cleanInfectedItem(infectedItem);
    }
  }
  
  public abstract boolean canProcessInfectedItem(String infectedItemPath);
  
  public abstract List<Map<String, String>> getInfectedItems(String infectedItemPath);
  
  public abstract void cleanInfectedItem(Map<String, String> infectedItem);
  
  protected static String getPathSeparator(String infectedFilePath) {
    String separator = "";
    if (infectedFilePath.contains(SLASH_SEPARATOR)) {
      separator = SLASH_SEPARATOR;
    }
    else if (infectedFilePath.contains(BACK_SLASH_SEPARATOR)) {
      separator = BACK_SLASH_SEPARATOR;
    }
    return separator;
  }
  
  private void sendInfectedItemNotification(Map<String, String> infectedItem) {
    NotificationContext ctx = NotificationContextImpl.cloneInstance();
    ctx.append(INFECTED_ITEM_NAME_ARGUMENT, infectedItem.get(INFECTED_ITEM_NAME));
    ctx.append(INFECTED_ITEM_LAST_MODIFIER_ARGUMENT, infectedItem.get(INFECTED_ITEM_LAST_MODIFIER));
    ctx.getNotificationExecutor().with(ctx.makeCommand(PluginKey.key(MALWARE_DETECTION_PLUGIN_ID))).execute(ctx);
  }
}