/*
 * Copyright (C) 2025 eXo Platform SAS
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see <gnu.org/licenses>.
 */

package org.exoplatform.antimalware.listener;

import jakarta.annotation.PostConstruct;
import org.exoplatform.commons.api.notification.NotificationContext;
import org.exoplatform.commons.api.notification.model.ArgumentLiteral;
import org.exoplatform.commons.api.notification.model.PluginKey;
import org.exoplatform.commons.notification.impl.NotificationContextImpl;
import org.exoplatform.services.listener.Asynchronous;
import org.exoplatform.services.listener.Event;
import org.exoplatform.services.listener.Listener;
import org.exoplatform.services.listener.ListenerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Map;

@Asynchronous
@Component
public class AntiMalwareNotificationListener extends Listener<String, Object> {

  @Autowired
  private ListenerService                      listenerService;

  private static final String                  MALWARE_DETECTION_PLUGIN_ID          = "MalwareDetectionPlugin";

  private static final String                  INFECTED_ITEM_NAME                   = "infectedItemName";

  private static final String                  INFECTED_ITEM_LAST_MODIFIER          = "infectedItemLastModifier";

  private static final String                  INFECTED_FILE_CLEANED                = "infected-file-cleaned";

  private static final ArgumentLiteral<String> INFECTED_ITEM_NAME_ARGUMENT          = new ArgumentLiteral<>(String.class,
                                                                                                            INFECTED_ITEM_NAME);

  private static final ArgumentLiteral<String> INFECTED_ITEM_LAST_MODIFIER_ARGUMENT =
                                                                                    new ArgumentLiteral<>(String.class,
                                                                                                          INFECTED_ITEM_LAST_MODIFIER);

  private final String[]                       eventTypes                           = { INFECTED_FILE_CLEANED };

  @PostConstruct
  public void init() {
    for (String eventType : eventTypes) {
      listenerService.addListener(eventType, this);
    }
  }

  @Override
  public void onEvent(Event<String, Object> event) throws Exception {
    Map<String, String> infectedItem = (Map<String, String>) event.getData();
    if (infectedItem == null) {
      return;
    }
    sendInfectedItemNotification(infectedItem);
  }

  private void sendInfectedItemNotification(Map<String, String> infectedItem) {
    NotificationContext ctx = NotificationContextImpl.cloneInstance();
    ctx.append(INFECTED_ITEM_NAME_ARGUMENT, infectedItem.get(INFECTED_ITEM_NAME));
    ctx.append(INFECTED_ITEM_LAST_MODIFIER_ARGUMENT, infectedItem.get(INFECTED_ITEM_LAST_MODIFIER));
    ctx.getNotificationExecutor().with(ctx.makeCommand(PluginKey.key(MALWARE_DETECTION_PLUGIN_ID))).execute(ctx);
  }
}
