/*
 * Copyright (C) 2020 eXo Platform SAS.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.exoplatform.antimalware.notification.plugin;

import org.exoplatform.antimalware.connector.MalwareDetectionItemConnector;
import org.exoplatform.commons.api.notification.NotificationContext;
import org.exoplatform.commons.api.notification.model.NotificationInfo;
import org.exoplatform.commons.api.notification.plugin.BaseNotificationPlugin;
import org.exoplatform.commons.utils.ListAccess;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.User;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

public class MalwareDetectionPlugin extends BaseNotificationPlugin {

  private static final Log          LOG          = ExoLogger.getLogger(MalwareDetectionPlugin.class);

  private final OrganizationService organizationService;

  public static final String        ID           = "MalwareDetectionPlugin";

  private static final String       DLP_GROUP    = "/platform/dlp";

  private static final String       ADMINS_GROUP = "/platform/administrators";

  public MalwareDetectionPlugin(InitParams initParams, OrganizationService organizationService) {
    super(initParams);
    this.organizationService = organizationService;
  }

  @Override
  public String getId() {
    return ID;
  }

  @Override
  public NotificationInfo makeNotification(NotificationContext ctx) {
    String infectedItemName = ctx.value(MalwareDetectionItemConnector.INFECTED_ITEM_NAME_ARGUMENT);
    String infectedItemLastModifier = ctx.value(MalwareDetectionItemConnector.INFECTED_ITEM_LAST_MODIFIER_ARGUMENT);
    NotificationInfo notificationInfo = NotificationInfo.instance().key(getId());
    notificationInfo.with(MalwareDetectionItemConnector.INFECTED_ITEM_NAME, infectedItemName);
    notificationInfo.to(getListReceivers(infectedItemLastModifier));
    return notificationInfo;
  }

  @Override
  public boolean isValid(NotificationContext ctx) {
    return true;
  }

  private List<String> getListReceivers(String owner) {
    try {
      ListAccess<User> usersList = organizationService.getUserHandler().findUsersByGroupId(DLP_GROUP);
      ListAccess<User> adminsList = organizationService.getUserHandler().findUsersByGroupId(ADMINS_GROUP);

      List<User> users = Arrays.asList(usersList.load(0, usersList.getSize()));
      List<User> admins = Arrays.asList(adminsList.load(0, adminsList.getSize()));

      return Stream.concat(Stream.concat(users.stream().map(User::getUserName), admins.stream().map(User::getUserName)),
                           Stream.of(owner))
                   .distinct()
                   .toList();
    } catch (Exception e) {
      LOG.warn("Failed to list receivers for MalwareDetection notification plugin", e);
      return List.of(owner);
    }
  }
}
