/*
 * Copyright (C) 2025 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.connector;

import jakarta.annotation.PostConstruct;
import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.exoplatform.services.listener.ListenerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Component
public class ExternalFileSystemItemConnector extends MalwareDetectionItemConnector {

  @Autowired
  private MalwareDetectionService malwareDetectionService;

  @Autowired
  private ListenerService         listenerService;

  private static final String     FILES = "files";

  private static final String     JCR   = "jcr";

  private static final String     TYPE  = "externalFileSystem";

  @PostConstruct
  public void init() {
    setType(TYPE);
    setEnable(true);
    this.malwareDetectionService.addMalwareDetectionItemConnector(this);
  }

  @Override
  public boolean canProcessInfectedItem(String infectedItemPath) {
    return !infectedItemPath.contains(File.separator + FILES + File.separator)
        && !infectedItemPath.contains(File.separator + JCR + File.separator);
  }

  @Override
  public List<Map<String, String>> getInfectedItems(String infectedItemPath) {
    Map<String, String> item = new HashMap<>();
    item.put(INFECTED_ITEM_PATH, infectedItemPath);
    item.put(TYPE, getType());
    return List.of(item);
  }

  @Override
  public void cleanInfectedItem(Map<String, String> infectedItem) {
    listenerService.broadcast("clean-file-infected", getType(), infectedItem);
  }

  @Override
  public void processInfectedItem(String infectedItemPath) {
    for (Map<String, String> infectedItem : getInfectedItems(infectedItemPath)) {
      cleanInfectedItem(infectedItem);
    }
  }
}
