/*
 * Copyright (C) 2022 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.connector;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.exoplatform.commons.file.model.FileItem;
import org.exoplatform.commons.file.services.FileService;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.listener.ListenerService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.manager.IdentityManager;

public class MalwareDetectionFilesConnector extends MalwareDetectionItemConnector {

  private static final Log    LOGGER                          = ExoLogger.getExoLogger(MalwareDetectionFilesConnector.class);

  private FileService         fileService;

  private IdentityManager     identityManager;

  private static final String FILES                           = "files";

  private static final String MALWARE_INFECTED_FILE_DETECTION = "malware-infected-file-detection";

  private static final String MALWARE_INFECTED_FILE_CLEAN     = "malware-infected-file-clean";
  
  public MalwareDetectionFilesConnector(InitParams initParams,
                                        FileService fileService,
                                        IdentityManager identityManager,
                                        ListenerService listenerService) {
    super(initParams, listenerService);
    this.fileService = fileService;
    this.identityManager = identityManager;
  }
  
  @Override
  public List<Map<String, String>> getInfectedItems(String infectedFilePath) {
    String fileChecksum = getInfectedFileChecksum(infectedFilePath);
    List<Map<String, String>> infectedFiles = new ArrayList<Map<String, String>>();
    long startTime = System.currentTimeMillis();
    try {
      List<FileItem> infectedFileItems = fileService.getFilesByChecksum(fileChecksum);
      for (FileItem infectedFileItem : infectedFileItems) {
        if (infectedFileItem != null && infectedFileItem.getFileInfo() != null && !infectedFileItem.getFileInfo().isDeleted()) {
          Map<String, String> infectedFile = new HashMap<String, String>();
          String updaterId = infectedFileItem.getFileInfo().getUpdater();
          String updaterUsername =
                         NumberUtils.isCreatable(updaterId) ? identityManager.getIdentity(Long.parseLong(updaterId)).getRemoteId()
                                                            : updaterId;
          infectedFile.put(INFECTED_ITEM_ID, String.valueOf(infectedFileItem.getFileInfo().getId()));
          infectedFile.put(INFECTED_ITEM_NAME, infectedFileItem.getFileInfo().getName());
          infectedFile.put(INFECTED_ITEM_PATH, infectedFilePath);
          infectedFile.put(INFECTED_ITEM_LAST_MODIFIER, updaterUsername);
          infectedFile.put(INFECTED_ITEM_MODIFICATION_DATE, String.valueOf(infectedFileItem.getFileInfo().getUpdatedDate().getTime()));
          infectedFiles.add(infectedFile);
          long endTime = System.currentTimeMillis();
          LOGGER.info("service={} operation={} parameters=\"fileId:{}\" \"fileName:{}\" \"fileLastModifier:{}\" status=ok duration_ms={}",
                      MALWARE_DETECTION_FEATURE,
                      MALWARE_INFECTED_FILE_DETECTION,
                      infectedFileItem.getFileInfo().getId(),
                      infectedFileItem.getFileInfo().getName(),
                      infectedFileItem.getFileInfo().getUpdater(),
                      endTime - startTime);
        }
      }
    } catch (Exception e) {
      long endTime = System.currentTimeMillis();
      LOGGER.error("service={} operation={} parameters=\"fileChecksum:{}\" status=ko duration_ms={} error_msg=\"Error when trying to get the infected items:{}\"",
                   MALWARE_DETECTION_FEATURE,
                   MALWARE_INFECTED_FILE_DETECTION,
                   fileChecksum,
                   endTime - startTime,
                   e);
    }
    return infectedFiles;
  }
  
  @Override
  public boolean canProcessInfectedItem(String infectedFilePath) {
    String infectedFilePathSeparator = getPathSeparator(infectedFilePath);
    return infectedFilePath.contains(infectedFilePathSeparator + FILES + infectedFilePathSeparator);
  }
  
  @Override
  public void cleanInfectedItem(Map<String, String> infectedItem) {
    long startTime = System.currentTimeMillis();
    fileService.deleteFile(Long.parseLong(infectedItem.get(INFECTED_ITEM_ID)));
    long endTime = System.currentTimeMillis();
    LOGGER.info("service={} operation={} parameters=\"fileId:{}\" \"fileName:{}\" status=ok duration_ms={}",
                MALWARE_DETECTION_FEATURE,
                MALWARE_INFECTED_FILE_CLEAN,
                infectedItem.get(INFECTED_ITEM_ID),
                infectedItem.get(INFECTED_ITEM_NAME),
                endTime - startTime);
  }
  
  private String getInfectedFileChecksum(String infectedFilePath) {
    String infectedItemPathSeparator = getPathSeparator(infectedFilePath);
    return StringUtils.substringAfterLast(infectedFilePath, infectedItemPathSeparator);
  }
}
