package org.exoplatform.antimalware.connector;

import org.exoplatform.commons.file.model.FileInfo;
import org.exoplatform.commons.file.model.FileItem;
import org.exoplatform.commons.file.services.FileService;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.PropertiesParam;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

import java.util.*;

import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class MalwareDetectionFilesConnectorTest {

  @Mock
  private FileService                    fileService;

  private MalwareDetectionFilesConnector malwareDetectionFilesConnector;

  @Before
  public void setUp() {
    InitParams params = new InitParams();
    PropertiesParam constructorParams = new PropertiesParam();
    constructorParams.setName("constructor.params");
    constructorParams.setProperty("type", "files");
    constructorParams.setProperty("enabled", "true");
    params.addParameter(constructorParams);
    this.malwareDetectionFilesConnector = new MalwareDetectionFilesConnector(params, fileService);
  }

  @Test
  public void getInfectedItems() throws Exception {
    List<FileItem> infectedFileItems = new ArrayList<>();
    FileInfo fileInfo = new FileInfo(1L,
                                     "file",
                                     "application/pdf",
                                     "anyNameSpaceApp",
                                     1253L,
                                     new Date(),
                                     "user",
                                     "62bea6cf7f000101262f2b4a4c7638680",
                                     false);
    FileItem fileItem = new FileItem(fileInfo.getId(),
                                     fileInfo.getName(),
                                     fileInfo.getMimetype(),
                                     fileInfo.getNameSpace(),
                                     fileInfo.getSize(),
                                     fileInfo.getUpdatedDate(),
                                     fileInfo.getUpdater(),
                                     fileInfo.isDeleted(),
                                     null);
    infectedFileItems.add(fileItem);
    when(fileService.getFilesByChecksum(anyString())).thenReturn(infectedFileItems);
    List<Map<String, String>> infectedItems =
                                            this.malwareDetectionFilesConnector.getInfectedItems("gatein/8/c/d/8/62bea6cf7f000101262f2b4a4c7638680");

    assertEquals(1, infectedItems.size());
    assertEquals("1", infectedItems.get(0).get("infectedItemId"));
    assertEquals("file", infectedItems.get(0).get("infectedItemName"));
    assertEquals("user", infectedItems.get(0).get("infectedItemLastModifier"));
  }

  @Test
  public void canProcessInfectedItem() {
    assertFalse(this.malwareDetectionFilesConnector.canProcessInfectedItem("gatein/data/jcr/values/collaboration_portal8/c/d/8/62bea6cf7f000101262f2b4a4c7638680"));
    assertTrue(this.malwareDetectionFilesConnector.canProcessInfectedItem("gatein/data/files/8/c/d/8/62bea6cf7f000101262f2b4a4c7638680"));
  }

  @Test
  public void cleanInfectedItem() {
    Map<String, String> infectedItem = new HashMap<>();
    infectedItem.put("infectedItemId", "1");
    infectedItem.put("infectedItemName", "file");
    infectedItem.put("infectedItemLastModifier", "user");
    this.malwareDetectionFilesConnector.cleanInfectedItem(infectedItem);
    verify(fileService, times(1)).deleteFile(Long.parseLong(infectedItem.get("infectedItemId")));
  }
}
