/*
 * Copyright (C) 2025 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.connector;

import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.junit.jupiter.api.*;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.InjectMocks;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.test.util.ReflectionTestUtils;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
class ClamAVMalwareDetectionConnectorTest {

  @InjectMocks
  private ClamAVMalwareDetectionConnector connector;

  @Mock
  private MalwareDetectionService         malwareDetectionService;

  private Path                            tempReport;

  @AfterEach
  void cleanup() throws IOException {
    if (tempReport != null && Files.exists(tempReport)) {
      Files.delete(tempReport);
    }
  }

  @Test
  void testInitRegistersConnector() {
    connector.init();
    verify(malwareDetectionService, times(1)).addMalwareDetectionConnector(connector);
  }

  @Test
  void testGetInfectedItemsPathsWithReport() throws IOException {
    tempReport = Files.createTempFile("clamav_test", ".txt");

    String content = """
        /tmp/malware.exe: FOUND
        C:\\Users\\John\\infected.doc: FOUND
        /tmp/cleanfile.txt: OK
        """;

    Files.writeString(tempReport, content);

    ReflectionTestUtils.setField(connector, "reportPath", tempReport.toString());

    List<String> infected = connector.getInfectedItemsPaths();

    assertEquals(2, infected.size());
    assertTrue(infected.contains("/tmp/malware.exe"));
    assertTrue(infected.contains("C:\\Users\\John\\infected.doc"));

    // Verify the report file was truncated (cleared)
    assertEquals(0, Files.size(tempReport), "ClamAV report file should be truncated (empty) after reading");
  }

  @Test
  void testGetInfectedItemsPathsWithNonExistingFile() {
    ReflectionTestUtils.setField(connector, "reportPath", "/does/not/exist.txt");

    List<String> infected = connector.getInfectedItemsPaths();

    assertTrue(infected.isEmpty());
  }
}
