package org.exoplatform.antimalware.connector;

import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.exoplatform.services.listener.ListenerService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.test.util.ReflectionTestUtils;

import java.io.File;
import java.util.List;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
class ExternalFileSystemItemConnectorTest {

  @InjectMocks
  private ExternalFileSystemItemConnector connector;

  @Mock
  private MalwareDetectionService         malwareDetectionService;

  @Mock
  private ListenerService                 listenerService;

  @BeforeEach
  void setUp() {
    connector.init();
    ReflectionTestUtils.setField(connector, "malwareDetectionService", malwareDetectionService);
    ReflectionTestUtils.setField(connector, "listenerService", listenerService);
  }

  @Test
  void testCanProcessInfectedItem() {
    String externalPath = "/external/path/file.txt";
    String jcrPath = File.separator + "jcr" + File.separator + "something";
    String filesPath = File.separator + "files" + File.separator + "something";

    assertTrue(connector.canProcessInfectedItem(externalPath));
    assertFalse(connector.canProcessInfectedItem(jcrPath));
    assertFalse(connector.canProcessInfectedItem(filesPath));
  }

  @Test
  void testGetInfectedItems() {
    String path = "/some/path/infected.txt";
    List<Map<String, String>> items = connector.getInfectedItems(path);

    assertEquals(1, items.size());
    Map<String, String> item = items.getFirst();
    assertEquals(path, item.get("infectedItemPath"));
    assertEquals("externalFileSystem", item.get("externalFileSystem"));
  }

  @Test
  void testCleanInfectedItem() {
    Map<String, String> infectedItem = Map.of("infectedItemPath",
                                              "/some/path/infected.txt",
                                              "externalFileSystem",
                                              "externalFileSystem");

    connector.cleanInfectedItem(infectedItem);

    verify(listenerService).broadcast(eq("clean-file-infected"), eq("externalFileSystem"), eq(infectedItem));
  }

  @Test
  void testProcessInfectedItem() {
    String path = "/external/path/infected.txt";

    connector.processInfectedItem(path);

    ArgumentCaptor<Map<String, String>> captor = ArgumentCaptor.forClass(Map.class);
    verify(listenerService).broadcast(eq("clean-file-infected"), eq("externalFileSystem"), captor.capture());

    Map<String, String> capturedItem = captor.getValue();
    assertEquals(path, capturedItem.get("infectedItemPath"));
    assertEquals("externalFileSystem", capturedItem.get("externalFileSystem"));
  }
}
