package org.exoplatform.antimalware.connector;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import org.exoplatform.services.listener.ListenerService;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.junit.MockitoJUnitRunner;

import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.PropertiesParam;
import org.exoplatform.services.jcr.RepositoryService;
import org.exoplatform.services.jcr.core.ExtendedSession;
import org.exoplatform.services.jcr.core.ManageableRepository;
import org.exoplatform.services.jcr.ext.common.SessionProvider;
import org.exoplatform.services.jcr.impl.core.NodeImpl;
import org.exoplatform.services.wcm.core.NodetypeConstant;
import org.exoplatform.services.wcm.utils.WCMCoreUtils;

@RunWith(MockitoJUnitRunner.class)
public class MalwareDetectionJcrConnectorTest {

  private static final MockedStatic<WCMCoreUtils>            WCM_CORE_UTILS             = mockStatic(WCMCoreUtils.class);

  @Mock
  private RepositoryService                       repositoryService;

  @Mock
  private ListenerService                         listenerService;

  private MalwareDetectionJcrConnector malwareDetectionJcrConnector;

  @AfterClass
  public static void afterRunBare() throws Exception { // NOSONAR
    WCM_CORE_UTILS.close();
  }

  @Before
  public void setUp() throws Exception {
    InitParams params = new InitParams();
    PropertiesParam constructorParams = new PropertiesParam();
    constructorParams.setName("constructor.params");
    constructorParams.setProperty("type", "files");
    constructorParams.setProperty("enabled", "true");
    params.addParameter(constructorParams);
    this.malwareDetectionJcrConnector = new MalwareDetectionJcrConnector(params, repositoryService, listenerService);
  }

  @Test
  public void getInfectedItems() throws Exception {
    ManageableRepository repository = mock(ManageableRepository.class);
    when(repositoryService.getCurrentRepository()).thenReturn(repository);
    ExtendedSession session = mock(ExtendedSession.class);
    SessionProvider sessionProvider = mock(SessionProvider.class);
    WCM_CORE_UTILS.when(() -> WCMCoreUtils.getSystemSessionProvider()).thenReturn(sessionProvider);
    when(sessionProvider.getSession("collaboration", repository)).thenReturn(session);
    Property jcrDataProperty = mock(Property.class);
    Property lastModifier = mock(Property.class);
    Node infectedJcrContentNode = mock(Node.class);
    NodeImpl infectedJcrNode = mock(NodeImpl.class);
    when(jcrDataProperty.getName()).thenReturn("jcr:data");
    when(infectedJcrNode.getIdentifier()).thenReturn("8cd8");
    when(infectedJcrNode.getName()).thenReturn("file");
    when(jcrDataProperty.getParent()).thenReturn(infectedJcrContentNode);
    when(infectedJcrContentNode.getParent()).thenReturn(infectedJcrNode);
    when(session.getPropertyByIdentifier(anyString())).thenReturn(jcrDataProperty);
    when(lastModifier.getString()).thenReturn("user");
    when(infectedJcrNode.getProperty(NodetypeConstant.EXO_LAST_MODIFIER)).thenReturn(lastModifier);
    List<Map<String, String>> infectedItems =
                                            this.malwareDetectionJcrConnector.getInfectedItems("gatein/data/jcr/values/collaboration_portal/8/c/d/8/62bea6cf7f000101262f2b4a4c7638680");
    assertEquals(1, infectedItems.size());
    assertEquals("8cd8", infectedItems.get(0).get("infectedItemId"));
    assertEquals("file", infectedItems.get(0).get("infectedItemName"));
    assertEquals("user", infectedItems.get(0).get("infectedItemLastModifier"));
  }

  @Test
  public void canProcessInfectedItem() {
    assertTrue(this.malwareDetectionJcrConnector.canProcessInfectedItem("gatein/data/jcr/values/collaboration_portal8/c/d/8/62bea6cf7f000101262f2b4a4c7638680"));
    assertFalse(this.malwareDetectionJcrConnector.canProcessInfectedItem("gatein/data/files/8/c/d/8/62bea6cf7f000101262f2b4a4c7638680"));
  }

  @Test
  public void cleanInfectedItem() throws RepositoryException {
    ManageableRepository repository = mock(ManageableRepository.class);
    when(repositoryService.getCurrentRepository()).thenReturn(repository);
    ExtendedSession session = mock(ExtendedSession.class);
    SessionProvider sessionProvider = mock(SessionProvider.class);
    WCM_CORE_UTILS.when(() -> WCMCoreUtils.getSystemSessionProvider()).thenReturn(sessionProvider);
    when(sessionProvider.getSession("collaboration", repository)).thenReturn(session);
    Node infectedJcrNode = mock(Node.class);
    when(session.getNodeByIdentifier("8cd8")).thenReturn(infectedJcrNode);
    Map<String, String> infectedItem = new HashMap<>();
    infectedItem.put("infectedItemId", "8cd8");
    infectedItem.put("infectedJcrNodeWorkspace", "collaboration");
    this.malwareDetectionJcrConnector.cleanInfectedItem(infectedItem);
    verify(infectedJcrNode, times(1)).remove();
    verify(session, times(1)).save();

  }
}
