/*
 * Copyright (C) 2025 eXo Platform SAS
 *
 *  This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <gnu.org/licenses>.
 */
package org.exoplatform.antimalware.listener;

import io.meeds.analytics.model.StatisticData;
import io.meeds.analytics.utils.AnalyticsUtils;
import org.exoplatform.antimalware.connector.MalwareDetectionConnector;
import org.exoplatform.antimalware.service.MalwareDetectionService;
import org.exoplatform.services.listener.Event;
import org.exoplatform.services.listener.ListenerService;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.*;

import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.test.util.ReflectionTestUtils;

import java.util.HashMap;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class AntiMalwareAnalyticsListenerTest {

  @Mock
  private ListenerService              listenerService;

  @Mock
  private MalwareDetectionService      malwareDetectionService;

  @InjectMocks
  private AntiMalwareAnalyticsListener listener;

  @Before
  public void setUp() {
    listener.init();
    ReflectionTestUtils.setField(listener, "malwareDetectionService", malwareDetectionService);
    MalwareDetectionConnector malwareDetectionConnector = mock(MalwareDetectionConnector.class);
    when(malwareDetectionConnector.getType()).thenReturn("type");
    when(malwareDetectionService.getDefaultMalwareDetectionConnector()).thenReturn(malwareDetectionConnector);
  }

  @Test
  public void testOnEvent() throws Exception {
    String connectorType = "ClamAV";
    int infectedCount = 5;
    Map<String, String> infectedFile = new HashMap<>();
    infectedFile.put("infectedItemId", "1");
    infectedFile.put("infectedItemName", "filetesteicar.txt");
    infectedFile.put("infectedItemLastModifier", "user");
    infectedFile.put("infectedItemPath", "path");

    Event<String, Object> event = new Event<>("anti-malware-scan-completed", connectorType, infectedCount);
    Event<String, Object> event1 = new Event<>("anti-malware-infectedFile-removed", "type", infectedFile);

    try (MockedStatic<AnalyticsUtils> mockedStatic = Mockito.mockStatic(AnalyticsUtils.class)) {

      ArgumentCaptor<StatisticData> captor = ArgumentCaptor.forClass(StatisticData.class);

      listener.onEvent(event);

      mockedStatic.verify(() -> AnalyticsUtils.addStatisticData(captor.capture()), times(1));

      StatisticData data = captor.getValue();

      assertEquals("AntiMalware", data.getModule());
      assertEquals("antimalwareCheck", data.getOperation());
      assertEquals(String.valueOf(infectedCount), data.getParameters().get("infectedFilesCount"));
      assertEquals(connectorType, data.getParameters().get("connectorType"));

      ArgumentCaptor<StatisticData> captor1 = ArgumentCaptor.forClass(StatisticData.class);

      listener.onEvent(event1);

      mockedStatic.verify(() -> AnalyticsUtils.addStatisticData(captor1.capture()), times(2));

      StatisticData data1 = captor1.getValue();

      assertEquals("AntiMalware", data1.getModule());
      assertEquals("InfectedFileRemoval", data1.getOperation());
      assertEquals(String.valueOf(infectedFile.get("infectedItemId")), data1.getParameters().get("fileId"));
      assertEquals(String.valueOf(infectedFile.get("infectedItemName")), data1.getParameters().get("fileName"));
      assertEquals(String.valueOf(infectedFile.get("infectedItemLastModifier")), data1.getParameters().get("fileAuthor"));
      assertEquals(String.valueOf(infectedFile.get("infectedItemPath")), data1.getParameters().get("filePath"));
      assertEquals("type", data1.getParameters().get("connectorType"));
    }
  }

}
