package org.exoplatform.antimalware.notification.plugin;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.*;

import org.exoplatform.commons.utils.ListAccess;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.User;
import org.exoplatform.services.organization.UserHandler;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.junit.MockitoJUnitRunner;

import org.exoplatform.antimalware.connector.MalwareDetectionItemConnector;
import org.exoplatform.commons.api.notification.NotificationContext;
import org.exoplatform.commons.api.notification.model.NotificationInfo;
import org.exoplatform.commons.notification.impl.NotificationContextImpl;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.services.idgenerator.IDGeneratorService;

import java.util.List;

@RunWith(MockitoJUnitRunner.class)
public class MalwareDetectionPluginTest {

  @Mock
  private OrganizationService                            organizationService;

  private static final MockedStatic<CommonsUtils>        COMMONS_UTILS         = mockStatic(CommonsUtils.class);

  private static final MockedStatic<ExoContainerContext> EXO_CONTAINER_CONTEXT = mockStatic(ExoContainerContext.class);

  @Mock
  private InitParams                                     initParams;

  private MalwareDetectionPlugin                         malwareDetectionPlugin;

  @AfterClass
  public static void afterRunBare() throws Exception { // NOSONAR
    EXO_CONTAINER_CONTEXT.close();
    COMMONS_UTILS.close();
  }

  @Before
  public void setUp() throws Exception {
    this.malwareDetectionPlugin = new MalwareDetectionPlugin(initParams, organizationService);
    EXO_CONTAINER_CONTEXT.when(() -> ExoContainerContext.getService(IDGeneratorService.class)).thenReturn(null);
  }

  @Test
  public void makeNotification() throws Exception {
    User dlpUser = mock(User.class);
    when(dlpUser.getUserName()).thenReturn("dlpUser");
    User adminUser = mock(User.class);
    when(adminUser.getUserName()).thenReturn("adminUser");

    ListAccess<User> dlpListAccess = mock(ListAccess.class);
    when(dlpListAccess.load(anyInt(), anyInt())).thenReturn(new User[] { dlpUser });
    when(dlpListAccess.getSize()).thenReturn(1);

    ListAccess<User> adminListAccess = mock(ListAccess.class);
    when(adminListAccess.load(anyInt(), anyInt())).thenReturn(new User[] { adminUser });
    when(adminListAccess.getSize()).thenReturn(1);

    UserHandler userHandler = mock(UserHandler.class);
    when(organizationService.getUserHandler()).thenReturn(userHandler);
    when(userHandler.findUsersByGroupId("/platform/dlp")).thenReturn(dlpListAccess);
    when(userHandler.findUsersByGroupId("/platform/administrators")).thenReturn(adminListAccess);

    NotificationContext ctx = NotificationContextImpl.cloneInstance();
    ctx.append(MalwareDetectionItemConnector.INFECTED_ITEM_NAME_ARGUMENT, "file.docx");
    ctx.append(MalwareDetectionItemConnector.INFECTED_ITEM_LAST_MODIFIER_ARGUMENT, "john");

    NotificationInfo notificationInfo = malwareDetectionPlugin.makeNotification(ctx);

    assertEquals("file.docx",
                 notificationInfo.getValueOwnerParameter(MalwareDetectionItemConnector.INFECTED_ITEM_NAME_ARGUMENT.getKey()));

    List<String> receivers = notificationInfo.getSendToUserIds();
    assertEquals(3, receivers.size());
    assertTrue(receivers.contains("dlpUser"));
    assertTrue(receivers.contains("adminUser"));
    assertTrue(receivers.contains("john"));
  }
}
