/*
 * Copyright (C) 2003-2007 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */

package org.exoplatform.services.jcr.webdav.command;

import org.exoplatform.services.jcr.webdav.WebDavStatus;
import org.exoplatform.services.rest.Response;

import javax.jcr.AccessDeniedException;
import javax.jcr.ItemExistsException;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.lock.LockException;

/**
 * Created by The eXo Platform SAS Author : Vitaly Guly <gavrikvetal@gmail.com>
 * 
 * @version $Id: CopyCommand.java 67273 2011-03-17 10:24:40Z trang_vu $
 */

public class CopyCommand {

  /**
   * To trace if an item on destination path existed.
   */

  final private boolean itemExisted;

  public CopyCommand() {
    this.itemExisted = false;
  }

  /**
   * Here we pass info about pre-existence of item on the copy destination path
   * If an item existed, we must respond with NO_CONTENT (204) HTTP status. If
   * an item did not exist, we must respond with CREATED (201) HTTP status More
   * info can be found <a
   * href=http://www.webdav.org/specs/rfc2518.html#METHOD_COPY>here</a>.
   */
  public CopyCommand(boolean itemExisted) {
    this.itemExisted = itemExisted;
  }

  public Response copy(Session destSession, String sourcePath, String destPath) {
    try {
      destSession.getWorkspace().copy(sourcePath, destPath);
      // If the source resource was successfully moved
      // to a pre-existing destination resource.
      if (itemExisted) {
        return Response.Builder.withStatus(WebDavStatus.NO_CONTENT).build();
      }
      // If the source resource was successfully moved,
      // and a new resource was created at the destination.
      else {
        return Response.Builder.withStatus(WebDavStatus.CREATED).build();
      }
    } catch (ItemExistsException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.METHOD_NOT_ALLOWED).errorMessage(e.getMessage()).build();
    } catch (PathNotFoundException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.CONFLICT).errorMessage(e.getMessage()).build();
    } catch (AccessDeniedException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.FORBIDDEN).errorMessage(e.getMessage()).build();
    } catch (LockException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.LOCKED).errorMessage(e.getMessage()).build();
    } catch (RepositoryException e) {
      e.printStackTrace();
      return Response.Builder.serverError().errorMessage(e.getMessage()).build();
    }
  }

  public Response copy(Session destSession,
                       String sourceWorkspace,
                       String sourcePath,
                       String destPath) {
    try {
      destSession.getWorkspace().copy(sourceWorkspace, sourcePath, destPath);
      // If the source resource was successfully moved
      // to a pre-existing destination resource.
      if (itemExisted) {
        return Response.Builder.withStatus(WebDavStatus.NO_CONTENT).build();
      }
      // If the source resource was successfully moved,
      // and a new resource was created at the destination.
      else {
        return Response.Builder.withStatus(WebDavStatus.CREATED).build();
      }
    } catch (ItemExistsException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.METHOD_NOT_ALLOWED).errorMessage(e.getMessage()).build();
    } catch (PathNotFoundException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.CONFLICT).errorMessage(e.getMessage()).build();
    } catch (AccessDeniedException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.FORBIDDEN).errorMessage(e.getMessage()).build();
    } catch (LockException e) {
      e.printStackTrace();
      return Response.Builder.withStatus(WebDavStatus.LOCKED).errorMessage(e.getMessage()).build();
    } catch (RepositoryException e) {
      e.printStackTrace();
      return Response.Builder.serverError().errorMessage(e.getMessage()).build();
    }
  }
}
