/*
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.services.jcr.impl.storage.jdbc.optimisation.db;

import org.exoplatform.services.jcr.datamodel.NodeData;
import org.exoplatform.services.jcr.datamodel.PropertyData;
import org.exoplatform.services.jcr.datamodel.ValueData;
import org.exoplatform.services.jcr.impl.Constants;
import org.exoplatform.services.jcr.impl.core.itemfilters.QPathEntryFilter;
import org.exoplatform.services.jcr.impl.dataflow.ValueDataUtil;
import org.exoplatform.services.jcr.impl.storage.jdbc.JDBCDataContainerConfig;
import org.exoplatform.services.jcr.impl.storage.jdbc.optimisation.CQJDBCStorageConnection;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.List;

import javax.jcr.InvalidItemStateException;
import javax.jcr.RepositoryException;

/**
 * Single database connection implementation.
 * 
 * Created by The eXo Platform SAS. </br>
 * 
 * @author <a href="mailto:gennady.azarenkov@exoplatform.com">Gennady
 *         Azarenkov</a>
 * @version $Id: MultiDbJDBCConnection.java 20950 2008-10-06 14:23:07Z
 *          pnedonosko $
 */

public class MultiDbJDBCConnection extends CQJDBCStorageConnection
{
   protected String FIND_NODES_BY_PARENTID_CQ_QUERY;

   protected String FIND_PROPERTIES_BY_PARENTID_CQ_QUERY;

   protected String FIND_ITEM_QPATH_BY_ID_CQ_QUERY;

   /**
    * Multidatabase JDBC Connection constructor.
    * 
    * @param dbConnection
    *          JDBC connection, should be opened before
    * @param readOnly
    *          boolean if true the dbConnection was marked as READ-ONLY.
    * @param containerConfig
    *          Workspace Storage Container configuration
    */
   public MultiDbJDBCConnection(Connection dbConnection, boolean readOnly, JDBCDataContainerConfig containerConfig)
      throws SQLException
   {
      super(dbConnection, readOnly, containerConfig);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected String getIdentifier(final String internalId)
   {
      return internalId;
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected String getInternalId(final String identifier)
   {
      return identifier;
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected void prepareQueries() throws SQLException
   {
      // ============================================================================
      FIND_NODES_BY_PARENTID_CQ_QUERY =
         "select I.*, P.NAME AS PROP_NAME, V.ORDER_NUM, V.DATA from " + JCR_ITEM + " I, " + JCR_ITEM + " P, "
            + JCR_VALUE + " V" + " where I.I_CLASS=1 and I.PARENT_ID=? and P.I_CLASS=2 and P.PARENT_ID=I.ID and"
            + " (P.NAME='[http://www.jcp.org/jcr/1.0]primaryType' or P.NAME='[http://www.jcp.org/jcr/1.0]mixinTypes'"
            + " or P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]owner'"
            + " or P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]permissions')"
            + " and V.PROPERTY_ID=P.ID order by I.N_ORDER_NUM, I.ID";

      FIND_PROPERTIES_BY_PARENTID_CQ_QUERY =
         "select I.ID, I.PARENT_ID, I.NAME, I.VERSION, I.I_CLASS, I.I_INDEX, I.N_ORDER_NUM, I.P_TYPE, I.P_MULTIVALUED,"
            + " V.ORDER_NUM, V.DATA, V.STORAGE_DESC from " + JCR_ITEM + " I LEFT OUTER JOIN " + JCR_VALUE
            + " V ON (V.PROPERTY_ID=I.ID)" + " where I.I_CLASS=2 and I.PARENT_ID=? order by I.NAME";

      FIND_ITEM_QPATH_BY_ID_CQ_QUERY =
         "select I.ID, I.PARENT_ID, I.NAME, I.I_INDEX" + " from " + JCR_ITEM + " I, (SELECT ID, PARENT_ID from "
            + JCR_ITEM + " where ID=?) J" + " where I.ID = J.ID or I.ID = J.PARENT_ID";
      // ============================================================================

      FIND_ITEM_BY_ID = "select * from " + JCR_ITEM + " where ID=?";

      FIND_ITEM_BY_NAME =
         "select * from " + JCR_ITEM + " I where PARENT_ID=? and NAME=? and I_INDEX=? order by I_CLASS, VERSION DESC";

      FIND_PROPERTY_BY_NAME =
         "select V.DATA" + " from " + JCR_ITEM + " I, " + JCR_VALUE + " V"
            + " where I.I_CLASS=2 and I.PARENT_ID=? and I.NAME=? and I.ID=V.PROPERTY_ID order by V.ORDER_NUM";

      FIND_REFERENCES =
         "select P.ID, P.PARENT_ID, P.VERSION, P.P_TYPE, P.P_MULTIVALUED, P.NAME" + " from " + JCR_REF + " R, "
            + JCR_ITEM + " P" + " where R.NODE_ID=? and P.ID=R.PROPERTY_ID and P.I_CLASS=2";

      FIND_VALUES_BY_PROPERTYID =
         "select PROPERTY_ID, ORDER_NUM, DATA, STORAGE_DESC from " + JCR_VALUE
            + " where PROPERTY_ID=? order by ORDER_NUM";

      FIND_NODES_BY_PARENTID =
         "select * from " + JCR_ITEM + " where I_CLASS=1 and PARENT_ID=?" + " order by N_ORDER_NUM";

      FIND_NODES_BY_PARENTID_CQ = FIND_NODES_BY_PARENTID_CQ_QUERY;

      FIND_NODE_MAIN_PROPERTIES_BY_PARENTID_CQ =
         "select I.NAME, V.DATA, V.ORDER_NUM from " + JCR_ITEM + " I, " + JCR_VALUE + " V"
            + " where I.I_CLASS=2 and I.PARENT_ID=? and (I.NAME='[http://www.jcp.org/jcr/1.0]primaryType' or"
            + " I.NAME='[http://www.jcp.org/jcr/1.0]mixinTypes' or"
            + " I.NAME='[http://www.exoplatform.com/jcr/exo/1.0]owner' or"
            + " I.NAME='[http://www.exoplatform.com/jcr/exo/1.0]permissions') and " + "I.ID=V.PROPERTY_ID";

      FIND_ITEM_QPATH_BY_ID_CQ = FIND_ITEM_QPATH_BY_ID_CQ_QUERY;

      FIND_LAST_ORDER_NUMBER_BY_PARENTID =
         "select count(*), max(N_ORDER_NUM) from " + JCR_ITEM + " where I_CLASS=1 and PARENT_ID=?";

      FIND_NODES_COUNT_BY_PARENTID = "select count(ID) from " + JCR_ITEM + " where I_CLASS=1 and PARENT_ID=?";

      FIND_PROPERTIES_BY_PARENTID = "select * from " + JCR_ITEM + " where I_CLASS=2 and PARENT_ID=?" + " order by NAME";

      // property may contain no values
      FIND_PROPERTIES_BY_PARENTID_CQ = FIND_PROPERTIES_BY_PARENTID_CQ_QUERY;

      FIND_PROPERTIES_BY_PARENTID_AND_PATTERN_CQ_TEMPLATE =
         "select I.ID, I.PARENT_ID, I.NAME, I.VERSION, I.I_CLASS, I.I_INDEX, I.N_ORDER_NUM, I.P_TYPE, I.P_MULTIVALUED,"
            + " V.ORDER_NUM, V.DATA, V.STORAGE_DESC from " + JCR_ITEM + " I LEFT OUTER JOIN " + JCR_VALUE
            + " V ON (V.PROPERTY_ID=I.ID)";

      FIND_NODES_BY_PARENTID_AND_PATTERN_CQ_TEMPLATE =
         "select I.*, P.NAME AS PROP_NAME, V.ORDER_NUM, V.DATA from " + JCR_ITEM + " I, " + JCR_ITEM + " P, "
            + JCR_VALUE + " V";

      FIND_MAX_PROPERTY_VERSIONS =
         "select max(VERSION) FROM " + JCR_ITEM + " WHERE PARENT_ID=? and NAME=? and I_INDEX=? and I_CLASS=2";

      INSERT_NODE =
         "insert into " + JCR_ITEM + " (ID, PARENT_ID, NAME, VERSION, I_CLASS, I_INDEX, N_ORDER_NUM) VALUES(?,?,?,?,"
            + I_CLASS_NODE + ",?,?)";
      INSERT_PROPERTY =
         "insert into " + JCR_ITEM
            + "(ID, PARENT_ID, NAME, VERSION, I_CLASS, I_INDEX, P_TYPE, P_MULTIVALUED) VALUES(?,?,?,?,"
            + I_CLASS_PROPERTY + ",?,?,?)";

      INSERT_VALUE = "insert into " + JCR_VALUE + "(DATA, ORDER_NUM, PROPERTY_ID, STORAGE_DESC) VALUES(?,?,?,?)";
      INSERT_REF = "insert into " + JCR_REF + "(NODE_ID, PROPERTY_ID, ORDER_NUM) VALUES(?,?,?)";

      RENAME_NODE =
         "update " + JCR_ITEM + " set PARENT_ID=?, NAME =?, VERSION=?, I_INDEX =?, N_ORDER_NUM =? where ID=?";

      UPDATE_NODE = "update " + JCR_ITEM + " set VERSION=?, I_INDEX=?, N_ORDER_NUM=? where ID=?";
      UPDATE_PROPERTY = "update " + JCR_ITEM + " set VERSION=?, P_TYPE=? where ID=?";
      //UPDATE_VALUE = "update "+JCR_VALUE+" set DATA=?, STORAGE_DESC=? where PROPERTY_ID=?, ORDER_NUM=?";

      DELETE_ITEM = "delete from " + JCR_ITEM + " where ID=?";
      DELETE_VALUE = "delete from " + JCR_VALUE + " where PROPERTY_ID=?";
      DELETE_REF = "delete from " + JCR_REF + " where PROPERTY_ID=?";

      FIND_NODES_AND_PROPERTIES =
         "select J.*, P.ID AS P_ID, P.NAME AS P_NAME, P.VERSION AS P_VERSION, P.P_TYPE, P.P_MULTIVALUED,"
            + " V.DATA, V.ORDER_NUM, V.STORAGE_DESC from " + JCR_VALUE + " V, " + JCR_ITEM + " P"
            + " join (select I.ID, I.PARENT_ID, I.NAME, I.VERSION, I.I_INDEX, I.N_ORDER_NUM from " + JCR_ITEM + " I"
            + " where I.I_CLASS=1 AND I.ID > ? order by I.ID LIMIT ? OFFSET ?) J on P.PARENT_ID = J.ID"
            + " where P.I_CLASS=2 and V.PROPERTY_ID=P.ID  order by J.ID";

      FIND_PROPERTY_BY_ID =
         "select length(DATA), I.P_TYPE, V.STORAGE_DESC from " + JCR_ITEM + " I, " + JCR_VALUE
            + " V where I.ID = ? and V.PROPERTY_ID = I.ID";
      DELETE_VALUE_BY_ORDER_NUM = "delete from " + JCR_VALUE + " where PROPERTY_ID=? and ORDER_NUM >= ?";
      DELETE_REFERENCE_BY_ORDER_NUM = "delete from " + JCR_REF + " where PROPERTY_ID=? and ORDER_NUM >= ?";
      UPDATE_VALUE = "update " + JCR_VALUE + " set DATA=?, STORAGE_DESC=? where PROPERTY_ID=? and ORDER_NUM=?";
      UPDATE_REFERENCE = "update " + JCR_REF + " set NODE_ID=? where PROPERTY_ID=? and ORDER_NUM=?";

      if (containerConfig.useSequenceForOrderNumber)
      {
         FIND_NODES_BY_PARENTID_LAZILY_CQ =
            "select I.*, P.NAME AS PROP_NAME, V.ORDER_NUM, V.DATA from " + JCR_VALUE + " V, " + JCR_ITEM + " P "
               + " join (select J.* from " + JCR_ITEM + " J where J.I_CLASS=1 and J.PARENT_ID=?"
               + " order by J.N_ORDER_NUM, J.ID  LIMIT ? OFFSET ?) I on P.PARENT_ID = I.ID"
               + " where P.I_CLASS=2 and P.PARENT_ID=I.ID and"
               + " (P.NAME='[http://www.jcp.org/jcr/1.0]primaryType' or"
               + " P.NAME='[http://www.jcp.org/jcr/1.0]mixinTypes' or"
               + " P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]owner' or"
               + " P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]permissions')"
               + " and V.PROPERTY_ID=P.ID order by I.N_ORDER_NUM, I.ID";
      }
      else
      {
         FIND_NODES_BY_PARENTID_LAZILY_CQ =
            "select I.*, P.NAME AS PROP_NAME, V.ORDER_NUM, V.DATA from " + JCR_ITEM + " I, " + JCR_ITEM + " P, "
               + JCR_VALUE + " V"
               + " where I.I_CLASS=1 and I.PARENT_ID=? and I.N_ORDER_NUM >= ? and I.N_ORDER_NUM <= ? and"
               + " P.I_CLASS=2 and P.PARENT_ID=I.ID and (P.NAME='[http://www.jcp.org/jcr/1.0]primaryType' or"
               + " P.NAME='[http://www.jcp.org/jcr/1.0]mixinTypes' or"
               + " P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]owner' or"
               + " P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]permissions')"
               + " and V.PROPERTY_ID=P.ID order by I.N_ORDER_NUM, I.ID";
      }
      FIND_ACL_HOLDERS =
         "select I.PARENT_ID, I.P_TYPE " + " from " + JCR_ITEM
            + " I where I.I_CLASS=2 and (I.NAME='[http://www.exoplatform.com/jcr/exo/1.0]owner'"
            + " or I.NAME='[http://www.exoplatform.com/jcr/exo/1.0]permissions')";

      FIND_NODES_COUNT = "select count(*) from " + JCR_ITEM + " I where I.I_CLASS=1";

      FIND_WORKSPACE_DATA_SIZE = "select sum(length(DATA)) from " + JCR_VALUE;

      FIND_WORKSPACE_PROPERTIES_ON_VALUE_STORAGE =
         "select PROPERTY_ID, STORAGE_DESC, ORDER_NUM from " + JCR_VALUE + " where STORAGE_DESC is not null";

      FIND_NODE_DATA_SIZE =
         "select sum(length(DATA)) from " + JCR_ITEM + " I, " + JCR_VALUE
            + " V  where I.PARENT_ID=? and I.I_CLASS=2 and I.ID=V.PROPERTY_ID";

      FIND_NODE_PROPERTIES_ON_VALUE_STORAGE =
         "select V.PROPERTY_ID, V.STORAGE_DESC, V.ORDER_NUM from " + JCR_ITEM + " I, " + JCR_VALUE + " V  "
            + "where I.PARENT_ID=? and I.I_CLASS=2 and I.ID=V.PROPERTY_ID and V.STORAGE_DESC is not null";

      FIND_VALUE_STORAGE_DESC_AND_SIZE = "select length(DATA), STORAGE_DESC from " + JCR_VALUE + " where PROPERTY_ID=?";
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int addNodeRecord(NodeData data) throws SQLException, InvalidItemStateException, RepositoryException
   {
      if (insertNode == null)
      {
         insertNode = dbConnection.prepareStatement(INSERT_NODE);
      }
      else
      {
         insertNode.clearParameters();
      }

      insertNode.setString(1, data.getIdentifier());
      insertNode.setString(2,
         data.getParentIdentifier() == null ? Constants.ROOT_PARENT_UUID : data.getParentIdentifier());
      insertNode.setString(3, data.getQPath().getName().getAsString());
      insertNode.setInt(4, data.getPersistedVersion());
      insertNode.setInt(5, data.getQPath().getIndex());
      insertNode.setInt(6, data.getOrderNumber());

      return executeUpdate(insertNode, TYPE_INSERT_NODE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int addPropertyRecord(PropertyData data) throws SQLException, InvalidItemStateException,
      RepositoryException
   {
      if (insertProperty == null)
      {
         insertProperty = dbConnection.prepareStatement(INSERT_PROPERTY);
      }
      else
      {
         insertProperty.clearParameters();
      }

      insertProperty.setString(1, data.getIdentifier());
      insertProperty.setString(2, data.getParentIdentifier());
      insertProperty.setString(3, data.getQPath().getName().getAsString());
      insertProperty.setInt(4, data.getPersistedVersion());
      insertProperty.setInt(5, data.getQPath().getIndex());
      insertProperty.setInt(6, data.getType());
      insertProperty.setBoolean(7, data.isMultiValued());

      return executeUpdate(insertProperty, TYPE_INSERT_PROPERTY);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int addReference(PropertyData data) throws SQLException, IOException, InvalidItemStateException,
      RepositoryException
   {

      if (data.getQPath().getAsString().indexOf("versionableUuid") > 0)
      {
         LOG.info("add ref versionableUuid " + data.getQPath().getAsString());
      }

      List<ValueData> values = data.getValues();
      int added = 0;
      for (int i = 0; i < values.size(); i++)
      {
         ValueData vdata = values.get(i);
         String refNodeIdentifier;
         try
         {
            refNodeIdentifier = ValueDataUtil.getString(vdata);
         }
         catch (RepositoryException e)
         {
            throw new IOException(e.getMessage(), e);
         }

         added += addReference(data.getIdentifier(), i, refNodeIdentifier);
      }
      return added;
   }
   
   /**
    * {@inheritDoc}
    */
   @Override
   protected int deleteReferenceByOrderNum(String id, int orderNum) throws SQLException, InvalidItemStateException,
      RepositoryException
   {
      if (deleteReferenceByOrderNum == null)
      {
         deleteReferenceByOrderNum = dbConnection.prepareStatement(DELETE_REFERENCE_BY_ORDER_NUM);
      }
      else
      {
         deleteReferenceByOrderNum.clearParameters();
      }

      deleteReferenceByOrderNum.setString(1, id);
      deleteReferenceByOrderNum.setInt(2, orderNum);

      return executeUpdate(deleteReferenceByOrderNum, TYPE_DELETE_REFERENCE_BY_ORDER_NUM);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int addReference(String cid, int i, String refNodeIdentifier) throws SQLException,
      InvalidItemStateException, RepositoryException
   {
      if (insertReference == null)
      {
         insertReference = dbConnection.prepareStatement(INSERT_REF);
      }
      else
      {
         insertReference.clearParameters();
      }
      insertReference.setString(1, refNodeIdentifier);
      insertReference.setString(2, cid);
      insertReference.setInt(3, i);

      return executeUpdate(insertReference, TYPE_INSERT_REFERENCE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int updateReference(String cid, int i, String refNodeIdentifier) throws SQLException,
      InvalidItemStateException, RepositoryException
   {
      if (updateReference == null)
      {
         updateReference = dbConnection.prepareStatement(UPDATE_REFERENCE);
      }
      else
      {
         updateReference.clearParameters();
      }

      updateReference.setString(1, refNodeIdentifier);
      updateReference.setString(2, cid);
      updateReference.setInt(3, i);

      return executeUpdate(updateReference, TYPE_UPDATE_REFERENCE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int deleteReference(String propertyIdentifier) throws SQLException, InvalidItemStateException,
      RepositoryException
   {
      if (deleteReference == null)
      {
         deleteReference = dbConnection.prepareStatement(DELETE_REF);
      }
      else
      {
         deleteReference.clearParameters();
      }

      deleteReference.setString(1, propertyIdentifier);

      return executeUpdate(deleteReference, TYPE_DELETE_REFERENCE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int deleteItemByIdentifier(String identifier) throws SQLException, InvalidItemStateException,
      RepositoryException
   {
      if (deleteItem == null)
      {
         deleteItem = dbConnection.prepareStatement(DELETE_ITEM);
      }
      else
      {
         deleteItem.clearParameters();
      }

      deleteItem.setString(1, identifier);

      return executeUpdate(deleteItem, TYPE_DELETE_ITEM);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int updateNodeByIdentifier(int version, int index, int orderNumb, String identifier) throws SQLException,
      InvalidItemStateException, RepositoryException
   {
      if (updateNode == null)
      {
         updateNode = dbConnection.prepareStatement(UPDATE_NODE);
      }
      else
      {
         updateNode.clearParameters();
      }

      updateNode.setInt(1, version);
      updateNode.setInt(2, index);
      updateNode.setInt(3, orderNumb);
      updateNode.setString(4, identifier);

      return executeUpdate(updateNode, TYPE_UPDATE_NODE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int updatePropertyByIdentifier(int version, int type, String identifier) throws SQLException,
      InvalidItemStateException, RepositoryException
   {
      if (updateProperty == null)
      {
         updateProperty = dbConnection.prepareStatement(UPDATE_PROPERTY);
      }
      else
      {
         updateProperty.clearParameters();
      }

      updateProperty.setInt(1, version);
      updateProperty.setInt(2, type);
      updateProperty.setString(3, identifier);

      return executeUpdate(updateProperty, TYPE_UPDATE_PROPERTY);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findItemByName(String parentId, String name, int index) throws SQLException
   {
      if (findItemByName == null)
      {
         findItemByName = dbConnection.prepareStatement(FIND_ITEM_BY_NAME);
      }
      else
      {
         findItemByName.clearParameters();
      }

      findItemByName.setString(1, parentId);
      findItemByName.setString(2, name);
      findItemByName.setInt(3, index);
      return findItemByName.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findPropertyByName(String parentId, String name) throws SQLException
   {
      if (findPropertyByName == null)
      {
         findPropertyByName = dbConnection.prepareStatement(FIND_PROPERTY_BY_NAME);
      }
      else
      {
         findPropertyByName.clearParameters();
      }

      findPropertyByName.setString(1, parentId);
      findPropertyByName.setString(2, name);
      return findPropertyByName.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findItemByIdentifier(String identifier) throws SQLException
   {
      if (findItemById == null)
      {
         findItemById = dbConnection.prepareStatement(FIND_ITEM_BY_ID);
      }
      else
      {
         findItemById.clearParameters();
      }

      findItemById.setString(1, identifier);
      return findItemById.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findReferences(String nodeIdentifier) throws SQLException
   {
      if (findReferences == null)
      {
         findReferences = dbConnection.prepareStatement(FIND_REFERENCES);
      }
      else
      {
         findReferences.clearParameters();
      }

      findReferences.setString(1, nodeIdentifier);
      return findReferences.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildNodesByParentIdentifier(String parentIdentifier) throws SQLException
   {
      if (findNodesByParentId == null)
      {
         findNodesByParentId = dbConnection.prepareStatement(FIND_NODES_BY_PARENTID);
      }
      else
      {
         findNodesByParentId.clearParameters();
      }

      findNodesByParentId.setString(1, parentIdentifier);
      return findNodesByParentId.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildNodesByParentIdentifierCQ(String parentIdentifier) throws SQLException
   {
      if (findNodesByParentIdCQ == null)
      {
         findNodesByParentIdCQ = dbConnection.prepareStatement(FIND_NODES_BY_PARENTID_CQ);
      }
      else
      {
         findNodesByParentIdCQ.clearParameters();
      }

      findNodesByParentIdCQ.setString(1, parentIdentifier);
      return findNodesByParentIdCQ.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildNodesByParentIdentifierCQ(String parentIdentifier, List<QPathEntryFilter> pattern)
      throws SQLException
   {
      if (pattern.isEmpty())
      {
         throw new SQLException("Pattern list is empty.");
      }
      else
      {
         if (findNodesByParentIdAndComplexPatternCQ == null)
         {
            findNodesByParentIdAndComplexPatternCQ = dbConnection.createStatement();
         }

         //create query from list
         StringBuilder query = new StringBuilder(FIND_NODES_BY_PARENTID_AND_PATTERN_CQ_TEMPLATE);
         query.append(" where I.I_CLASS=1 and I.PARENT_ID='");
         query.append(parentIdentifier);
         query.append("' and ( ");
         appendPattern(query, pattern.get(0).getQPathEntry(), true);
         for (int i = 1; i < pattern.size(); i++)
         {
            query.append(" or ");
            appendPattern(query, pattern.get(i).getQPathEntry(), true);
         }
         query.append(" ) and P.I_CLASS=2 and P.PARENT_ID=I.ID and (P.NAME='[http://www.jcp.org/jcr/1.0]primaryType'");
         query.append(" or P.NAME='[http://www.jcp.org/jcr/1.0]mixinTypes'");
         query.append(" or P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]owner'");
         query.append(" or P.NAME='[http://www.exoplatform.com/jcr/exo/1.0]permissions')");
         query.append(" and V.PROPERTY_ID=P.ID order by I.N_ORDER_NUM, I.ID");

         return findNodesByParentIdAndComplexPatternCQ.executeQuery(query.toString());
      }
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findLastOrderNumberByParentIdentifier(String parentIdentifier) throws SQLException
   {
      if (findLastOrderNumberByParentId == null)
      {
         findLastOrderNumberByParentId = dbConnection.prepareStatement(FIND_LAST_ORDER_NUMBER_BY_PARENTID);
      }
      else
      {
         findLastOrderNumberByParentId.clearParameters();
      }

      findLastOrderNumberByParentId.setString(1, parentIdentifier);
      return findLastOrderNumberByParentId.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findLastOrderNumber(int localMaxOrderNumber, boolean increment) throws SQLException
   {
      if (findLastOrderNumber == null)
      {
         findLastOrderNumber = dbConnection.prepareStatement(FIND_LAST_ORDER_NUMBER);
      }
      else
      {
         findLastOrderNumber.clearParameters();
      }
      int value = increment ? 1 : 0;
      findLastOrderNumber.setInt(1, localMaxOrderNumber);
      findLastOrderNumber.setInt(2, value);
      return findLastOrderNumber.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildNodesCountByParentIdentifier(String parentIdentifier) throws SQLException
   {
      if (findNodesCountByParentId == null)
      {
         findNodesCountByParentId = dbConnection.prepareStatement(FIND_NODES_COUNT_BY_PARENTID);
      }
      else
      {
         findNodesCountByParentId.clearParameters();
      }

      findNodesCountByParentId.setString(1, parentIdentifier);
      return findNodesCountByParentId.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildPropertiesByParentIdentifier(String parentIdentifier) throws SQLException
   {
      if (findPropertiesByParentId == null)
      {
         findPropertiesByParentId = dbConnection.prepareStatement(FIND_PROPERTIES_BY_PARENTID);
      }
      else
      {
         findPropertiesByParentId.clearParameters();
      }

      findPropertiesByParentId.setString(1, parentIdentifier);
      return findPropertiesByParentId.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildPropertiesByParentIdentifierCQ(String parentCid, List<QPathEntryFilter> pattern)
      throws SQLException
   {
      if (pattern.isEmpty())
      {
         throw new SQLException("Pattern list is empty.");
      }
      else
      {
         if (findPropertiesByParentIdAndComplexPatternCQ == null)
         {
            findPropertiesByParentIdAndComplexPatternCQ = dbConnection.createStatement();
         }

         //create query from list
         StringBuilder query = new StringBuilder(FIND_PROPERTIES_BY_PARENTID_AND_PATTERN_CQ_TEMPLATE);
         query.append(" where I.I_CLASS=2 and I.PARENT_ID='");
         query.append(parentCid);
         query.append("' and ( ");
         appendPattern(query, pattern.get(0).getQPathEntry(), false);
         for (int i = 1; i < pattern.size(); i++)
         {
            query.append(" or ");
            appendPattern(query, pattern.get(i).getQPathEntry(), false);
         }
         query.append(" ) order by I.NAME");

         return findPropertiesByParentIdAndComplexPatternCQ.executeQuery(query.toString());

      }
   }

   /**
    * {@inheritDoc}
    */
   protected ResultSet findChildNodesByParentIdentifier(String parentCid, int fromOrderNum, int offset, int limit)
      throws SQLException
   {
      if (findNodesByParentIdLazilyCQ == null)
      {
         findNodesByParentIdLazilyCQ = dbConnection.prepareStatement(FIND_NODES_BY_PARENTID_LAZILY_CQ);
      }
      else
      {
         findNodesByParentIdLazilyCQ.clearParameters();
      }

      findNodesByParentIdLazilyCQ.setString(1, parentCid);
      if (containerConfig.useSequenceForOrderNumber)
      {
         findNodesByParentIdLazilyCQ.setInt(2, limit);
         findNodesByParentIdLazilyCQ.setInt(3, offset);
      }
      else
      {
         findNodesByParentIdLazilyCQ.setInt(2, fromOrderNum);
         findNodesByParentIdLazilyCQ.setInt(3, fromOrderNum + limit - 1);
      }

      return findNodesByParentIdLazilyCQ.executeQuery();
   }

   // -------- values processing ------------

   /**
    * {@inheritDoc}
    */
   @Override
   protected int addValueData(String cid, int orderNumber, InputStream stream, int streamLength, String storageDesc)
      throws SQLException, InvalidItemStateException, RepositoryException
   {
      if (insertValue == null)
      {
         insertValue = dbConnection.prepareStatement(INSERT_VALUE);
      }
      else
      {
         insertValue.clearParameters();
      }

      if (stream == null)
      {
         // [PN] store vd reference to external storage etc.
         insertValue.setNull(1, Types.BINARY);
         insertValue.setString(4, storageDesc);
      }
      else
      {
         insertValue.setBinaryStream(1, stream, streamLength);
         insertValue.setNull(4, Types.VARCHAR);
      }

      insertValue.setInt(2, orderNumber);
      insertValue.setString(3, cid);

      return executeUpdate(insertValue, TYPE_INSERT_VALUE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int deleteValueData(String cid) throws SQLException, InvalidItemStateException, RepositoryException
   {
      if (deleteValue == null)
      {
         deleteValue = dbConnection.prepareStatement(DELETE_VALUE);
      }
      else
      {
         deleteValue.clearParameters();
      }

      deleteValue.setString(1, cid);

      return executeUpdate(deleteValue, TYPE_DELETE_VALUE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findValuesByPropertyId(String cid) throws SQLException
   {
      if (findValuesByPropertyId == null)
      {
         findValuesByPropertyId = dbConnection.prepareStatement(FIND_VALUES_BY_PROPERTYID);
      }
      else
      {
         findValuesByPropertyId.clearParameters();
      }

      findValuesByPropertyId.setString(1, cid);
      return findValuesByPropertyId.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected int renameNode(NodeData data) throws SQLException, InvalidItemStateException, RepositoryException
   {
      if (renameNode == null)
      {
         renameNode = dbConnection.prepareStatement(RENAME_NODE);
      }
      else
      {
         renameNode.clearParameters();
      }

      renameNode.setString(1,
         data.getParentIdentifier() == null ? Constants.ROOT_PARENT_UUID : data.getParentIdentifier());
      renameNode.setString(2, data.getQPath().getName().getAsString());
      renameNode.setInt(3, data.getPersistedVersion());
      renameNode.setInt(4, data.getQPath().getIndex());
      renameNode.setInt(5, data.getOrderNumber());
      renameNode.setString(6, data.getIdentifier());

      return executeUpdate(renameNode, TYPE_RENAME_NODE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findChildPropertiesByParentIdentifierCQ(String parentIdentifier) throws SQLException
   {
      if (findPropertiesByParentIdCQ == null)
      {
         findPropertiesByParentIdCQ = dbConnection.prepareStatement(FIND_PROPERTIES_BY_PARENTID_CQ);
      }
      else
      {
         findPropertiesByParentIdCQ.clearParameters();
      }

      findPropertiesByParentIdCQ.setString(1, parentIdentifier);
      return findPropertiesByParentIdCQ.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findNodeMainPropertiesByParentIdentifierCQ(String parentIdentifier) throws SQLException
   {
      if (findNodeMainPropertiesByParentIdentifierCQ == null)
      {
         findNodeMainPropertiesByParentIdentifierCQ =
            dbConnection.prepareStatement(FIND_NODE_MAIN_PROPERTIES_BY_PARENTID_CQ);
      }
      else
      {
         findNodeMainPropertiesByParentIdentifierCQ.clearParameters();
      }

      findNodeMainPropertiesByParentIdentifierCQ.setString(1, parentIdentifier);
      return findNodeMainPropertiesByParentIdentifierCQ.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findItemQPathByIdentifierCQ(String identifier) throws SQLException
   {
      if (findItemQPathByIdentifierCQ == null)
      {
         findItemQPathByIdentifierCQ = dbConnection.prepareStatement(FIND_ITEM_QPATH_BY_ID_CQ);
      }
      else
      {
         findItemQPathByIdentifierCQ.clearParameters();
      }

      findItemQPathByIdentifierCQ.setString(1, identifier);
      return findItemQPathByIdentifierCQ.executeQuery();
   }

   protected int deleteValueDataByOrderNum(String id, int orderNum) throws SQLException, InvalidItemStateException,
      RepositoryException
   {
      if (deleteValueDataByOrderNum == null)
      {
         deleteValueDataByOrderNum = dbConnection.prepareStatement(DELETE_VALUE_BY_ORDER_NUM);
      }
      else
      {
         deleteValueDataByOrderNum.clearParameters();
      }

      deleteValueDataByOrderNum.setString(1, id);
      deleteValueDataByOrderNum.setInt(2, orderNum);

      return executeUpdate(deleteValueDataByOrderNum, TYPE_DELETE_VALUE_BY_ORDER_NUM);
   }

   protected ResultSet findPropertyById(String id) throws SQLException
   {
      if (findPropertyById == null)
      {
         findPropertyById = dbConnection.prepareStatement(FIND_PROPERTY_BY_ID);
      }
      else
      {
         findPropertyById.clearParameters();
      }

      findPropertyById.setString(1, id);
      return findPropertyById.executeQuery();
   }

   protected int updateValueData(String cid, int orderNumber, InputStream stream, int streamLength, String storageDesc)
      throws SQLException, InvalidItemStateException, RepositoryException
   {

      if (updateValue == null)
      {
         updateValue = dbConnection.prepareStatement(UPDATE_VALUE);
      }
      else
      {
         updateValue.clearParameters();
      }

      if (stream == null)
      {
         // [PN] store vd reference to external storage etc.
         updateValue.setNull(1, Types.BINARY);
         updateValue.setString(2, storageDesc);
      }
      else
      {
         updateValue.setBinaryStream(1, stream, streamLength);
         updateValue.setNull(2, Types.VARCHAR);
      }

      updateValue.setString(3, cid);
      updateValue.setInt(4, orderNumber);

      return executeUpdate(updateValue, TYPE_UPDATE_VALUE);
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findNodesAndProperties(String lastNodeId, int offset, int limit) throws SQLException
   {
      if (findNodesAndProperties == null)
      {
         findNodesAndProperties = dbConnection.prepareStatement(FIND_NODES_AND_PROPERTIES);
      }
      else
      {
         findNodesAndProperties.clearParameters();
      }

      findNodesAndProperties.setString(1, lastNodeId);
      findNodesAndProperties.setInt(2, limit);
      findNodesAndProperties.setInt(3, offset);

      return findNodesAndProperties.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findACLHolders() throws SQLException
   {
      if (findACLHolders == null)
      {
         findACLHolders = dbConnection.prepareStatement(FIND_ACL_HOLDERS);
      }

      return findACLHolders.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   protected void deleteLockProperties() throws SQLException, InvalidItemStateException, RepositoryException
   {
      addChange(TYPE_DELETE_LOCK);
      PreparedStatement removeValuesStatement = null;
      PreparedStatement removeItemsStatement = null;

      try
      {
         removeValuesStatement =
            dbConnection.prepareStatement("DELETE FROM " + JCR_VALUE + " WHERE PROPERTY_ID IN" + " (SELECT ID FROM "
               + JCR_ITEM + " WHERE I_CLASS = 2 AND NAME = '[http://www.jcp.org/jcr/1.0]lockIsDeep' OR"
               + " NAME = '[http://www.jcp.org/jcr/1.0]lockOwner')");

         removeItemsStatement =
            dbConnection.prepareStatement("DELETE FROM " + JCR_ITEM + " WHERE I_CLASS = 2 AND "
               + " NAME = '[http://www.jcp.org/jcr/1.0]lockIsDeep' OR"
               + " NAME = '[http://www.jcp.org/jcr/1.0]lockOwner'");

         removeValuesStatement.executeUpdate();
         removeItemsStatement.executeUpdate();
      }
      finally
      {
         if (removeValuesStatement != null)
         {
            try
            {
               removeValuesStatement.close();
            }
            catch (SQLException e)
            {
               LOG.error("Can't close statement", e);
            }
         }

         if (removeItemsStatement != null)
         {
            try
            {
               removeItemsStatement.close();
            }
            catch (SQLException e)
            {
               LOG.error("Can't close statement", e);
            }
         }
      }
   }

   /**
    * {@inheritDoc}
    */
   @Override
   protected ResultSet findNodesCount() throws SQLException
   {
      if (findNodesCount == null)
      {
         findNodesCount = dbConnection.prepareStatement(FIND_NODES_COUNT);
      }
      return findNodesCount.executeQuery();
   }

   /** 
    * {@inheritDoc} 
    */
   protected ResultSet findMaxPropertyVersion(String parentId, String name, int index) throws SQLException
   {
      if (findMaxPropertyVersions == null)
      {
         findMaxPropertyVersions = dbConnection.prepareStatement(FIND_MAX_PROPERTY_VERSIONS);
      }

      findMaxPropertyVersions.setString(1, getInternalId(parentId));
      findMaxPropertyVersions.setString(2, name);
      findMaxPropertyVersions.setInt(3, index);

      return findMaxPropertyVersions.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   protected ResultSet findWorkspaceDataSize() throws SQLException
   {
      if (findWorkspaceDataSize == null)
      {
         findWorkspaceDataSize = dbConnection.prepareStatement(FIND_WORKSPACE_DATA_SIZE);
      }

      return findWorkspaceDataSize.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   protected ResultSet findWorkspacePropertiesOnValueStorage() throws SQLException
   {
      if (findWorkspacePropertiesOnValueStorage == null)
      {
         findWorkspacePropertiesOnValueStorage =
            dbConnection.prepareStatement(FIND_WORKSPACE_PROPERTIES_ON_VALUE_STORAGE);
      }

      return findWorkspacePropertiesOnValueStorage.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   protected ResultSet findNodeDataSize(String parentId) throws SQLException
   {
      if (findNodeDataSize == null)
      {
         findNodeDataSize = dbConnection.prepareStatement(FIND_NODE_DATA_SIZE);
      }

      findNodeDataSize.setString(1, parentId);

      return findNodeDataSize.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   protected ResultSet findNodePropertiesOnValueStorage(String parentId) throws SQLException
   {
      if (findNodePropertiesOnValueStorage == null)
      {
         findNodePropertiesOnValueStorage = dbConnection.prepareStatement(FIND_NODE_PROPERTIES_ON_VALUE_STORAGE);
      }

      findNodePropertiesOnValueStorage.setString(1, parentId);

      return findNodePropertiesOnValueStorage.executeQuery();
   }

   /**
    * {@inheritDoc}
    */
   protected ResultSet findValueStorageDescAndSize(String cid) throws SQLException
   {
      if (findValueStorageDescAndSize == null)
      {
         findValueStorageDescAndSize = dbConnection.prepareStatement(FIND_VALUE_STORAGE_DESC_AND_SIZE);
      }

      findValueStorageDescAndSize.setString(1, cid);

      return findValueStorageDescAndSize.executeQuery();
   }
}
