/*
 * Isomorphic SmartClient
 * Version v9.0_2013-07-03 (2013-07-03)
 * Copyright(c) 1998 and beyond Isomorphic Software, Inc. All rights reserved.
 * "SmartClient" is a trademark of Isomorphic Software, Inc.
 *
 * licensing@smartclient.com
 *
 * http://smartclient.com/license
 */




//> @groupDef gwtFloatVsDouble
// In GWT code, you should generally use Java Doubles rather than Java Float values.
// <p>
// In the current implementation of GWT, Float and Double types are both represented as
// JavaScript Number, so there is no storage or performance advantage to using Float over
// Double, and double provides higher precision.
// <p>
// In addition, because GWT uses true Java Floats <i>in development mode</i> but uses higher
// precision JavaScript Number values in compiled mode, math operations on Float can differ
// between development mode vs compiled mode.
// <p>
// The SmartGWT field type "float" is represented as a JavaScript Number, the same storage GWT
// uses for Doubles, so in any code that accesses or manipulates values stored in a field of
// type "float", use Record.getAttributeAsDouble(), DoubleItem.getValueAsDouble(), and similar
// APIs to avoid being tripped up by GWT's different behavior in development mode.
//
// @title Float vs Double
// @visibility sgwt 
//<

//> @class DoubleItem
// TextForm item for managing a text field that displays a decimal value.
// @visibility external
//<

//> @method DoubleItem.getValueAsDouble()
// Return the value tracked by this form item.
// @return (Double) the value of the form, converted to a double.
// @visibility external
//<

//> @class FloatItem


// <p>
// A TextItem for managing a text field that displays a floating point value.  FloatItem is the
// default FormItem if the +link{formItem.type} is "float".
// <p>
// FloatItem displays its value according to the +link{formItem.decimalPrecision} and
// +link{formItem.decimalPad} properties of the FormItem.  While the value is being edited,
// the item will display the value with its original precision and without extra zero-padding.
// </p>
//
// @group gwtFloatVsDouble
// @visibility external
//<
isc.ClassFactory.defineClass("FloatItem", "TextItem");

isc.FloatItem.addProperties({
// A boolean flag to store whether the item is currently displaying an editor.  This is
// used in the override of mapValueToDisplay() to display the full floating-point value while
// in "editor mode" and a formatted version of that value (where the format is specified by
// decimalPrecision and decimalPad) outside of "editor mode".
_inEditorMode: false
});

isc.FloatItem.addMethods({

    // Note: similar code appears in StaticTextItem
    mapValueToDisplay : function (value) {
        if (!this._inEditorMode) {
            var floatValue = null;
            if (isc.isA.String(value)) {
                var parsedValue = window.parseFloat(value);
                if (!window.isNaN(parsedValue) && parsedValue == value) {
                    floatValue = parsedValue;
                }
            } else if (isc.isA.Number(value)) {
                floatValue = value;
            }
            if (floatValue != null) {
                if (this.decimalPrecision != null || this.decimalPad != null) {
                    return isc.Canvas.getFloatValueAsString(floatValue,
                        this.decimalPrecision, this.decimalPad);
                } else if (this.precision != null) {
                    return isc.Canvas.getNumberValueAsString(floatValue, 
                        this.precision, "float");
                }
            }
        }
        return this.Super("mapValueToDisplay", arguments);
    },

    handleEditorEnter : function () {
        this.Super("handleEditorEnter", arguments);

        this._inEditorMode = true;
        var value = this.getValue(),
            displayValue = this.mapValueToDisplay(value);
        this.setElementValue(displayValue, value);
    },

    handleEditorExit : function () {
        this.Super("handleEditorExit", arguments);

        this._inEditorMode = false;
        var value = this.getValue(),
            displayValue = this.mapValueToDisplay(value);
        this.setElementValue(displayValue, value);
    }
});

