/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.editor.codemirror;

import java.util.List;

import org.exoplatform.gwtframework.commons.component.Handlers;
import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.commons.util.BrowserResolver;
import org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser;
import org.exoplatform.gwtframework.editor.api.EditorConfiguration;
import org.exoplatform.gwtframework.editor.api.GWTTextEditor;
import org.exoplatform.gwtframework.editor.api.Token;
import org.exoplatform.gwtframework.editor.event.EditorActivityEvent;
import org.exoplatform.gwtframework.editor.event.EditorAutoCompleteEvent;
import org.exoplatform.gwtframework.editor.event.EditorAutoCompleteHandler;
import org.exoplatform.gwtframework.editor.event.EditorContentChangedEvent;
import org.exoplatform.gwtframework.editor.event.EditorInitializedEvent;
import org.exoplatform.gwtframework.editor.event.EditorSaveContentEvent;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.TextArea;

/**
 * Created by The eXo Platform SAS .
 * 
 * @author <a href="mailto:gavrikvetal@gmail.com">Vitaliy Gulyy</a>
 * @version $
 */

public class CodeMirror extends GWTTextEditor implements EditorAutoCompleteHandler
{

   private TextArea textArea;

   private JavaScriptObject editorObject;
   
   private Handlers handlers;
   
   private FlowPanel lineHighlighter;

   private final Browser currentBrowser = BrowserResolver.CURRENT_BROWSER;
   
   private final Parser parser = Parser.getParser(configuration.getMimeType());

   private List<Token> tokenList;

   private boolean needUpdateTokenList = false;   // update token list only after the "initCallback" handler has been called  
   
   int lineHeight = 16;  // size of line in the CodeMirror in px
   int characterWidth = 8;  // width of character in the CodeMirror in px
   
   public CodeMirror(HandlerManager eventBus, EditorConfiguration configuration)
   {
      super(eventBus, configuration);
            
      textArea = new TextArea();
      DOM.setElementAttribute(textArea.getElement(), "id", getEditorId());
      editorPanel().add(textArea);

      lineHighlighter = getLineHighlighter();
      editorPanel().add(lineHighlighter);
      editorPanel().setWidgetPosition(lineHighlighter, 0, 5);   
         
      handlers = new Handlers(eventBus);
      handlers.addHandler(EditorAutoCompleteEvent.TYPE, this);
   }

   protected void onLoad()
   {
      super.onLoad();
      
      String id = getEditorId();
      String width = "";
      String height = "100%";
      boolean readOnly = configuration.isReadOnly();
      
      int continuousScanning = CodemirrorConfiguration.CONTINUOUS_SCANNING;
      boolean textWrapping = CodemirrorConfiguration.TEXT_WRAPPING;
      
      boolean lineNumbers = configuration.isLineNumbers();
      
      String styleURLs = CodemirrorConfiguration.getStyleUrl(configuration.getMimeType());
      String parserNames = CodemirrorConfiguration.getParserNames(configuration.getMimeType());
      
      String javaScriptDirectory = CodemirrorConfiguration.JS_DIRECTORY;
      
      editorObject = initCodeMirror(id, width, height, readOnly,
         continuousScanning, textWrapping, lineNumbers, styleURLs, parserNames, javaScriptDirectory);
   }
   
   private native JavaScriptObject initCodeMirror(String id, String w, String h, boolean readOnly,
      int cs, boolean tr, boolean lineNumbers, String styleURLs, String parserNames, String jsDirectory) /*-{
       var instance = this;       
       var changeFunction = function() {
          instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::onContentChanged()();
       };
        
       var cursorActivity = function() {
         instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::onCursorActivity()();
       };
  
       var initCallback = function() {
         instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::onInitialized()();
       };
    
       var editor = $wnd.CodeMirror.fromTextArea(id, {
           width: w,
           height: h,
           parserfile: eval(parserNames),
           stylesheet: eval(styleURLs),
           path: jsDirectory,
           continuousScanning: cs || false,
           undoDelay: 50,   // decrease delay before calling 'onChange' callback
           lineNumbers: lineNumbers,
           readOnly: readOnly,
           textWrapping: tr,
           tabMode: "spaces",
           content: "",     // to fix bug with blocked deleting function of CodeMirror just after opening file [WBT-223]
           onChange: changeFunction,
//           saveFunction: saveFunction,
           reindentOnLoad: true,   // to fix problem with getting token list after the loading content
           cursorActivity: cursorActivity,
           initCallback: initCallback,
           autoMatchParens: true,
           
           // Take the token before the cursor. If it contains a character in '()[]{}', search for the matching paren/brace/bracket, and
           // highlight them in green for a moment, or red if no proper match was found.
           markParen: function(node, ok) {
              node.id = ok ? "parenCorrect" : "parenIncorrect";
           },
           unmarkParen: function(node) {
              node.id = null;
           }
         });
         
       // set hot keys click listener
       instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::setHotKeysClickListener(Lcom/google/gwt/core/client/JavaScriptObject;)(editor);
       
       return editor;
    }-*/;

   private void onContentChanged()
   {
      this.needUpdateTokenList = true;
      eventBus.fireEvent(new EditorContentChangedEvent(getEditorId()));
   }

   private void onSaveContent()
   {
      eventBus.fireEvent(new EditorSaveContentEvent(getEditorId()));
   }

   private void onCursorActivity()
   {
      // highlight current line
      highlightLine(0);

      fireEditorActivityEvent(getEditorId(), getCursorRow(), getCursorCol());
   };

   private void fireEditorActivityEvent(String editorId, int cursorRow, int cursorCol) 
   {
      eventBus.fireEvent(new EditorActivityEvent(editorId, cursorRow, cursorCol));
   }
   
   private void onInitialized()
   {
      needUpdateTokenList = true;   // update token list after the document had been loaded and reindented
      eventBus.fireEvent(new EditorInitializedEvent(getEditorId()));
   }

   public native String getText()/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject; 
      return editor.getCode();
   }-*/;

   public native void setText(String text)/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      var instance = this;
      $wnd.setTimeout(function(){
         if (text === "") {
            text = "\n";     // fix error with initial cursor positiion and size (WBT-324)
         }
         editor.setCode(text);
         editor.focus();
      }, 200);
   }-*/;

   public native void undo()/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      editor.undo();
   }-*/;

   public native void redo()/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      editor.redo();
   }-*/;

   public native void formatSource()/*-{
      if (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getText()() != ' ') {
         var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
         editor.reindent();
      }
   }-*/;

   public native void replaceText(String text)/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      editor.replaceSelection(text);
   }-*/;

   public native void setLineNumbers(boolean showLineNumbers)/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return;

      editor.setLineNumbers(showLineNumbers);
      var configuration = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getConfiguration()();
      configuration.@org.exoplatform.gwtframework.editor.api.EditorConfiguration::setLineNumbers(Z)(showLineNumbers);                  
   }-*/;

   public native void setFocus()/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor != null)
      {
         $wnd.setTimeout(function(a, b){
            editor.focus();         
         }, 200);
      }
   }-*/;

   public native boolean hasRedoChanges()/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) {
        return false;
      }
      
      return editor.historySize().redo > 0;
   }-*/;

   public native boolean hasUndoChanges()/*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) {
        return false;
      }
      
      return editor.historySize().undo > 0;
   }-*/;

   public boolean canFormatSource()
   {
      return true;
   }

   public boolean canSetLineNumbers()
   {
      return true;
   }

   public boolean isReadOnly()
   {
      return false;
   }

   public void setReadOnly(boolean readOnly)
   {
   }
  
   /**
    * Set listeners of hot keys clicking. Listen "Ctrl+S" key pressing if hotKeyList is null. Listen Ctrl+Space in any case
    */
   private native void setHotKeysClickListener(JavaScriptObject editor) /*-{
      var instance = this;
      if (editor) {
         $wnd.setTimeout(function(){
            instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::addScrollAndResizeListener(Lorg/exoplatform/gwtframework/editor/codemirror/CodeMirror;)(instance);
  
            editor.grabKeys(
               function (event) {},
               
               function (keyCode, event){
                  // filter key pressed without ctrl or alt or event type not "keypress"                  
                  if (! (event.ctrlKey || event.altKey) || (event.type != "keydown")) 
                     return false;

                  // check if this is the Ctrl+Space
                  if ( keyCode == 32 && event.ctrlKey ) {
                     event.stop();
                     instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::ctrlSpaceClickHandler()();                                             
                     return true;
                  }
                  
                  // for Ctrl+l firstly called <event.keyCode = 76, event.charCode = 0>, then <event.keyCode = 0, event.charCode = 108>  
                  // for Ctrl+L firstly called <event.keyCode = 76, event.charCode = 0>, then <event.keyCode = 0, event.charCode = 76>            
                  var keyPressed = "";
                  event.ctrlKey ? keyPressed += "Ctrl+" : keyPressed += "";
                  event.altKey ? keyPressed += "Alt+" : keyPressed += "";
                  keyPressed += keyCode;              
                  
                  // find similar key ammong the hotKeyList
                  var configuration = instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getConfiguration()();
                  var hotKeyList = configuration.@org.exoplatform.gwtframework.editor.api.EditorConfiguration::getHotKeyList()();                  
                  
                  // listen Ctrl+S key pressing if hotKeyList is null
                  if (hotKeyList === null) { 
                     if (keyPressed == "Ctrl+" + "S".charCodeAt(0)) {
                        event.stop();
                        instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::onSaveContent()();
                        return true;                        
                     } else {
                        return false;
                     }                    
                  }
                  
                  for (var i = 0; i < hotKeyList.@java.util.List::size()(); i++) {
                    var currentHotKey = hotKeyList.@java.util.List::get(I)(i); 
                    if (currentHotKey == keyPressed) {
                      event.stop();

                      // fire EditorHotKeyCalledEvent
                      var editorHotKeyCalledEventInstance = @org.exoplatform.gwtframework.editor.event.EditorHotKeyCalledEvent::new(Ljava/lang/String;)(
                        currentHotKey
                      );
                      var eventBus = instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getEventBus()();
                      eventBus.@com.google.gwt.event.shared.HandlerManager::fireEvent(Lcom/google/gwt/event/shared/GwtEvent;)(editorHotKeyCalledEventInstance);

                      return true;                
                    }
                  }
                                    
                  return false;
               });
         }, 1500);
      }
   }-*/;
 
   /**
    * Set listener to call this.onCtrlSpaceClick() method just after the clicking on "Ctrl + Space" keys
    */   
   public native void ctrlSpaceClickHandler() /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return;   

      // calculate cursorOffsetY
      var cursorOffsetY = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getAbsoluteTop()() + this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCursorOffsetY(I)(0);
      
      // calculate cursorOffsetX
      var cursorCol = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCursorCol()();
      var cursorOffsetX = (cursorCol - 2) * this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::characterWidth + this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getAbsoluteLeft()() + 11;   // 8px per symbol 
     
      var configuration = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getConfiguration()();
      var hasLineNumbers = configuration.@org.exoplatform.gwtframework.editor.api.EditorConfiguration::isLineNumbers()();
      if (hasLineNumbers) {
         cursorOffsetX += 31;
      }
      
      var cursor = editor.cursorPosition(true);     
      var lineContent = editor.lineContent(cursor.line);

      // read mimeType
      var mimeType = configuration.@org.exoplatform.gwtframework.editor.api.EditorConfiguration::getMimeType()();
           
      // fire editorAutoCompleteCalledEvent
      var editorAutoCompleteCalledEventInstance = @org.exoplatform.gwtframework.editor.event.EditorAutoCompleteCalledEvent::new(Ljava/lang/String;Ljava/lang/String;IILjava/lang/String;IILjava/util/List;Ljava/lang/String;)(
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getEditorId()(),
         mimeType,
         cursorOffsetX,
         cursorOffsetY,
         lineContent,
         cursorCol,
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCursorRow()(),
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getTokenList()(),
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCurrentLineMimeType()()         
      );
      
      var eventBus = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getEventBus()();
      eventBus.@com.google.gwt.event.shared.HandlerManager::fireEvent(Lcom/google/gwt/event/shared/GwtEvent;)(editorAutoCompleteCalledEventInstance); 
   }-*/;

   public int getAbsoluteTop() 
   {
      return editorPanel().getAbsoluteTop();
   }
   
   public int getAbsoluteLeft() 
   {
      return editorPanel().getAbsoluteLeft();
   }   
   
   /**
    * returns line position number of vertical scroll bar in the body with text in the CodeMirror iframe
    * @param currentLine if equals o or null, then will get current line position
    */
   public native int getCursorOffsetY(int currentLine) /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return;   

      var verticalScrollBarPosition = 0;
   
      switch (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::currentBrowser) {          
         case @org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser::IE :
            if (editor.frame && editor.frame.contentWindow) {
               verticalScrollBarPosition = editor.frame.contentWindow.document.documentElement.scrollTop;
            }
            break;
            
         default:
            if (editor.editor && editor.editor.container) {
               verticalScrollBarPosition = editor.editor.container.scrollTop;
            }
      }
      
      // calculate cursorOffsetY
      var cursorOffsetY = 0;

      if (! currentLine) {
         currentLine = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCursorRow()();
      }      

      cursorOffsetY = (currentLine - 1) * this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::lineHeight;         
      cursorOffsetY -= verticalScrollBarPosition;
      
      return cursorOffsetY;
   }-*/;

   public void onEditorAutoComplete(EditorAutoCompleteEvent event)
   {
      // test if this file tab is existed
      if (this.editorPanel().getOffsetHeight() == 0)
      {
         this.onUnload();
         return;
      }

      // test if this is the same editor
      if (this.getEditorId().equals(event.getEditorId()))
      {
         this.replaceCurrentLine(event.getLineContent(), event.getCursorPosition() - 1);
      }
   }   
  
   private native void unsetKeyHandlers() /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor != null) {
         editor.ungrabKeys();
      }    
   }-*/;  

   public native String getCurrentLineContent() /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor != null) {
         return editor.lineContent(editor.cursorPosition(true).line);
      }
   }-*/;
   
   public native void replaceCurrentLine(String newText, int cursorPosition) /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor != null && newText) {
         var currentLine = editor.cursorPosition(true).line;
         editor.setLineContent(currentLine, newText);
         
         // set cursor at the cursor position  
         editor.selectLines(currentLine, cursorPosition);
      }
   }-*/;
   
   public native void insertAtCursor(String newText) /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor != null && newText) {
         var cursor = editor.cursorPosition(true);
         var cursorPosition = cursor.character;
         var newCursorPosition = cursorPosition + newText.length;
         editor.insertIntoLine(cursor.line, cursorPosition, newText);
         
         // set cursor just after insertion
         editor.selectLines(cursor.line, newCursorPosition);
      }
   }-*/;
   
   /*
    * remove listeners
    */
   protected void onUnload()
   {
      unsetKeyHandlers();
      handlers.removeHandlers();      
   }

   public boolean canDeleteCurrentLine()
   {
      return true;
   }

   public boolean canFindAndReplace()
   {
      return true;
   }

   public boolean canGoToLine()
   {
      return true;
   }

   public native void deleteCurrentLine() /*-{
     var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
     if (editor == null) return;

     var currentLineNumber = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCursorRow()();
     var currentLine = editor.nthLine(currentLineNumber);

     if (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::currentBrowser != @org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser::IE
           && 
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getLastLineNumber(Ljava/lang/String;)(editor.getCode()) == currentLineNumber) 
     {
       // clear current line
       this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::clearLastLine()();
     }
     else 
     {
       editor.removeLine(currentLine);
     }
        
     currentLineNumber = editor.lineNumber(currentLine);
      
     this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::goToLine(I)(currentLineNumber);
   }-*/;

   /**
    * returns line quantity in the content
    * @param content
    * @return
    */
   private native int getLastLineNumber(String content) /*-{
     if (! content) return 1;

     // test if content is not ended with line break
     if (content.charAt(content.length - 1) !== "\n") {
        return content.split("\n").length;
     }

     // in the Internet Explorer editor.setCode("\n") is displayed as 2 lines 
     if (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::currentBrowser == @org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser::IE) 
     {          
       return content.split("\n").length;
     }
       
     return content.split("\n").length - 1;                                                          
   }-*/;     

   
   /**
    * Correct clear the last line of content that the line break is being remained
    */
   private native void clearLastLine() /*-{       
     var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
     if (editor == null) return;

     var content = editor.getCode();
     var lastLineHandler = editor.nthLine(this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getLastLineNumber(Ljava/lang/String;)(content));
      
     if (content.charAt(content.length - 1) == "\n") {
       editor.setLineContent(lastLineHandler, "");
     } else {
       editor.setLineContent(lastLineHandler, "\n");          
     }
   }-*/;
      
   
   /**
    * Find text, select and return true
    * @param find text to find and select
    * @param caseSensitive <b>true</b> means case sensitive search 
    * @return <b>true</b> if something is found, or <b>false</b>  if the end of document is reached. 
    */   
   public native boolean findAndSelect(String find, boolean caseSensitive) /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject; 
      if (editor == null) return;

      var isFound = false;
      var cursor = editor.getSearchCursor(find, true, !caseSensitive);  // getSearchCursor(string, atCursor, caseFold) -> cursor
      if (isFound = cursor.findNext()) {
        cursor.select();        
      }
      
      return isFound; 
   }-*/;

   /**
    * Replace selected text on "replace" parameter if it equals to "find" parameter
    * @param find text to find 
    * @param replace text to replace
    * @param caseSensitive <b>true</b> means case sensitive search 
    * @return 
    */
   public native void replaceFoundedText(String find, String replace, boolean caseSensitive) /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return;
      var selected = editor.selection();

      if (! caseSensitive) {
        selected = selected.toLowerCase();
        find = find.toLowerCase();
      }

      if (selected == find) {
        editor.replaceSelection(replace);
      }
     
      editor.focus(); 
   }-*/;
   
   public native void goToLine(int lineNumber) /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return;
      
      if (lineNumber && !isNaN(Number(lineNumber))) {
         editor.selectLines(editor.nthLine(lineNumber), 0);
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::highlightLine(I)(lineNumber);
         this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::fireEditorActivityEvent(Ljava/lang/String;II)(
            this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getEditorId()(),
            lineNumber,
            this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::getCursorCol()()
         );
      }     
   }-*/;
   
   private HandlerManager getEventBus()
   {
      return eventBus;
   }

   private EditorConfiguration getConfiguration()
   {
      return configuration;
   }
      
   public native int getCursorRow() /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return 1;

      switch (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::currentBrowser) {          
         case @org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser::IE :
            if (editor.editor.selectionSnapshot) {
               return editor.lineNumber(editor.editor.selectionSnapshot.from.node);
            } else {
               return 1;
            }

         default:
            var cursor = editor.cursorPosition(true);     
            return editor.lineNumber(cursor.line) || 1;
      }
   }-*/;

   public native int getCursorCol() /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor == null) return 1;

      switch (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::currentBrowser) {          
         case @org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser::IE :
            if (editor.editor.selectionSnapshot) {
               return editor.editor.selectionSnapshot.from.offset + 1;
            } else {
               return 1;
            }

         default:
            var cursor = editor.cursorPosition(true);     
            return (cursor.character + 1) || 1;
      }
   }-*/;

   public void setHotKeyList(List<String> hotKeyList)
   {
      configuration.setHotKeyList(hotKeyList);
   }
 
   private native void addScrollAndResizeListener(CodeMirror instance) /*-{
      var editor = instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      var highlightLine = function() {
         instance.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::highlightLine(I)(0);
      };

      // draw highligher at start           
      highlightLine();
      
      if (editor.win) {
         switch (this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::currentBrowser) {          
            case @org.exoplatform.gwtframework.commons.util.BrowserResolver.Browser::IE :
               if (editor.win.attachEvent) {
                 editor.win.attachEvent("onscroll", highlightLine);
                 editor.win.attachEvent("onresize", highlightLine);
               }
               break;
            default:
            if (editor.win.addEventListener) {
               editor.win.addEventHandler(editor.win, "scroll", highlightLine, true);
               editor.win.addEventHandler(editor.win, "resize", highlightLine, true);
            }
        }
     }
   }-*/;
   
   private FlowPanel getLineHighlighter()
   {
      FlowPanel highlighter = new FlowPanel();
      highlighter.setStyleName("CodeMirror-line-highlighter");
      return highlighter;
   }

   private void highlightLine(int lineNumber)
   {
      if (this.currentBrowser == Browser.IE)
      {
         fixCodeMirrorIframeTransparencyInIE();
      }

      editorPanel().setWidgetPosition(lineHighlighter, 0, 5 + getCursorOffsetY(lineNumber));
   }   
   
   /**
    * set codemirror iframe transparency for the IE
    */
   private native void fixCodeMirrorIframeTransparencyInIE() /*-{
      var editor = this.@org.exoplatform.gwtframework.editor.codemirror.CodeMirror::editorObject;
      if (editor !== null && editor.frame !== null && editor.frame.allowTransparency !== true) {
         editor.frame.allowTransparency = true;
      }
   }-*/;

   public boolean canCreateTokenList()
   {
      return true;
   }   
   
   public List<Token> getTokenList() 
   {
      if (this.needUpdateTokenList) 
      {
         this.tokenList = this.parser.getTokenList(editorObject);
         this.needUpdateTokenList = false;
      }
      
      return this.tokenList;
   }
   
   /**
    * 
    * @return mimeType of current line content  
    */
   private String getCurrentLineMimeType()
   {
      String genericMimeType = this.configuration.getMimeType();
      if (genericMimeType.equals(MimeType.GOOGLE_GADGET) || genericMimeType.equals(MimeType.TEXT_HTML)
         || genericMimeType.equals(MimeType.GROOVY_TEMPLATE))
      {
         String mimeType = Parser.getLineMimeType(getCursorRow(), getTokenList());

         if (mimeType != null)
         {
            return mimeType;
         }
      }

      return genericMimeType;
   }
}
