/*
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.gwtframework.editor.codemirror;

import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.editor.api.Token;
import org.exoplatform.gwtframework.editor.api.Token.TokenType;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * @author <a href="mailto:dmitry.ndp@gmail.com">Dmytro Nochevnov</a>
 * @version $Id: $
 *
 */
public class GroovyParser extends Parser
{

   private String lastNodeContent;

   private String lastNodeType;

   private boolean inBrackets;

   private String currentJavaType;

   private TokenType testTokenType;

   @Override
   public void init()
   {
      super.init();

      inBrackets = false;
      currentJavaType = lastNodeType = lastNodeContent = null;
      testTokenType = null;
   }

   @Override
   Token parseLine(JavaScriptObject node, int lineNumber, Token currentToken, boolean hasParentParser)
   {
      // interrupt at the end of the line or content
      if ((node == null) || getName(node).equals("BR"))
         return currentToken;

      String nodeContent = getContent(node).trim(); // returns text without ended space " " in the text
      String nodeType = getType(node);

      // recognize "(" and ")"
      if (isOpenBrackets(nodeType, nodeContent))
      {
         inBrackets = true;
      }
      else if (isCloseBrackets(nodeType, nodeContent))
      {
         inBrackets = false;
      }

      if (isGroovyVariable(nodeType) && !inBrackets)
      {
         // recognize "class" token
         if (isJavaClassNode(lastNodeType, lastNodeContent))
         {
            Token newToken = new Token(nodeContent, TokenType.CLASS, lineNumber, MimeType.APPLICATION_GROOVY);
            currentToken.addSubToken(newToken);
            currentToken = newToken;
         }

         // filter "void function_name"
         else if (isJavaType(lastNodeType) && "void".equals(lastNodeContent))
         {
            currentToken.addSubToken(new Token(nodeContent, TokenType.METHOD, lineNumber, MimeType.APPLICATION_GROOVY));
         }

         // filter variable/method definition "def var = "
         else if (isGroovyDef(lastNodeType, lastNodeContent))
         {
            currentToken
               .addSubToken(new Token(nodeContent, TokenType.PROPERTY, lineNumber, MimeType.APPLICATION_GROOVY));
         }

         // filter variable/method declaration like "String hello(..." or "boolean hello"
         else if (isGroovyVariable(lastNodeType) || isJavaType(lastNodeType))
         {
            currentToken.addSubToken(new Token(nodeContent + " : " + lastNodeContent, TokenType.PROPERTY, lineNumber,
               MimeType.APPLICATION_GROOVY));
            currentJavaType = lastNodeContent;
         }

         // filter variables like this "String a, b, c;" 
         else if (currentJavaType != null && isComma(lastNodeType, lastNodeContent))
         {
            currentToken.addSubToken(new Token(nodeContent + " : " + currentJavaType, TokenType.PROPERTY, lineNumber,
               MimeType.APPLICATION_GROOVY));
         }

         testTokenType = TokenType.METHOD;
      }

      // to discard the replacement of property on method in code like this "a = 1; if (...) {"
      else if (isEqualSign(nodeType, nodeContent) || isJavaKeyword(nodeType))
      {
         testTokenType = null;
      }

      // replace last sub token type from variable to method
      else if (isGroovyMethodOpenBraces(lastNodeType, lastNodeContent, nodeType, nodeContent))
      {
         currentToken.updateTypeOfLastSubToken(TokenType.METHOD); // update last token
      }

      lastNodeContent = nodeContent;
      lastNodeType = nodeType;

      if (hasParentParser)
      {
         return currentToken; // return current token to parent parser
      }

      return parseLine(getNext(node), lineNumber, currentToken, false);
   }

   /**
    * Recognize java keywords 'if', 'switch', 'while', 'else', 'do', 'try', 'finally', 'break', 'continue', 'extends', 'implements', 'import', 'new', 'package', 'return', 'super', 'this', 'throws'
    * @param nodeType
    * @return
    */
   private boolean isJavaKeyword(String nodeType)
   {
      return (nodeType != null) && nodeType.equals("javaKeyword");
   }

   /**
    * Recognize sign ","
    * @param nodeType
    * @param nodeContent
    * @return
    */
   private boolean isComma(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("groovyPunctuation")
         && nodeContent.equals(",");
   }

   private boolean isJavaType(String nodeType)
   {
      return (nodeType != null) && nodeType.equals("javaType");
   }

   /**
    * Recognize "{" in the pattern "method_name(... ){"
    * @param lastNodeType
    * @param lastNodeContent
    * @param nodeType
    * @param nodeContent    
    * @return true if there is open braces of method definition
    */
   private boolean isGroovyMethodOpenBraces(String lastNodeType, String lastNodeContent, String nodeType,
      String nodeContent)
   {
      return (lastNodeType != null) && (lastNodeContent != null)
         && lastNodeType.equals("groovyPunctuation")
         && lastNodeContent.equals(")") // filter ")"
         && (nodeType != null) && (nodeContent != null) && nodeType.equals("groovyPunctuation")
         && nodeContent.equals("{") && TokenType.METHOD.equals(testTokenType); // filter "a = 1; if (...) {"
   }

   private boolean isGroovyVariable(String nodeType)
   {
      return (nodeType != null) && nodeType.equals("groovyVariable");
   }

   /**
    * Recognize "=" operation
    * @param nodeType
    * @param nodeContent
    * @return
    */
   private boolean isEqualSign(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("groovyOperator")
         && nodeContent.equals("=");
   }

   /**
    * Recognize open brackets "(" 
    * @param nodeType
    * @param nodeContent
    * @return
    */
   private boolean isOpenBrackets(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("groovyPunctuation")
         && nodeContent.equals("(");
   }

   /**
    * Recognize open brackets "(" 
    * @param nodeType
    * @param nodeContent
    * @return
    */
   private boolean isCloseBrackets(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("groovyPunctuation")
         && nodeContent.equals(")");
   }

   private boolean isJavaClassNode(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("javaType") && nodeContent.equals("class");
   }

   /**
    * Recognize "def varname"
    * @param nodeType
    * @param nodeContent
    * @return
    */
   private boolean isGroovyDef(String nodeType, String nodeContent)
   {
      return (nodeType != null) && (nodeContent != null) && nodeType.equals("groovyKeyword")
         && nodeContent.equals("def");
   }

}