/*
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.gwtframework.editor.codemirror;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.editor.api.Token;
import org.exoplatform.gwtframework.editor.api.Token.TokenType;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * @author <a href="mailto:dmitry.ndp@gmail.com">Dmytro Nochevnov</a>
 * @version $Id: $
 *
 */
public abstract class Parser
{

   private static HashMap<String, Parser> factory = new HashMap<String, Parser>();

   static
   {
      factory.put(MimeType.APPLICATION_JAVASCRIPT, new JavaScriptParser());
      factory.put(MimeType.GOOGLE_GADGET, new GoogleGadgetParser());
      factory.put(MimeType.TEXT_CSS, new CssParser());
      factory.put(MimeType.TEXT_HTML, new HtmlParser());
      factory.put(MimeType.TEXT_XML, new XmlParser());
      factory.put(MimeType.APPLICATION_GROOVY, new GroovyParser());
      factory.put(MimeType.GROOVY_TEMPLATE, new GroovyTemplateParser());
   }

   public static Parser getParser(String mimeType)
   {
      if (factory.containsKey(mimeType))
      {
         return factory.get(mimeType);
      }

      return new DefaultParser();
   }

   public List<Token> getTokenList(JavaScriptObject editor)
   {
      List<Token> emptyTokenList = new ArrayList<Token>();

      if (editor == null)
         return emptyTokenList;

      init();

      JavaScriptObject node;

      Token rootToken = new Token(null, TokenType.ROOT);
      Token currentToken = rootToken;
      currentToken.setSubTokenList(emptyTokenList);

      // fix error when editor.nthLine(1) = null
      // parse first line
      if (getFirstLine(editor) != null)
      {
         currentToken = parseLine(getFirstLine(editor), 1, currentToken, false);
      }

      // parse lines from second
      if (getLastLineNumber(editor) > 1)
      {
         for (int lineNumber = 2; lineNumber <= getLastLineNumber(editor); lineNumber++)
         {
            node = getNode(editor, lineNumber);

            // fix error when editor.nthLine(1) = null
            if (node == null)
            {
               continue;
            }

            currentToken = parseLine(getNext(node), lineNumber, currentToken, false);
         }
      }

      return rootToken.getSubTokenList();
   };

   private static String possibleMimeType;

   private static int nearestTokenLineNumber;

   /**
    * Recognize mimeType of line with lineNumber.  
    * @param targetLineNumber
    * @param tokenList
    * @return Returns mimeType of closes token.START_DELIMITER with token.lineNumber <= lineNumber. If there is no such START_DELIMITER in the tokenList, then returns mimeType of last token.FINISH_DELIMITER with token.lineNumber > lineNumber, or MimeType of firstToken, or null if TokenList is empty.
    */
   public static String getLineMimeType(int targetLineNumber, List<Token> tokenList)
   {
      if (tokenList == null || tokenList.size() == 0)
         return null;

      possibleMimeType = tokenList.get(0).getMimeType();
      nearestTokenLineNumber = tokenList.get(0).getLineNumber();

      for (Token token : tokenList)
      {
         if (token.getLineNumber() > targetLineNumber)
            break;

         searchLineMimeType(targetLineNumber, token);
      }

      return possibleMimeType;
   }

   private static void searchLineMimeType(int targetLineNumber, Token currentToken)
   {
      // search appropriate token among the sub token
      List<Token> subTokenList = currentToken.getSubTokenList();

      if (subTokenList != null && subTokenList.size() != 0)
      {
         for (Token token : subTokenList)
         {
            if (token.getLineNumber() > targetLineNumber)
               break;

            searchLineMimeType(targetLineNumber, token);
         }
      }

      int currentTokenLineNumber = currentToken.getLineNumber();
      if ((currentTokenLineNumber <= targetLineNumber) && (currentTokenLineNumber >= nearestTokenLineNumber) // taking in mind the last token among them in the line
      )
      {
         nearestTokenLineNumber = currentTokenLineNumber;
         possibleMimeType = currentToken.getMimeType();
      }
   }

   /** 
    * @param node
    * @param lineNumber
    * @param tokenList
    * @param hasParentParser indicates is parser calles by another parser, e.g. JavaScriptParser is called by HtmlParser
    * @return token list with tokens gathered from node chains from start node to <br> node
    */
   Token parseLine(JavaScriptObject node, int lineNumber, Token currentToken, boolean hasParentParser)
   {
      return currentToken;
   }

   public void init()
   {
   }

   /** 
    * @param editor
    * @return get first line of document opened in editor
    */
   native JavaScriptObject getFirstLine(JavaScriptObject editor) /*-{     
      if (editor.editor && editor.editor.container)
      {
         return editor.editor.container.firstChild;
      }
         
      return null;
   }-*/;

   native JavaScriptObject getNode(JavaScriptObject editor, int lineNumber) /*-{
      return editor.nthLine(lineNumber);
   }-*/;

   /** 
    * @param editor - CodeMirror editor object
    * @return last line number of document opened in editor
    */
   native int getLastLineNumber(JavaScriptObject editor) /*-{
      return editor.lineNumber(editor.lastLine()); 
   }-*/;

   /**
    * @param node from CodeMirror container
    * @return
    */
   native String getName(JavaScriptObject node) /*-{
      return node.nodeName || "";
   }-*/;

   /**
    * 
    * @param node from CodeMirror container
    * @return
    */
   native String getType(JavaScriptObject node) /*-{
      return node.className || "";
   }-*/;

   /**
    * @param node from CodeMirror container
    * @return node content defined in the innerHTML DOM property
    */
   native String getContent(JavaScriptObject node) /*-{
      if (node.innerHTML) 
      {
         return node.innerHTML.replace(/&nbsp;/g, " ");    // replace all occurrences of "&nbsp;" on " " to prevent code like this "%&gt;&nbsp;&nbsp;"
      } 
      else
      {
         return "";
      }
   }-*/;

   /**
    * @param node from CodeMirror container
    * @return next node defined in the nextSibling DOM property
    */
   native JavaScriptObject getNext(JavaScriptObject node) /*-{
      return node.nextSibling;
   }-*/;

}