/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
package org.exoplatform.gwtframework.editor.codemirror;

import java.util.HashMap;

import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.editor.api.Editor;
import org.exoplatform.gwtframework.editor.api.EditorConfiguration;
import org.exoplatform.gwtframework.editor.codemirror.autocompletehelper.GroovyAutocompleteHelper;
import org.exoplatform.gwtframework.editor.codemirror.autocompletehelper.GroovyTemplateAutocompleteHelper;
import org.exoplatform.gwtframework.editor.codemirror.autocompletehelper.HtmlAutocompleteHelper;
import org.exoplatform.gwtframework.editor.codemirror.autocompletehelper.JavaScriptAutocompleteHelper;
import org.exoplatform.gwtframework.editor.codemirror.codevalidator.GroovyCodeValidator;
import org.exoplatform.gwtframework.editor.codemirror.codevalidator.GroovyTemplateCodeValidator;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.shared.HandlerManager;

/**
 * Created by The eXo Platform SAS .
 * 
 * @author <a href="mailto:gavrikvetal@gmail.com">Vitaliy Gulyy</a>
 * @version $
 */

public class CodemirrorProducer implements Editor
{

   private static final String DESCRIPTION = "Code Editor";

   private String mimeType;

   private boolean isDefaultEditor;

   public static final boolean TEXT_WRAPPING = false;

   /**
    * 0 to turn off continuous scanning, or value like 100 in millisec as scanning period
    */
   public static final int CONTINUOUS_SCANNING = 0;

   static final String DEFAULT_MIMETYPE = MimeType.TEXT_PLAIN;

   public static final String PATH = GWT.getModuleBaseURL() + "codemirror/";

   public static final String JS_DIRECTORY = CodemirrorProducer.PATH + "js/";
   
   public static final HashMap<String, FileConfiguration> fileConfigurations = new HashMap<String, FileConfiguration>(){{          

      // text plain file
      put(MimeType.TEXT_PLAIN, new FileConfiguration(
            "['parsexml.js', 'parsecss.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/xmlcolors.css']" // code styles
         )
      );
      
      // xml file
      put(MimeType.TEXT_XML, new FileConfiguration(
            "['parsexml.js', 'tokenize.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/xmlcolors.css']", // code styles
            true, // can be outlined
            true, // can be autocompleted
            new XmlParser() // exoplatform code parser 
         )
      );
      
      put(MimeType.APPLICATION_XML, get(MimeType.TEXT_XML));
      
      // css file
      put(MimeType.TEXT_CSS, new FileConfiguration( 
            "['parsecss.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/csscolors.css']", // code styles
            false, // can be outlined
            true, // can be autocompleted
            new CssParser() // exoplatform code parser 
         )
      );
      
      // javascript file
      put(MimeType.TEXT_JAVASCRIPT, new FileConfiguration(
            "['tokenizejavascript.js', 'parsejavascript.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/jscolors.css']", // code styles
            true, // can be outlined
            true, // can be autocompleted
            new JavaScriptParser(), // exoplatform code parser
            new JavaScriptAutocompleteHelper()  // autocomplete helper
         )
      );
      put(MimeType.APPLICATION_JAVASCRIPT, get(MimeType.TEXT_JAVASCRIPT));
      put(MimeType.APPLICATION_X_JAVASCRIPT, get(MimeType.TEXT_JAVASCRIPT));
      
      // html file
      put(MimeType.TEXT_HTML, new FileConfiguration(
            "['parsexml.js', 'parsecss.js', 'tokenizejavascript.js', 'parsejavascript.js', 'parsehtmlmixed.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/xmlcolors.css', '" + CodemirrorProducer.PATH + "css/jscolors.css', '" + CodemirrorProducer.PATH + "css/csscolors.css']", // code styles
            true, // can be outlined
            true, // can be autocompleted
            new HtmlParser(), // exoplatform code parser
            new HtmlAutocompleteHelper()  // autocomplete helper            
         )
      );
      
      // Netwibes file
      put(MimeType.UWA_WIDGET, get(MimeType.TEXT_HTML));
      
      // groovy service file
      put(MimeType.GROOVY_SERVICE, new FileConfiguration(
            "['parsegroovy.js', 'tokenizegroovy.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/groovycolors.css']", // code styles
            true, // can be outlined
            true, // can be autocompleted
            new GroovyParser(), // exoplatform code parser
            new GroovyAutocompleteHelper(),   // autocomplete helper
            true,  // can be validated
            new GroovyCodeValidator()  // code validator
         )
      );
      
      // POGO file
      put(MimeType.APPLICATION_GROOVY, get(MimeType.GROOVY_SERVICE));
           
      // Chromattic Data Object file
      put(MimeType.CHROMATTIC_DATA_OBJECT, get(MimeType.GROOVY_SERVICE));

      // groovy template file
      put(MimeType.GROOVY_TEMPLATE, new FileConfiguration(
            "['parsegtmpl.js', 'parsecss.js', 'tokenizejavascript.js', 'parsejavascript.js', 'tokenizegroovy.js', 'parsegroovy.js', 'parsegtmplmixed.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/gtmplcolors.css', '" + CodemirrorProducer.PATH + "css/jscolors.css', '" + CodemirrorProducer.PATH + "css/csscolors.css', '" + CodemirrorProducer.PATH + "css/groovycolors.css']", // code styles
            true, // can be outlined
            true, // can be autocompleted
            new GroovyTemplateParser(), // exoplatform code parser 
            new GroovyTemplateAutocompleteHelper(),   // autocomplete helper            
            true,  // can be validated
            new GroovyTemplateCodeValidator()  // code validator
         )
      );      
      
      // google gadget file
      put(MimeType.GOOGLE_GADGET, new FileConfiguration(
            "['parsegadgetxml.js', 'parsecss.js', 'tokenizejavascript.js', 'parsejavascript.js', 'parsehtmlmixed.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/xmlcolors.css', '" + CodemirrorProducer.PATH + "css/jscolors.css', '" + CodemirrorProducer.PATH + "css/csscolors.css']", // code styles
            true, // can be outlined
            true, // can be autocompleted
            new GoogleGadgetParser() // exoplatform code parser 
         )
      );
      
      // JSON file
      put(MimeType.APPLICATION_JSON, new FileConfiguration(
            "['tokenizejavascript.js', 'parsejavascript.js']",  // generic code parsers
            "['" + CodemirrorProducer.PATH + "css/jscolors.css']", // code styles
            true, // can be outlined
            false, // can be autocompleted
            new JSONParser() // exoplatform code parser 
         )
      );
      
   }};
   
   
   
   public CodemirrorProducer(String mimeType)
   {
      this.mimeType = mimeType;
   }

   public String getDescription()
   {
      return DESCRIPTION;
   }

   public CodeMirror createTextEditor(HandlerManager eventBus, EditorConfiguration editorConfiguration)
   {
      editorConfiguration.setMimeType(mimeType);

      // get file configuration
      FileConfiguration fileConfiguration = fileConfigurations.get(mimeType);
      if (fileConfigurations == null)
      {
         fileConfiguration = fileConfigurations.get(MimeType.TEXT_PLAIN);
      }
      
      return new CodeMirror(eventBus, editorConfiguration, fileConfiguration);
   }

   public boolean isDefaultEditor()
   {
      return isDefaultEditor;
   }

   public void setIsDefaultEditor(boolean isDefaultEditor)
   {
      this.isDefaultEditor = isDefaultEditor;
   }

}
