/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */
package org.exoplatform.gwtframework.ui.client.loader;

import org.exoplatform.gwtframework.ui.client.util.ExoStyle;

import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Grid;
import com.google.gwt.user.client.ui.SimplePanel;

/**
 * Created by The eXo Platform SAS .
 * 
 * @author <a href="mailto:gavrikvetal@gmail.com">Vitaliy Gulyy</a>
 * @version $
 */

public class LoadingIndicator extends Composite
{

   private class ButtonPanel extends FlowPanel
   {

      public ButtonPanel()
      {
         sinkEvents(Event.ONMOUSEOVER | Event.ONMOUSEOUT | Event.ONMOUSEDOWN | Event.ONMOUSEUP);
      }

      @Override
      public void onBrowserEvent(Event event)
      {
         if (!enabled)
         {
            return;
         }

         switch (DOM.eventGetType(event))
         {
            case Event.ONMOUSEOVER :
               setStyleHover();
               break;

            case Event.ONMOUSEOUT :
               setStyleNormal();
               break;

            case Event.ONMOUSEDOWN :
               setStyleDown();
               break;

            case Event.ONMOUSEUP :
               setStyleHover();
               if (command != null)
               {
                  command.execute();
               }
               break;
         }
      }

   }

   public interface Style
   {

      static final String BUTTON_ICON = "exoComponentLoaderButtonIcon";

      static final String BUTTON_PANEL = "exoComponentLoaderButtonPanel";

      static final String LOADER = "exoComponentLoader";

      static final String LOADER_HIDEN = "exoComponentLoaderHidden";

      static final String LOADER_NOICON = "exoComponentLoaderNoIcon";

      static final String LOADER_PROGRESS_IMAGE = "exoComponentLoaderProgressImage";

      static final String LOADER_PROGRESS_PANEL = "exoComponentLoaderProgressPanel";

      static final String TABLE = "exoComponentLoaderTable";

      static final String TABLE_BUTTON = "exoComponentLoaderTableButton";

      static final String TABLE_DELIMITER = "exoComponentLoaderTableDelimiter";

      static final String TABLE_LEFT = "exoComponentLoaderTableLeft";

      static final String TABLE_NOICON = "exoComponentLoaderTableNoIcon";

      static final String TABLE_RIGHT = "exoComponentLoaderTableRight";

      static final String TABLE_STRETCH = "exoComponentLoaderTableStretch";

   }

   /**
    * Loader image.
    */
   private static final String LOADER_PROGRESS_IMAGE = "<img src=\"" + ExoStyle.getEXoStyleURL()
      + "component/loader/ajax-loader.gif" + "\" class=\"" + Style.LOADER_PROGRESS_IMAGE + "\" />";

   /**
    * Button panel for handling mouse events.
    */
   private ButtonPanel buttonPanel;

   /**
    * Command which will be executed when user click on loader's icon 
    */
   private Command command;

   /**
    * Enabled state
    */
   private boolean enabled = true;

   /**
    * Component's grid ( visual element )
    */
   private Grid grid;

   /**
    * Icon
    */
   private String icon;

   /**
    * Component's root element
    */
   private FlowPanel panel;

   /**
    * Create new instance of Loader without icon.
    * Using command will be unavailable until icon will not be specified. 
    */
   public LoadingIndicator()
   {
      this(null);
   }

   /**
    * Create new instance of Loader with specified icon.
    * Icon must be represented as HTML image and must be like "<img ... />" tag.
    * 
    * @param icon specified icon 
    */
   public LoadingIndicator(String icon)
   {
      this.icon = icon;

      panel = new FlowPanel();
      initWidget(panel);

      redraw();
   }

   /**
    * Get assigned command.
    * 
    * @return command which will be execute when user click on loader's icon
    */
   public Command getCommand()
   {
      return command;
   }

   /**
    * Get icon.
    * 
    * @return icon
    */
   public String getIcon()
   {
      return icon;
   }

   /**
    * Get width of loader.
    * 
    * @return width of loader
    */
   public int getLoaderWidth()
   {
      if (icon != null)
      {
         return 247;
      }
      else
      {
         return 222;
      }
   }

   /**
    * Get is enabled.
    * 
    * @return is enabled
    */
   public boolean isEnabled()
   {
      return enabled;
   }

   /**
    * Render component.
    */
   private void redraw()
   {
      if (grid != null)
      {
         panel.clear();
      }

      if (icon != null)
      {
         panel.setStyleName(Style.LOADER);
      }
      else
      {
         panel.setStyleName(Style.LOADER_NOICON);
      }

      grid = new Grid(1, icon != null ? 5 : 3);

      grid.setStyleName(icon != null ? Style.TABLE : Style.TABLE_NOICON);
      grid.setBorderWidth(0);
      grid.setCellPadding(0);
      grid.setCellSpacing(0);

      grid.setHTML(0, 0, ExoStyle.getBlankImage());

      SimplePanel progressImagePanel = new SimplePanel();
      progressImagePanel.setStyleName(Style.LOADER_PROGRESS_PANEL);
      DOM.setInnerHTML(progressImagePanel.getElement(), LOADER_PROGRESS_IMAGE);
      grid.setWidget(0, 1, progressImagePanel);

      if (icon != null)
      {
         grid.setHTML(0, 2, ExoStyle.getBlankImage());

         buttonPanel = new ButtonPanel();
         buttonPanel.setStyleName(Style.BUTTON_PANEL);

         refreshIcon();

         grid.setWidget(0, 3, buttonPanel);

         grid.setHTML(0, 4, ExoStyle.getBlankImage());
      }
      else
      {
         grid.setHTML(0, 2, ExoStyle.getBlankImage());
      }

      panel.add(grid);

      setStyleNormal();
   }

   /**
    * Refresh icon element.
    */
   private void refreshIcon()
   {
      if (buttonPanel == null)
      {
         return;
      }

      Element e = buttonPanel.getElement();
      e.setInnerHTML(icon);

      Element imageElement = DOM.getChild(e, 0);
      DOM.setElementAttribute(imageElement, "class", Style.BUTTON_ICON);
   }

   /**
    * Set command which will be executed when user click on loader's icon.
    * 
    * @param command command which will be executed when user click on loader's icon
    */
   public void setCommand(Command command)
   {
      this.command = command;
   }

   /**
    * Set is enabled.
    * 
    * @param enabled enabled state
    */
   public void setEnabled(boolean enabled)
   {
      this.enabled = enabled;
   }

   /**
    * Set icon. Icon must be represented as HTML image. Image must be prepared like "<img ... />" tag. 
    * If there are no needs to show the icon, set icon parameter in null.
    * 
    * @param icon new icon
    */
   public void setIcon(String icon)
   {
      if (this.icon != null && icon != null)
      {
         this.icon = icon;
         refreshIcon();
      }
      else
      {
         this.icon = icon;
         redraw();
      }

   }

   /**
    * Refresh styles on mouse down. 
    */
   private void setStyleDown()
   {
      if (icon != null)
      {
         grid.getCellFormatter().setStyleName(0, 0, Style.TABLE_LEFT + "Down");
         grid.getCellFormatter().setStyleName(0, 1, Style.TABLE_STRETCH + "Down");
         grid.getCellFormatter().setStyleName(0, 2, Style.TABLE_DELIMITER + "Down");
         grid.getCellFormatter().setStyleName(0, 3, Style.TABLE_BUTTON + "Down");
         grid.getCellFormatter().setStyleName(0, 4, Style.TABLE_RIGHT + "Down");
      }
      else
      {
         grid.getCellFormatter().setStyleName(0, 0, Style.TABLE_LEFT + "Down");
         grid.getCellFormatter().setStyleName(0, 1, Style.TABLE_STRETCH + "Down");
         grid.getCellFormatter().setStyleName(0, 2, Style.TABLE_RIGHT + "Down");
      }
   }

   /**
    * Refresh styles on mouse hovered.
    */
   private void setStyleHover()
   {
      if (icon != null)
      {
         grid.getCellFormatter().setStyleName(0, 0, Style.TABLE_LEFT + "Over");
         grid.getCellFormatter().setStyleName(0, 1, Style.TABLE_STRETCH + "Over");
         grid.getCellFormatter().setStyleName(0, 2, Style.TABLE_DELIMITER + "Over");
         grid.getCellFormatter().setStyleName(0, 3, Style.TABLE_BUTTON + "Over");
         grid.getCellFormatter().setStyleName(0, 4, Style.TABLE_RIGHT + "Over");
      }
      else
      {
         grid.getCellFormatter().setStyleName(0, 0, Style.TABLE_LEFT + "Over");
         grid.getCellFormatter().setStyleName(0, 1, Style.TABLE_STRETCH + "Over");
         grid.getCellFormatter().setStyleName(0, 2, Style.TABLE_RIGHT + "Over");
      }
   }

   /**
    * Reset styles to default.
    */
   private void setStyleNormal()
   {
      if (icon != null)
      {
         grid.getCellFormatter().setStyleName(0, 0, Style.TABLE_LEFT);
         grid.getCellFormatter().setStyleName(0, 1, Style.TABLE_STRETCH);
         grid.getCellFormatter().setStyleName(0, 2, Style.TABLE_DELIMITER);
         grid.getCellFormatter().setStyleName(0, 3, Style.TABLE_BUTTON);
         grid.getCellFormatter().setStyleName(0, 4, Style.TABLE_RIGHT);
      }
      else
      {
         grid.getCellFormatter().setStyleName(0, 0, Style.TABLE_LEFT);
         grid.getCellFormatter().setStyleName(0, 1, Style.TABLE_STRETCH);
         grid.getCellFormatter().setStyleName(0, 2, Style.TABLE_RIGHT);
      }
   }

}
