/**
 * Copyright (C) 2009 eXo Platform SAS.
 * 
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.exoplatform.portal.webui.application;

import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * This class is a wrapper on the events generated by the portlet container and
 * to be propagated to other portlet instances deployed in the same portal page.
 * 
 * It also wraps a counter list that tells for each portlet windowId, how many
 * times the processEvent() method has been called. This is to avoid infinite
 * cycles where a processEvent() method will generate a new event which will
 * target the same portlet
 */
public class EventsWrapper
{

   protected static Log log = ExoLogger.getLogger("portal:EventsWrapper");

   private List<javax.portlet.Event> events;

   private List<CounterWrapper> counters = new ArrayList<CounterWrapper>();

   public static final int THRESHOLD = 4;

   public EventsWrapper(List<javax.portlet.Event> events)
   {
      this.events = events;
   }

   public List<javax.portlet.Event> getEvents()
   {
      return this.events;
   }

   public List<CounterWrapper> getCounters()
   {
      return counters;
   }

   public void increaseCounter(UIPortlet portlet)
   {
      for (CounterWrapper counter : counters)
      {
         if (portlet.getWindowId().equals(counter.portletId))
         {
            counter.counter++;
            return;
         }
      }
      counters.add(new CounterWrapper(portlet.getStorageId()));
   }

   public boolean isInvokedTooManyTimes(UIPortlet portlet)
   {
      for (CounterWrapper counter : counters)
      {
         if (portlet.getWindowId().equals(counter.portletId))
         {
            if (counter.counter + 1 > THRESHOLD)
            {
               log.info("Portlet " + portlet.getWindowId() + " has already been invokated " + THRESHOLD
                  + " times and will not be more to avoid infinite cycles");
               return true;
            }
            return false;
         }
      }
      return false;
   }

   public class CounterWrapper
   {
      public String portletId;

      public int counter = 0;

      public CounterWrapper(String portletId)
      {
         this.portletId = portletId;
      }
   }

}
