/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.commons.initializer;

import org.exoplatform.gwtframework.commons.exception.ExceptionThrownEvent;
import org.exoplatform.gwtframework.commons.initializer.event.ApplicationConfigurationReceivedEvent;
import org.exoplatform.gwtframework.commons.initializer.event.ApplicationConfigurationReceivedFailedEvent;
import org.exoplatform.gwtframework.commons.initializer.event.ApplicationConfigurationReceivedFailedHandler;
import org.exoplatform.gwtframework.commons.initializer.event.ApplicationConfigurationUpdatedEvent;
import org.exoplatform.gwtframework.commons.initializer.marshal.DefaultApplicationConfigurationUnmarshaler;
import org.exoplatform.gwtframework.commons.rest.AsyncRequest;
import org.exoplatform.gwtframework.commons.rest.AsyncRequestCallback;
import org.exoplatform.gwtframework.commons.rest.AsyncRequestLoader;
import org.exoplatform.gwtframework.commons.rest.HTTPHeader;
import org.exoplatform.gwtframework.commons.rest.HTTPMethod;
import org.exoplatform.gwtframework.commons.rest.HTTPStatus;
import org.exoplatform.gwtframework.commons.rest.MimeType;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.json.client.JSONObject;
import com.google.gwt.user.client.Random;

/**
 * Created by The eXo Platform SAS.
 * @author <a href="mailto:vitaly.parfonov@gmail.com">Vitaly Parfonov</a>
 * @version $Id: $
*/
public class ApplicationInitializer implements ApplicationConfigurationReceivedFailedHandler
{

   private HandlerManager eventBus;

   private String applicationName;

   private AsyncRequestLoader loader;

   public ApplicationInitializer(HandlerManager eventBus, String applicationName, AsyncRequestLoader loader)
   {
      this.eventBus = eventBus;
      this.applicationName = applicationName;
      this.loader = loader;
      this.eventBus.addHandler(ApplicationConfigurationReceivedFailedEvent.TYPE, this);
   }

   public void getApplicationConfiguration()
   {
      getApplicationConfiguration(null);
   }

   public void getApplicationConfiguration(String configNodeName)
   {
      String appURL = getConfigurationUrl(applicationName, configNodeName);
      ApplicationConfiguration configuration = new ApplicationConfiguration();
      configuration.setAppName(applicationName);
      ApplicationConfigurationReceivedEvent event = new ApplicationConfigurationReceivedEvent(configuration);
      DefaultApplicationConfigurationUnmarshaler unmarshaller =
         new DefaultApplicationConfigurationUnmarshaler(eventBus, configuration, configNodeName);
      int[] success = {HTTPStatus.OK};
      AsyncRequestCallback callback =
         new AsyncRequestCallback(eventBus, unmarshaller, event, new ApplicationConfigurationReceivedFailedEvent(),
            success);
      AsyncRequest.build(RequestBuilder.GET, appURL, loader).send(callback);
   }

   private static String getConfigurationUrl(String applicationName, String configNodeName)
   {
      String registryURL = getRegistryURL();
      if (!registryURL.endsWith("/"))
      {
         registryURL += "/";
      }

      registryURL += RegistryConstants.EXO_APPLICATIONS + "/";

      String url = registryURL + applicationName;
      if (configNodeName != null)
      {
         url += "/" + configNodeName;
      }
      url += "?nocache=" + Random.nextInt();
      return url;
   }

   public void setApplicationConfiguration(ApplicationConfiguration applicationConfiguration)
   {
      setApplicationConfiguration(applicationConfiguration, null);
   }

   public void setApplicationConfiguration(ApplicationConfiguration applicationConfiguration, String path)
   {
      String appURL = getConfigurationUrl(applicationConfiguration.getAppName(), path);
      AsyncRequestCallback callback = new AsyncRequestCallback(eventBus, new ApplicationConfigurationUpdatedEvent());
      String requestData =
         "<?xml version=\"1.0\" encoding=\"UTF-8\"?><configuration>" + "<![CDATA["
            + applicationConfiguration.getConfiguration().toString() + "]]></configuration>";
      AsyncRequest.build(RequestBuilder.POST, appURL + "&createIfNotExist=true", loader).header(
         HTTPHeader.CONTENT_TYPE, MimeType.APPLICATION_XML).header(HTTPHeader.X_HTTP_METHOD_OVERRIDE, HTTPMethod.PUT)
         .data(requestData).send(callback);
   }

   /**
    * To access the RegistryService client (i. e. Registry)
    * has to be aware of its address.
    * For this purpose current mechanism suppose to use global
    * JavaScript variable. This method get this variable.
    * @return the the RegistryService address.
    */
   private static native String getRegistryURL() /*-{
        return $wnd.registryURL;
     }-*/;

   private static native boolean isSetup() /*-{
      if ($wnd.setup) 
         return true;   
      else
         return false;
   }-*/;

   private static native JavaScriptObject getAppConfig() /*-{
      return $wnd.appConfig;
   }-*/;

   public void onApplicationConfigurationReceivedFailed(ApplicationConfigurationReceivedFailedEvent event)
   {
      try
      {
         ApplicationConfiguration applicationConfiguration =
            new ApplicationConfiguration(applicationName, new JSONObject(getAppConfig()));
         
         ApplicationConfigurationReceivedEvent receivedEvent =
            new ApplicationConfigurationReceivedEvent(applicationConfiguration);

         eventBus.fireEvent(receivedEvent);
      }
      catch (Exception e)
      {
         eventBus.fireEvent(new ExceptionThrownEvent(new Exception(
            "Application can't load configuration from registry and don't found default configuration!")));
      }

   }

}
