/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.commons.webdav;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import org.exoplatform.gwtframework.commons.xml.QName;

import com.google.gwt.xml.client.Document;
import com.google.gwt.xml.client.Element;
import com.google.gwt.xml.client.Node;
import com.google.gwt.xml.client.NodeList;
import com.google.gwt.xml.client.XMLParser;

/**
 * Created by The eXo Platform SAS        .
 * 
 * PROPFIND's response data 
 * 
 * @version $Id: $
 */

public class PropfindResponse
{

   protected static final String SUCCESS = "HTTP/1.1 200 OK";

   private static String DAV = "DAV:";

   private static String JCR = "jcr:";

   public static final QName RESOURCETYPE = new QName("resourcetype", DAV);

   public static final QName COLLECTION = new QName("collection", DAV);

   public static final QName DISPLAYNAME = new QName("displayname", DAV);

   public static final QName CONTENTTYPE = new QName("getcontenttype", DAV);

   public static final QName JCRNODETYPE = new QName("nodetype", JCR);

   private Resource resource;

   public PropfindResponse()
   {

   }

   /**
    * @return this resource
    */
   public final Resource getResource()
   {
      return resource;
   }

   public String dump()
   {
      String ch = "";
      for (int i = 0; i < resource.children.size(); i++)
      {
         ch += "" + resource.children.get(i).href + "\n";
      }
      String pt = "";
      for (Property p : resource.properties.values())
      {
         pt +=
            "" + p.getName().getLocalName() + "=" + p.value + " childProperties num: " + p.getChildProperties().size()
               + "\n";
      }

      return "PROPFIND RESPONSE: " + "\n href: " + resource.href + "\n properties number: "
         + resource.properties.size() + "\n properties: " + pt + "\n children number: " + resource.children.size()
         + "\n children: " + ch;
   }

   /**
    * parser for incoming XML text
    * @param text
    * @return
    */
   public static PropfindResponse parse(String text)
   {

      PropfindResponse response = new PropfindResponse();

      Document dom = XMLParser.parse(text);

      NodeList responseElements = dom.getElementsByTagName("response");

      for (int i = 0; i < responseElements.getLength(); i++)
      {

         Element resourceElement = (Element)responseElements.item(i);

         Element propstat = (Element)resourceElement.getElementsByTagName("propstat").item(0);
         String status = propstat.getElementsByTagName("status").item(0).getFirstChild().toString();

         if (!status.equals(SUCCESS))
            continue;

         String href = resourceElement.getElementsByTagName("href").item(0).getFirstChild().toString();

         Element prop = (Element)propstat.getElementsByTagName("prop").item(0);

         Resource resource = new Resource(href);

         recurseProperties(prop, resource, null);

         // first response tag is connected with requested folder, and other resource tags is connected with child of requested folder.
         if (i == 0)
         {
            response.resource = resource;
         }
         else
         {
            response.resource.children.add(resource);
         }
      }
      return response;
   }


   /**
    * parser for incoming XML text
    * @param text
    * @return
    */
   public static List<Resource> getResources(String text)
   {
      List<Resource> resources = new ArrayList<PropfindResponse.Resource>();
      
      Document dom = XMLParser.parse(text);
      NodeList responseElements = dom.getElementsByTagName("response");

      for (int i = 0; i < responseElements.getLength(); i++)
      {

         Element resourceElement = (Element)responseElements.item(i);

         Element propstat = (Element)resourceElement.getElementsByTagName("propstat").item(0);
         String status = propstat.getElementsByTagName("status").item(0).getFirstChild().toString();

         if (!status.equals(SUCCESS))
            continue;

         String href = resourceElement.getElementsByTagName("href").item(0).getFirstChild().toString();

         Element prop = (Element)propstat.getElementsByTagName("prop").item(0);

         Resource resource = new Resource(href);

         recurseProperties(prop, resource, null);
         
         resources.add(resource);
      }
      
      return resources;
   }
   
   protected static void recurseProperties(Element element, Resource resource, Property curProperty)
   {

      NodeList nodes = element.getChildNodes();

      for (int i = 0; i < nodes.getLength(); i++)
      {
         Node node = nodes.item(i);

         if (node.getNodeType() == Node.TEXT_NODE)
         {
            curProperty.value = node.getNodeValue();
         }
         else if (node.getNodeType() == Node.ELEMENT_NODE)
         {
            Property p = new Property(new QName(node.getNodeName(), node.getNamespaceURI()));
            if (curProperty == null)
            { // 
               resource.properties.put(p.getName(), p);
            }
            else
            {
               curProperty.getChildProperties().add(p);
            }
            recurseProperties((Element)node, resource, p);
         }
      }
   }

   public static class Resource
   {

      private final String href;

      private HashMap<QName, Property> properties;

      private List<Resource> children;

      public Resource(String href)
      {
         this.href = href;
         this.properties = new HashMap<QName, Property>();
         this.children = new ArrayList<Resource>();
      }

      public String getHref()
      {
         return href;
      }

      public Collection<Property> getProperties()
      {
         return properties.values();
      }

      /**
       * @return child resources
       */
      public final List<Resource> getChildren()
      {
         return children;
      }

      public boolean isCollection()
      {
         Property restype = getProperty(RESOURCETYPE);

         return restype != null && restype.getChildProperty(COLLECTION) != null;
      }

      public Property getProperty(QName name)
      {
         return properties.get(name);
      }

      public String getDisplayname()
      {
         return properties.get(DISPLAYNAME).getValue();
      }

   }

}
