#!/bin/bash
#
# $Id: run-eppsp.sh 8180 2010-09-22 09:34:02Z hgomez $
#
# Copyright (C) 2003-2010 eXo Platform SAS.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see<http://www.gnu.org/licenses/>.#
#
#
# Purpose : Run a JBoss EPP single or clustering environnement for QA teams with/without WCM
#
#
echo CARGOSH-EPPSP-RUN cargosh-binaries 0.4

#
# default vars
#
NUM_INSTANCES=1
CLEARDATA=false
MYSQL=false
VERBOSE=false
DB_LOGIN=root
DB_PASSWORD=
CLUSTERING=false
TAIL_MODE=false
SETUP_NAME=eppsp
SHUTDOWN_WAIT=120
SHUTDOWN_VERBOSE=true
DB_HOST=localhost
DB_LOGIN=root

#
# help / usage about program
#
usage()
{
cat << EOF
usage: $0 options start/stop/restart/doublestart

This script run JBoss instances.

OPTIONS:
   -h      Show this message
   -n      Define number of instances to be launched
   -i      Define instance number to launch (1-4)
   -c      Clear gatein data
   -m      MySQL DB (used in Clear gatein data)
   -d      db host (default: $DB_HOST)
   -l      db login (default: $DB_LOGIN)
   -p      db password (default: )
   -u      activate clustering (default: false)
   -s      setup name (default: $SETUP_NAME)
   -t	   tail mode (server start in background, default: false)
   -v      Verbose
EOF
}

#
# Decode command line parameters
#
init()
{
    cygwin=false;
    linux=false;
    darwin=false;
    case "`uname`" in
        CYGWIN*) cygwin=true;;
        Linux*) linux=true;;
        Darwin*) darwin=true;;
    esac

    CURR_DATE=`date +%Y%m%d%H%M%S`
    export LAUNCH_JBOSS_IN_BACKGROUND=1

    #
    # without parameters, provide help
    #
    if [ $# -lt 1 ]; then
      usage
      exit 1;
    fi

    while getopts "n:i:l:p:s:d:hcmvut" OPTION
    do
         case $OPTION in
             h)
                 usage
                 exit 1
                 ;;
             n)
                 NUM_INSTANCES=$OPTARG
                 INSTANCE_NUMBER=""
                 ;;
             i)
                 INSTANCE_NUMBER=$OPTARG
                 NUM_INSTANCES=""
                 ;;
             c)
                 CLEARDATA=true
                 ;;
             m)
                 MYSQL=true
                 ;;
             d)
                 DB_HOST=$OPTARG
                 ;;
             l)
                 DB_LOGIN=$OPTARG
                 ;;
             p)
                 DB_PASSWORD=$OPTARG
                 ;;
             s)
                 SETUP_NAME=$OPTARG
                 ;;
             v)
                 VERBOSE=true
                 ;;
             c)
                 VERBOSE=true
                 ;;
             u)
                 CLUSTERING=true
                 ;;
             t)
                 TAIL_MODE=true
                 ;;
             ?)
                 usage
                 exit
                 ;;
         esac
    done

    # skip getopt parms
    shift $((OPTIND-1))

    if [ $# -lt 1 ]; then
        usage
        exit 1;
    fi

    MODE=$1
}

#
# Prepare dirs
#
prepare()
{
    INSTALL_DIR=`pwd`/target/install

    # fix location provided by Master to Agents
    if [ ! -z "$WORKSPACE" ]; then
       INSTALL_DIR="$WORKSPACE/../../../$SETUP_NAME"
    else

	    if [ ! -z "$HUDSON_HOME" ]; then
	        if [ -d $HUDSON_HOME ]; then
	            INSTALL_DIR=$HUDSON_HOME/$SETUP_NAME
	        fi
        fi
    fi

    if [ ! -z "$EXO_INSTALL_HOME" ]; then

        if [ -d $EXO_INSTALL_HOME ]; then
            INSTALL_DIR=$EXO_INSTALL_HOME
        fi

    fi

    echo "using $INSTALL_DIR as working eppsp"

    JBOSS_DISTR_NAME=jboss-epp

    # directory where JBossAS will be installed (if not defined outside)
    if [ -z "$JBOSS_BASE" ]; then
        JBOSS_BASE=$INSTALL_DIR/$JBOSS_DISTR_NAME
    fi

    # define jboss.server.data.dir since gatein.data.dir=${jboss.server.data.dir}/gatein (if not defined outside)
    if [ -z "$GATEIN_DATA_DIR" ]; then
        GATEIN_DATA_DIR=$INSTALL_DIR/jboss-data/gatein-data
    fi

    if [ ! -d $GATEIN_DATA_DIR ]; then
        mkdir -p $GATEIN_DATA_DIR
    fi
}

#
# Gatein data
#
clear_gatein_data_if() {

    if $CLEARDATA; then

        if [ -d $GATEIN_DATA_DIR ]; then
            echo "clearing gatein datadir"
            rm -rf $GATEIN_DATA_DIR
        fi

        #
        # Handle OS/X - Linux mysql location
        #
        if [ -x /usr/local/mysql/bin/mysqladmin ]; then
            MYSQLADMIN=/usr/local/mysql/bin/mysqladmin
        fi

        if [ -x /Applications/MAMP/Library/bin/mysqladmin ]; then
            MYSQLADMIN=/Applications/MAMP/Library/bin/mysqladmin
        fi

        if [ -x /usr/bin/mysqladmin ]; then
            MYSQLADMIN=/usr/bin/mysqladmin
        fi

        if $MYSQL; then
            echo "clearing MySQL Databases"

            $MYSQLADMIN drop $SETUP_NAME-portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN drop $SETUP_NAME-idm_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN drop $SETUP_NAME-jcr_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN drop $SETUP_NAME-idm_ecmdemo -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN drop $SETUP_NAME-jcr_ecmdemo -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD

            echo "creating MySQL Databases"
            $MYSQLADMIN create $SETUP_NAME-portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN create $SETUP_NAME-idm_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN create $SETUP_NAME-jcr_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN create $SETUP_NAME-idm_ecmdemo -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN create $SETUP_NAME-jcr_ecmdemo -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD

        fi

    fi
}

#
# Start gatein instance
#
start()
{

    INST_ID=$1
    JBOSSAS_INST_BASE=$JBOSS_BASE/server/all$INST_ID
    RUN_ALLOWED=true

    JBOSS_LOG=$JBOSS_BASE/server/all$INST_ID/log/jboss.log
    JBOSS_PID=$JBOSS_BASE/server/all$INST_ID/log/jboss.pid

    if [ -f $JBOSS_PID ]; then

        read KPID <  $JBOSS_PID

        if [ ! "$(ps -p $KPID | grep -c $KPID)" -eq "0" ]; then
            echo "jboss instance is allready running (pid=$KPID), aborting launch."
            RUN_ALLOWED=false
            exit -1
        fi
    fi

    if $RUN_ALLOWED; then

        case $INST_ID in
            1)
                PORT_BINDING="ports-default"
                PORT_HTTP=8080
                ;;
            2)
                PORT_BINDING="ports-01"
                PORT_HTTP=8180
                ;;
            3)
                PORT_BINDING="ports-02"
                PORT_HTTP=8280
                ;;
            4)
                PORT_BINDING="ports-03"
                PORT_HTTP=8380
                ;;
            ?)
                echo "only four instances could be launched"
                exit
                ;;
        esac

        # prepare log dir
        mkdir -p $JBOSSAS_INST_BASE/log
        # rm -f $JBOSSAS_INST_BASE/log/*

        # avoid to get JAVA_OPTS from current config (ie: hudson)
       unset JAVA_OPTS

        # set JMX (8002, 8102, 8202, 8302)
        #export JRMP_OPTS=-Dcom.sun.management.jmxremote.port=8$[$INST_ID - 1]02
        #export JRMP_OPTS="-Dcom.sun.management.jmxremote"
        # jboss.messaging.ServerPeerID set the JMS Id.
        if $CLUSTERING; then
            echo "launching JBOSS instance in clustering mode with : "
            echo "$JBOSS_BASE/bin/run.sh -c all$INST_ID -Dexo.profiles=cluster -Djboss.service.binding.set=$PORT_BINDING -Djboss.messaging.ServerPeerID=$INST_ID $JRMP_OPTS"
            $JBOSS_BASE/bin/run.sh -c all$INST_ID -Dexo.profiles=cluster -Djboss.service.binding.set=$PORT_BINDING -Djboss.messaging.ServerPeerID=$INST_ID $JRMP_OPTS > $JBOSS_LOG 2>&1 &
        else
            echo "launching JBOSS instance in non clustering mode with : "
            echo "$JBOSS_BASE/bin/run.sh -c all$INST_ID -Djboss.service.binding.set=$PORT_BINDING -Djboss.messaging.ServerPeerID=$INST_ID $JRMP_OPTS"
            $JBOSS_BASE/bin/run.sh -c all$INST_ID -Djboss.service.binding.set=$PORT_BINDING -Djboss.messaging.ServerPeerID=$INST_ID $JRMP_OPTS > $JBOSS_LOG 2>&1 &
        fi

        # keep process id
        echo $! > $JBOSS_PID

        echo "Waiting JBOSS server to be ready"
        echo "( logs are here:  $JBOSS_LOG )"

        if $TAIL_MODE; then
            tail -f  $JBOSS_LOG
        else

            START_ERROR=true

            # Hack to avoid Hudson to kill spawned job (
            # http://hudson.361315.n4.nabble.com/Hudson-killing-background-server-process-even-with-daemonize-td384633.html
            if [ ! -z "$WORKSPACE" ]; then
                export BUILD_ID=dontKillMe
            fi
            # 10mn wait
            # get 100 latest line and search for "Started in "
            for i in {1..600}; do
                tail -100 $JBOSS_LOG | grep "Started in " >>/dev/null

                if [ $? = "0" ]; then
                    START_ERROR=false
                    echo "JBoss server started after $i seconds, follow its log file"
                    cat $JBOSS_LOG
                       break;
                fi

                   sleep 1

            done

            if $START_ERROR; then
                echo "JBoss server didn't start after $i seconds, follow its log file and aborting..."
                cat $JBOSS_LOG
                exit -1
            fi

        fi
    fi
}

#
# Stop JBoss instance
#
stop() {

    INST_ID=$1
    JBOSSAS_INST_BASE=$JBOSS_BASE/server/all$INST_ID

    JBOSS_LOG=$JBOSS_BASE/server/all$INST_ID/log/jboss.log
    JBOSS_PID=$JBOSS_BASE/server/all$INST_ID/log/jboss.pid

    if [ -f $JBOSS_PID ]; then

        read KPID <  $JBOSS_PID

        if [ ! -z "$KPID" ]; then

            # send kill to process
            kill $KPID

            count="0"
            # wait it to finish
            until [ "$(ps -p $KPID | grep -c $KPID)" -eq "0" ] || \
                    [ "$count" -gt "$SHUTDOWN_WAIT" ]; do
                if [ "$SHUTDOWN_VERBOSE" = "true" ]; then
                    echo "waiting for processes $KPID to exit"
                fi
                sleep 1
                let count="${count}+1"
            done

            # If process still active after SHUTDOWN_WAIT seconds, brutal kill ;(
            if [ "$count" -gt "$SHUTDOWN_WAIT" ]; then
                if [ "$SHUTDOWN_VERBOSE" = "true" ]; then
                    echo "killing processes which didn't stop after $SHUTDOWN_WAIT seconds"
                fi
                kill -9 $KPID
            fi

            # remove process id
            rm -f $JBOSS_PID

            # Under Hudson get back in console JBoss log
            if [ ! -z "$WORKSPACE" ]; then
                echo "@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
                echo "@ dumping JBoss #$INST_ID log file @"
                echo "@@@@@@@@@@@@@@@@@@@@@@@@@@@@"

                cat $JBOSS_LOG
            fi

            # backup previous log (jboss.log-yyyymmdd)

            mv $JBOSS_LOG $JBOSS_LOG-$CURR_DATE

          fi
    fi
}

#
# Start JBoss instance(s)
#
start_instances()
{
    #
    # Only on instance to launch ?
    #
    if [ ! -z "$INSTANCE_NUMBER" ]; then

        echo "launch JBoss instance #$INSTANCE_NUMBER"

        start $INSTANCE_NUMBER


    #
    # Launch many instances
    #

    else

         for GINST in `eval echo {1..$NUM_INSTANCES}`; do

            echo "launching JBoss instance #$GINST"
            start $GINST

        done

    fi
}


#
# Stop JBoss instance(s)
#
stop_instances() {

    #
    # Only on instance to stop ?
    #
    if [ ! -z "$INSTANCE_NUMBER" ]; then

        echo "stoping JBoss instance #$INSTANCE_NUMBER"
        stop $INSTANCE_NUMBER

        clear_gatein_data_if
    #
    # Stop many instances
    #

    else

        for GINST in `eval echo {1..$NUM_INSTANCES}`; do

            echo "stoping JBoss instance #$GINST"
            stop $GINST

        done

        clear_gatein_data_if

    fi
}

#
# MAIN HERE
#

init $@
prepare

if [ "$MODE" = "start" ]; then

    clear_gatein_data_if
    start_instances

fi

if [ "$MODE" = "stop" ]; then

    stop_instances

fi

if [ "$MODE" = "restart" ]; then

    stop_instances
    start_instances

fi

if [ "$MODE" = "doublestart" ]; then

    clear_gatein_data_if
    start_instances
    stop_instances
    start_instances

fi
