#!/bin/bash
#
# $Id: cargosh-plf-tomcat-run.sh 8180 2010-09-22 09:34:02Z hgomez $
#
# Copyright (C) 2003-2010 eXo Platform SAS.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see<http://www.gnu.org/licenses/>.#
#
#
# Purpose : Run a PLF single or clustering environnement for QA teams
#
#
echo CARGOSH-PLF-TOMCAT-RUN cargosh-binaries 0.4

#
# default vars
#
NUM_INSTANCES=1
CLEARDATA=false
MYSQL=false
VERBOSE=false
DB_LOGIN=root
DB_PASSWORD=
CLUSTERING=false
TAIL_MODE=false
SETUP_NAME="plf"
SHUTDOWN_WAIT=30
SHUTDOWN_VERBOSE=true
DB_HOST=localhost
DB_LOGIN=root
QA_JAVA_OPTS=
TOMCAT_PLF_RUNTIME_LOCATION=bin/tomcat6-bundle

#
# help / usage about program
#
usage()
{
cat << EOF
usage: $0 options start/stop/restart/doublestart

This script run PLF instances.

OPTIONS:
   -h      Show this message
   -n      Define number of instances to be launched
   -i      Define instance number to launch (1-4)
   -c      Clear gatein data
   -m      MySQL DB (used in Clear gatein data)
   -d      db host (default: $DB_HOST)
   -l      db login (default: $DB_LOGIN)
   -p      db password (default: )
   -u      activate clustering (default: false)
   -s      setup name (default: $SETUP_NAME)
   -t	   tail mode (server start in background, default: false)
   -v      java opts (passed to process as JAVA_OPTS)
   -r	   plf runtime location relative to package (default: $TOMCAT_PLF_RUNTIME_LOCATION)
   -w      shutdown wait (default: ${SHUTDOWN_WAIT}s)

EOF
}

#
# Decode command line parameters
#
init()
{
    cygwin=false;
    linux=false;
    darwin=false;
    case "`uname`" in
        CYGWIN*) cygwin=true;;
        Linux*) linux=true;;
        Darwin*) darwin=true;;
    esac

    CURR_DATE=`date +%Y%m%d%H%M%S`

    #
    # without parameters, provide help
    #
    if [ $# -lt 1 ]; then
      usage
      exit 1;
    fi

    while getopts "n:i:l:p:s:d:v:w:r:hcmut" OPTION
    do
         case $OPTION in
             h)
                 usage
                 exit 1
                 ;;
             n)
                 NUM_INSTANCES=$OPTARG
                 INSTANCE_NUMBER=""
                 ;;
             i)
                 INSTANCE_NUMBER=$OPTARG
                 NUM_INSTANCES=""
                 ;;
             d)
                 DB_HOST=$OPTARG
                 ;;
             l)
                 DB_LOGIN=$OPTARG
                 ;;
             p)
                 DB_PASSWORD=$OPTARG
                 ;;
             s)
                 SETUP_NAME=$OPTARG
                 ;;
             v)
                 QA_JAVA_OPTS=$OPTARG
                 ;;
             w)
                 SHUTDOWN_WAIT=$OPTARG
                 ;;
             r)
                 TOMCAT_PLF_RUNTIME_LOCATION=$OPTARG
                 ;;
             c)
                 CLEARDATA=true
                 ;;
             m)
                 MYSQL=true
                 ;;
             u)
                 CLUSTERING=true
                 ;;
             t)
                 TAIL_MODE=true
                 ;;
             ?)
                 usage
                 exit
                 ;;
         esac
    done

    # skip getopt parms
    shift $((OPTIND-1))

    if [ $# -lt 1 ]; then
        usage
        exit 1;
    fi

    MODE=$1
}

#
# Prepare dirs
#
prepare()
{
    INSTALL_DIR=`pwd`/target/install

    # fix location provided by Master to Agents
    if [ ! -z "$WORKSPACE" ]; then
       INSTALL_DIR="$WORKSPACE/../../../$SETUP_NAME"
    else

	    if [ ! -z "$HUDSON_HOME" ]; then
	        if [ -d $HUDSON_HOME ]; then
	            INSTALL_DIR=$HUDSON_HOME/$SETUP_NAME
	        fi
        fi
    fi

    if [ ! -z "$EXO_INSTALL_HOME" ]; then

        if [ -d $EXO_INSTALL_HOME ]; then
            INSTALL_DIR=$EXO_INSTALL_HOME
        fi

    fi

    echo "using $INSTALL_DIR as working plf"

    # data directory
    GATEIN_DATA_DIR=$INSTALL_DIR/datas
    echo "using $GATEIN_DATA_DIR to store Gatein datas"

    if [ ! -d $GATEIN_DATA_DIR ]; then
        mkdir -p $GATEIN_DATA_DIR
    fi
}

#
# Gatein data
#
clear_gatein_data_if() {

    if $CLEARDATA; then

        if [ -d $GATEIN_DATA_DIR ]; then
            echo "clearing Gatein datas"
            rm -rf $GATEIN_DATA_DIR
        fi

        #
        # Handle OS/X - Linux mysql location
        #
        if [ -x /usr/local/mysql/bin/mysqladmin ]; then
            MYSQLADMIN=/usr/local/mysql/bin/mysqladmin
        fi

        if [ -x /Applications/MAMP/Library/bin/mysqladmin ]; then
            MYSQLADMIN=/Applications/MAMP/Library/bin/mysqladmin
        fi

        if [ -x /usr/bin/mysqladmin ]; then
            MYSQLADMIN=/usr/bin/mysqladmin
        fi

        if $MYSQL; then
            echo "clearing MySQL Databases"

            $MYSQLADMIN drop $SETUP_NAME-exo-idm_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN drop $SETUP_NAME-exo-jcr_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD

            echo "creating MySQL Databases"
            $MYSQLADMIN create $SETUP_NAME-exo-idm_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD
            $MYSQLADMIN create $SETUP_NAME-exo-jcr_portal -f --host=$DB_HOST --user=$DB_LOGIN --password=$DB_PASSWORD

        fi

    fi
}

#
# Start PLF instance
#
start()
{

    INST_ID=$1
    TOMCAT_INST_BASE=$INSTALL_DIR/tomcat$INST_ID
    RUN_ALLOWED=true

    TOMCAT_LOG=$TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/logs/tomcat.log
    TOMCAT_PID=$TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/logs/tomcat.pid

    if [ -f $TOMCAT_PID ]; then

        read KPID <  $TOMCAT_PID

        if [ ! "$(ps -p $KPID | grep -c $KPID)" -eq "0" ]; then
            echo "PLF instance is allready running (pid=$KPID), aborting launch."
            RUN_ALLOWED=false
            exit -1
        fi
    fi

    if $RUN_ALLOWED; then

        # also unset Tomcat vars
        unset CATALINA_BASE
        unset CATALINA_HOME
        unset CATALINA_LOG
        unset CATALINA_OPTS
        unset CATALINA_PID
        unset CATALINA_SUBID
        unset CATALINA_TMPDIR
        unset CATALINA_USER

        export JAVA_OPTS=$QA_JAVA_OPTS

        echo "launching PLF instance with : "
        echo "$TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/bin/gatein.sh run"
        $TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/bin/gatein.sh run > $TOMCAT_LOG 2>&1 &

        # keep process id
        echo $! > $TOMCAT_PID

        echo "Waiting PLF server to be ready"
        echo "( logs are here:  $TOMCAT_LOG )"

        if $TAIL_MODE; then
            tail -f $TOMCAT_LOG
        else

            START_ERROR=true

            # Hack to avoid Hudson to kill spawned job (
            # http://hudson.361315.n4.nabble.com/Hudson-killing-background-server-process-even-with-daemonize-td384633.html
            if [ ! -z "$WORKSPACE" ]; then
                export BUILD_ID=dontKillMe
            fi
            # 10mn wait
            # get 100 latest line and search for "Started in "
            for i in {1..600}; do
                tail -100 $TOMCAT_LOG | grep "Server startup in " >>/dev/null

                if [ $? = "0" ]; then
                    START_ERROR=false
                    echo "PLF server started after $i seconds, follow its log file"
                    cat $TOMCAT_LOG
                    break;
                fi

                sleep 1

            done

            if $START_ERROR; then
                echo "PLF server didn't start after $i seconds, follow its log file and aborting..."
                cat $TOMCAT_LOG
                exit -1
            fi

        fi
    fi
}

#
# Stop PLF instance
#
stop() {

    INST_ID=$1
    TOMCAT_INST_BASE=$INSTALL_DIR/tomcat$INST_ID

    TOMCAT_LOG=$TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/logs/tomcat.log
    TOMCAT_PID=$TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/logs/tomcat.pid

    if [ -f $TOMCAT_PID ]; then

        read KPID <  $TOMCAT_PID

        if [ ! -z "$KPID" ]; then

            # avoid to get JAVA_OPTS from current config (ie: hudson)
           unset JAVA_OPTS
           # also unset Tomcat vars
           unset CATALINA_BASE
           unset CATALINA_HOME
           unset CATALINA_LOG
           unset CATALINA_OPTS
           unset CATALINA_PID
           unset CATALINA_SUBID
           unset CATALINA_TMPDIR
           unset CATALINA_USER

            export JAVA_OPTS=$QA_JAVA_OPTS
            # set CATALINA_PID for catalina.sh stop procedure
            export CATALINA_PID=$TOMCAT_PID

            echo "stoping PLF instance with : "
            echo "$TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/bin/gatein.sh stop $SHUTDOWN_WAIT -force"
            $TOMCAT_INST_BASE/$TOMCAT_PLF_RUNTIME_LOCATION/bin/gatein.sh stop $SHUTDOWN_WAIT -force

            # remove process id
            rm -f $TOMCAT_PID

            # Under Hudson get back in console PLF log
            if [ ! -z "$WORKSPACE" ]; then
                echo "@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
                echo "@ dumping PLF #$INST_ID log file @"
                echo "@@@@@@@@@@@@@@@@@@@@@@@@@@@@"

                cat $TOMCAT_LOG
            fi

            # backup previous log (tomcat.log-yyyymmdd)

            mv $TOMCAT_LOG $TOMCAT_LOG-$CURR_DATE

          fi
    fi
}

#
# Start PLF instance(s)
#
start_instances()
{
    #
    # Only on instance to launch ?
    #
    if [ ! -z "$INSTANCE_NUMBER" ]; then

        echo "launch PLF instance #$INSTANCE_NUMBER"

        start $INSTANCE_NUMBER


    #
    # Launch many instances
    #

    else

         for GINST in `eval echo {1..$NUM_INSTANCES}`; do

            echo "launching PLF instance #$GINST"
            start $GINST

        done

    fi
}


#
# Stop gatein instance(s)
#
stop_instances() {

    #
    # Only on instance to stop ?
    #
    if [ ! -z "$INSTANCE_NUMBER" ]; then

        echo "stoping PLF instance #$INSTANCE_NUMBER"
        stop $INSTANCE_NUMBER

        clear_gatein_data_if
    #
    # Stop many instances
    #

    else

        for GINST in `eval echo {1..$NUM_INSTANCES}`; do

            echo "stoping PLF instance #$GINST"
            stop $GINST

        done

        clear_gatein_data_if

    fi
}

#
# MAIN HERE
#

init $@
prepare

if [ "$MODE" = "start" ]; then

    clear_gatein_data_if
    start_instances

fi

if [ "$MODE" = "stop" ]; then

    stop_instances

fi

if [ "$MODE" = "restart" ]; then

    stop_instances
    start_instances

fi

if [ "$MODE" = "doublestart" ]; then

    clear_gatein_data_if

    # in clustering mode, only first node need to be started/stopped
    if $CLUSTERING; then
        start 1
        stop 1
    else
        start_instances
        stop_instances
    fi

    start_instances

fi
