/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shindig.gadgets.uri;

import java.util.Collection;
import java.util.Collections;

import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.gadgets.Gadget;
import org.apache.shindig.gadgets.GadgetException;

import com.google.inject.ImplementedBy;

/**
 * Interface defining methods used to generate Uris for the /js servlet.
 */
public interface JsUriManager {
  /**
   * @param gadget The gadget in which the requested JS will be externed.
   * @param extern The list of features that js is needed for.
   * @return The uri for the externed javascript that includes all listed extern libraries.
   */
  Uri makeExternJsUri(Gadget gadget, Collection<String> extern);

  /**
   * Processes the inbound URL, for use by serving code in determining which JS to serve
   * and with what caching properties.
   *
   * @param uri Generated extern JS Uri
   * @return Processed status of the provided Uri.
   */
  JsUri processExternJsUri(Uri uri) throws GadgetException;

  public static class JsUri {
    private final UriStatus status;
    private final Collection<String> libs;

    public JsUri(UriStatus status, Collection<String> libs) {
      this.status = status;
      this.libs = libs;
    }

    public UriStatus getStatus() {
      return status;
    }

    public Collection<String> getLibs() {
      return Collections.unmodifiableCollection(libs);
    }
  }

  @ImplementedBy(DefaultJsVersioner.class)
  public interface Versioner {
    /**
     * @param gadgeUri Gadget for which extern Uri was generated.
     * @param extern Collection of libs externed.
     * @return Version string for the Uri.
     */
    String version(Uri gadgetUri, String container, Collection<String> extern);

    /**
     * @param gadgetUri Gadget for which extern Uri was generated.
     * @param extern Collection of libs externed.
     * @param version Version string generated by the Versioner.
     * @return Validation status of the version.
     */
    UriStatus validate(Uri gadgetUri, String container, Collection<String> extern, String version);
  }
}
