package org.jasig.web.service;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletSession;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jasig.web.util.DefaultModelPasser;
import org.jasig.web.util.ModelPasser;
import org.jasig.web.util.SecureSessionKeyGenerator;
import org.jasig.web.util.SessionKeyGenerator;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.ServletRequestUtils;

/**
 * Service intended for annotation-based controllers.
 * 
 * @author Eric Dalquist
 * @author Jen Bourey
 * @revision $Revision: 47362 $
 */
public class AjaxPortletSupportService implements AjaxPortletSupport {
    public static final String SESSION_KEY = "key";
    private Log logger = LogFactory.getLog(AjaxPortletSupportService.class);

    private SessionKeyGenerator sessionKeyGenerator = new SecureSessionKeyGenerator();
    private ModelPasser modelPasser = new DefaultModelPasser();
    private String ajaxHandlerErrorKey = DEFAULT_AJAX_HANDLER_ERROR_KEY;
    
    
    public SessionKeyGenerator getSessionKeyGenerator() {
        return this.sessionKeyGenerator;
    }
    /**
     * Used to generate unique keys within a portlet session. Defaults to {@link SecureSessionKeyGenerator}
     */
    public void setSessionKeyGenerator(SessionKeyGenerator sessionKeyGenerator) {
        Validate.notNull(sessionKeyGenerator);
        this.sessionKeyGenerator = sessionKeyGenerator;
    }
    
    public ModelPasser getModelPasser() {
        return this.modelPasser;
    }
    /**
     * Used to pass the model data to the rendering servlet. Defaults to {@link DefaultModelPasser}
     */
    public void setModelPasser(ModelPasser modelPasser) {
        Validate.notNull(modelPasser);
        this.modelPasser = modelPasser;
    }
    
    public String getAjaxHandlerErrorKey() {
        return this.ajaxHandlerErrorKey;
    }
    /**
     * Model attribute used to pass an exception object to the rendering serlvet if one occurs
     * from calling {@link #handleRenderRequestInternal(javax.portlet.RenderRequest, javax.portlet.RenderResponse)}.
     * Defaults to the value of {@link #DEFAULT_AJAX_HANDLER_ERROR_KEY}
     */
    public void setAjaxHandlerErrorKey(String ajaxHandlerErrorKey) {
        Validate.notNull(ajaxHandlerErrorKey);
        this.ajaxHandlerErrorKey = ajaxHandlerErrorKey;
    }
    
    /* (non-Javadoc)
     * @see org.jasig.web.service.AjaxPortletSupport#redirectAjaxResponse(java.util.Map, javax.portlet.ActionRequest, javax.portlet.ActionResponse)
     */
    public void redirectAjaxResponse(Map<Object, Object> model, ActionRequest request, ActionResponse response) throws IOException {
        this.redirectAjaxResponse(DEFAULT_AJAX_SERVLET_NAME, model, request, response);
    }
    
    /* (non-Javadoc)
     * @see org.jasig.web.service.AjaxPortletSupport#redirectAjaxResponse(java.lang.String, java.util.Map, javax.portlet.ActionRequest, javax.portlet.ActionResponse)
     */
    public final void redirectAjaxResponse(String ajaxServletName, Map<Object, Object> model, ActionRequest request, ActionResponse response) throws IOException {
        //Generate the session-unique key
        final PortletSession session = request.getPortletSession();
        final String sessionKey = this.sessionKeyGenerator.getNextSessionKey(session);
        
        this.modelPasser.passModelToServlet(request, response, sessionKey, model);
        
        //Get the servlet url and redirect
        final String ajaxServletUrl = this.getAjaxServletUrl(request, response, ajaxServletName, sessionKey);
        final String encodedAjaxServletUrl = response.encodeURL(ajaxServletUrl);
        
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Redirecting ActionRequest to: " + encodedAjaxServletUrl);
        }
        response.sendRedirect(encodedAjaxServletUrl);
    }
    
    public void redirectAjaxResponse(ActionRequest request, ActionResponse response, ModelGenerator generator) throws IOException {
        this.redirectAjaxResponse(DEFAULT_AJAX_SERVLET_NAME, request, response, generator);
        
    }
    
    public void redirectAjaxResponse(String ajaxServletName, ActionRequest request, ActionResponse response, ModelGenerator generator) throws IOException {
        Map<Object, Object> model;
        try {
            model = generator.generate(request, response);
        }
        catch (Exception e) {
            model = new HashMap<Object, Object>();
            this.logger.warn("An exception occurred during handleAjaxRequestInternal()", e);
            model.put(this.ajaxHandlerErrorKey, e);
            //Ensure there is an attribute with the default error key so the servlet can look for an error condition
            if (!AjaxPortletSupportService.DEFAULT_AJAX_HANDLER_ERROR_KEY.equals(this.ajaxHandlerErrorKey)) {
                model.put(AjaxPortletSupportService.DEFAULT_AJAX_HANDLER_ERROR_KEY, true);
            }
        }
        
        this.redirectAjaxResponse(ajaxServletName, model, request, response);
    }
    /* (non-Javadoc)
     * @see org.jasig.web.service.AjaxPortletSupport#getAjaxModel(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    public final Map<Object, Object> getAjaxModel(HttpServletRequest request, HttpServletResponse response) throws ServletRequestBindingException, IOException {
        final String key = ServletRequestUtils.getRequiredStringParameter(request, SESSION_KEY);
        
        final Map<Object, Object> model = this.modelPasser.getModelFromPortlet(request, response, key);
        
        //If not model send a 404 status to signal no data was found
        if (model == null) {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, "No model exists in the session for key '" + key + "'");
        }
        //If an error response send a 500 status code to signal an error
        else if (model.containsKey(DEFAULT_AJAX_HANDLER_ERROR_KEY)) {
            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        }
        
        return model;
    }

    /**
     * Generate the servlet URL to redirect the AJAX request to for rendering the response model. The
     * URL will be encoded via {@link PortletResponse#encodeURL(String)} before the redirect is sent.
     * 
     * @param sessionKey The session key to provide on the URL
     */
    protected String getAjaxServletUrl(ActionRequest request, ActionResponse response, String ajaxServletName, String sessionKey) throws UnsupportedEncodingException {
        final String contextPath = request.getContextPath();
        final String characterEncoding = request.getCharacterEncoding();
        final String encodedKeyParam = URLEncoder.encode(SESSION_KEY, characterEncoding);
        final String encodedSessionKey = URLEncoder.encode(sessionKey, characterEncoding);

        return contextPath + "/" + ajaxServletName + "?" + encodedKeyParam + "=" + encodedSessionKey;
    }
}
