package org.codehaus.plexus.configuration;


/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Very similar to DefaultPlexusConfiguration but allows for replacing children
 * 
 * @author Eric Dalquist
 * @version $Revision: 25140 $
 */
class MutablePlexusConfiguration implements PlexusConfiguration {

    // ----------------------------------------------------------------------
    // Constants
    // ----------------------------------------------------------------------

    private static final PlexusConfiguration[] NO_CHILDREN = new PlexusConfiguration[0];

    // ----------------------------------------------------------------------
    // Implementation fields
    // ----------------------------------------------------------------------

    private final String name;

    private String value;

    private List<PlexusConfiguration> childIndex = Collections.emptyList();

    private Map<String, List<PlexusConfiguration>> childMap = Collections.emptyMap();

    private Map<String, String> attributeMap = Collections.emptyMap();

    // ----------------------------------------------------------------------
    // Constructors
    // ----------------------------------------------------------------------

    public MutablePlexusConfiguration( final PlexusConfiguration source )
    {
        this.name = source.getName();
        this.value = source.getValue();
        
        for (final String attributeName : source.getAttributeNames()) {
            this.setAttribute(attributeName, source.getAttribute(attributeName));
        }
        
        for (final PlexusConfiguration child : source.getChildren()) {
            this.addChild(child);
        }
    }
    
    public MutablePlexusConfiguration( final String name )
    {
        this( name, null );
    }

    public MutablePlexusConfiguration( final String name, final String value )
    {
        this.name = name;
        this.value = value;
    }

    // ----------------------------------------------------------------------
    // Public methods
    // ----------------------------------------------------------------------

    public final String getName()
    {
        return name;
    }

    public final String getValue()
    {
        return value;
    }

    public final String getValue( final String defaultValue )
    {
        return null != value ? value : defaultValue;
    }

    public final void setValue( final String value )
    {
        this.value = value;
    }

    public final String[] getAttributeNames()
    {
        return attributeMap.keySet().toArray( new String[attributeMap.size()] );
    }

    public final String getAttribute( final String attributeName )
    {
        return attributeMap.get( attributeName );
    }

    public final String getAttribute( final String attributeName, final String defaultValue )
    {
        final String attributeValue = attributeMap.get( attributeName );
        return null != attributeValue ? attributeValue : defaultValue;
    }

    public final void setAttribute( final String attributeName, final String attributeValue )
    {
        if ( attributeMap.isEmpty() )
        {
            attributeMap = new HashMap<String, String>();
        }
        attributeMap.put( attributeName, attributeValue );
    }

    public final PlexusConfiguration getChild( final String childName )
    {
        return getChild( childName, true );
    }
    
    public final void replaceLastChild(PlexusConfiguration newChild) {
        final String childName = newChild.getName();
        
        final List<PlexusConfiguration> children = childMap.get( childName );
        if ( null == children )
        {
            addChild(newChild);
        }
        
        children.set(children.size() - 1, newChild);
    }

    public final PlexusConfiguration getChild( final String childName, final boolean create )
    {
        final List<PlexusConfiguration> children = childMap.get( childName );
        if ( null != children )
        {
            return children.get( 0 );
        }
        return create ? add( createChild( childName ) ) : null;
    }

    public final PlexusConfiguration[] getChildren()
    {
        return childIndex.toArray( new PlexusConfiguration[childIndex.size()] );
    }

    public final PlexusConfiguration[] getChildren( final String childName )
    {
        final List<PlexusConfiguration> children = childMap.get( childName );
        if ( null != children )
        {
            return children.toArray( new PlexusConfiguration[children.size()] );
        }
        return NO_CHILDREN;
    }

    public final int getChildCount()
    {
        return childIndex.size();
    }

    public final PlexusConfiguration getChild( final int index )
    {
        return childIndex.get( index );
    }

    public final void addChild( final PlexusConfiguration child )
    {
        add( child );
    }

    public final PlexusConfiguration addChild( final String childName, final String childValue )
    {
        add( createChild( childName ) ).setValue( childValue );
        return this;
    }



    // ----------------------------------------------------------------------
    // Customizable methods
    // ----------------------------------------------------------------------

    protected PlexusConfiguration createChild( final String childName )
    {
        return new MutablePlexusConfiguration( childName );
    }

    // ----------------------------------------------------------------------
    // Shared methods
    // ----------------------------------------------------------------------

    protected final PlexusConfiguration add( final PlexusConfiguration child )
    {
        final String childName = child.getName();

        List<PlexusConfiguration> children = childMap.get( childName );
        if ( null == children )
        {
            children = new ArrayList<PlexusConfiguration>();
            if ( childMap.isEmpty() )
            {
                // create mutable map and index at the same time
                childMap = new LinkedHashMap<String, List<PlexusConfiguration>>();
                childIndex = new ArrayList<PlexusConfiguration>();
            }
            childMap.put( childName, children );
        }

        childIndex.add( child );
        children.add( child );

        return child;
    }

    @Override
    public String toString()
    {
        final StringBuilder buf = new StringBuilder().append( '<' ).append( getName() );
        for ( final String a : getAttributeNames() )
        {
            buf.append( ' ' ).append( a ).append( "=\"" ).append( getAttribute( a ) ).append( '"' );
        }
        if ( getChildCount() > 0 )
        {
            buf.append( '>' );
            for ( final PlexusConfiguration c : getChildren() )
            {
                buf.append( c );
            }
            buf.append( "</" ).append( getName() ).append( '>' );
        }
        else if ( null != getValue() )
        {
            buf.append( '>' ).append( getValue() ).append( "</" ).append( getName() ).append( '>' );
        }
        else
        {
            buf.append( "/>" );
        }
        return buf.append( '\n' ).toString();
    }
}
