/* Copyright 2004 The JA-SIG Collaborative.  All rights reserved.
*  See license distributed with this file and
*  available online at http://www.uportal.org/license.html
*/

package org.jasig.services.persondir.support;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.Validate;



/**
 * @author Eric Dalquist
 * @version $Revision: 43105 $ $Date: 2008-02-14 09:49:44 -0600 (Thu, 14 Feb 2008) $
 * @since uPortal 2.5
 */
public final class MultivaluedPersonAttributeUtils {

    
    /**
     * Translate from a more flexible Attribute to Attribute mapping format to a Map
     * from String to Set of Strings.
     * 
     * The point of the map is to map from attribute names in the underlying data store
     * (e.g., JDBC column names, LDAP attribute names) to uPortal attribute names. 
     * Any given underlying data store attribute might map to zero uPortal
     * attributes (not appear in the map at all), map to exactly one uPortal attribute
     * (appear in the Map as a mapping from a String to a String or as a mapping
     * from a String to a Set containing just one String), or map to several uPortal
     * attribute names (appear in the Map as a mapping from a String to a Set
     * of Strings).
     * 
     * This method takes as its argument a {@link Map} that must have keys of 
     * type {@link String} and values of type {@link String} or {@link Set} of 
     * {@link String}s.  The argument must not be null and must have no null
     * keys.  It must contain no keys other than Strings and no values other
     * than Strings or Sets of Strings.  This method will throw
     * IllegalArgumentException if the method argument doesn't meet these 
     * requirements.
     * 
     * This method returns a Map equivalent to its argument except whereever there
     * was a String value in the Map there will instead be an immutable Set containing
     * the String value.  That is, the return value is normalized to be a Map from
     * String to Set (of String).
     * 
     * @param mapping {@link Map} from String names of attributes in the underlying store 
     * to uP attribute names or Sets of such names.
     * @return a Map from String to Set of Strings
     * @throws IllegalArgumentException If the {@link Map} doesn't follow the rules stated above.
     */
    @SuppressWarnings("unchecked")
    public static Map<String, Set<String>> parseAttributeToAttributeMapping(final Map<String, ? extends Object> mapping) {
        //null is assumed to be an empty map
        if (mapping == null) {
            return Collections.emptyMap();
        }
        
        final Map<String, Set<String>> mappedAttributesBuilder = new HashMap<String, Set<String>>();
        
        for (final Map.Entry<String, ? extends Object> mappingEntry : mapping.entrySet()) {
            final String sourceAttrName = mappingEntry.getKey();
            Validate.notNull(sourceAttrName, "attribute name can not be null in Map");
            
            final Object mappedAttribute = mappingEntry.getValue();
            
            //Create a mapping to null
            if (mappedAttribute == null) {
                mappedAttributesBuilder.put(sourceAttrName, null);
            }
            //Create a single item set for the string mapping
            else if (mappedAttribute instanceof String) {
                final Set<String> mappedSet = Collections.singleton((String)mappedAttribute);
                mappedAttributesBuilder.put(sourceAttrName, mappedSet);
            }
            //Create a defenisve copy of the mapped set & verify its contents are strings
            else if (mappedAttribute instanceof Set) {
                final Set<String> sourceSet = (Set<String>)mappedAttribute;
                
                //Verify the collection only contains strings.
                CollectionUtils.typedCollection(sourceSet, String.class);
                
                final Set<String> mappedSet = new HashSet<String>(sourceSet);
                mappedAttributesBuilder.put(sourceAttrName, Collections.unmodifiableSet(mappedSet));
            }
            //Not a valid type for the mapping
            else {
                throw new IllegalArgumentException("Invalid mapped type. key='" + sourceAttrName + "', value type='" + mappedAttribute.getClass().getName() + "'");
            }
        }
        
        return Collections.unmodifiableMap(mappedAttributesBuilder);
    }
    
    /**
     * Adds a key/value pair to the specified {@link Map}, creating multi-valued
     * values when appropriate.
     * <br>
     * Since multi-valued attributes end up with a value of type
     * {@link List}, passing in a {@link List} of any type will
     * cause its contents to be added to the <code>results</code>
     * {@link Map} directly under the specified <code>key</code>
     * 
     * @param results The {@link Map} to modify.
     * @param key The key to add the value for.
     * @param value The value to add for the key.
     * @throws IllegalArgumentException if any argument is null
     */
    @SuppressWarnings("unchecked")
    public static <K, V> void addResult(final Map<K, List<V>> results, final K key, final Object value) {
        Validate.notNull(results, "Cannot add a result to a null map.");
        Validate.notNull(key, "Cannot add a result with a null key.");
        
        // don't put null values into the Map.
        if (value == null) {
            return; 
        }
        
        List<V> currentValue = results.get(key);
        if (currentValue == null) {
            currentValue = new LinkedList<V>();
            results.put(key, currentValue);
        }
        
        if (value instanceof List) {
            currentValue.addAll((List<V>)value);
        }
        else {
            currentValue.add((V)value);
        }
    }
    
    /**
     * Takes a {@link Collection} and creates a flattened {@link Collection} out
     * of it.
     * 
     * @param source The {@link Collection} to flatten.
     * @return A flattened {@link Collection} that contains all entries from all levels of <code>source</code>.
     */
    @SuppressWarnings("unchecked")
    public static <T> Collection<T> flattenCollection(final Collection<? extends Object> source) {
        Validate.notNull(source, "Cannot flatten a null collection.");
        
        final Collection<T> result = new LinkedList<T>();
        
        for (final Object value : source) {
            if (value instanceof Collection) {
                final Collection<Object> flatCollection = flattenCollection((Collection<Object>)value);
                result.addAll((Collection<T>)flatCollection);
            }
            else {
                result.add((T)value);
            }   
        }

        return result;
    }
    
    /**
     * This class is not meant to be instantiated.
     */
    private MultivaluedPersonAttributeUtils() {
        // private constructor makes this static utility method class
        // uninstantiable.
    }
}
