/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.jbpm.pvm.internal.stream.ByteArrayStreamSource;
import org.jbpm.pvm.internal.stream.FileStreamSource;
import org.jbpm.pvm.internal.stream.InputStreamSource;
import org.jbpm.pvm.internal.stream.ResourceStreamSource;
import org.jbpm.pvm.internal.stream.StreamSource;
import org.jbpm.pvm.internal.stream.UrlStreamSource;
import org.jbpm.pvm.internal.util.IoUtil;
import org.jbpm.pvm.internal.xml.Parser;
import org.w3c.dom.Document;


/** a deployment unit, containing all information to create a process 
 * definition that will be deployed in the persistent store of the 
 * Process Virtual Machine.
 *  
 * @author Tom Baeyens
 */
public class Deployment implements Serializable {

  private static final long serialVersionUID = 1L;

  private static final Parser parser = new Parser();

  protected String name;
  protected String language;

  protected Map<String, StreamSource> files;
  protected Map<String, Document> documents;
  
  protected ProcessDefinition processDefinition;
  
  public Deployment() {
  }
  
  public Deployment(ProcessDefinition processDefinition) {
    setProcessDefinition(processDefinition);
  }
  

  public String getLanguage() {
    return language;
  }
  
  public void setLanguage(String language) {
    this.language = language;
  }

  public void addResource(String resource) {
    addStreamSource(resource, new ResourceStreamSource(resource));
  }
  
  public void addFile(File file) {
    addStreamSource(file.getAbsolutePath(), new FileStreamSource(file));
  }
  
  public void addUrl(URL url) {
    addStreamSource(url.toString(), new UrlStreamSource(url));
  }
  
  public void addInputStream(String name, InputStream inputStream) {
    addStreamSource(name, new InputStreamSource(inputStream));
  }
  
  public void addArchiveResource(String resource) {
    this.name = resource;
    ResourceStreamSource streamSource = new ResourceStreamSource(resource);
    addStreamSource(resource, streamSource);
  }
  
  public void addArchiveFile(File file) {
    addStreamSource(file.getAbsolutePath(), new FileStreamSource(file));
  }
  
  public void addArchiveUrl(URL url) {
    addStreamSource(url.toString(), new UrlStreamSource(url));
  }
  
  public void addArchive(ZipInputStream zipInputStream) {
    try {
      ZipEntry zipEntry = zipInputStream.getNextEntry();
      while(zipEntry!=null) {
        String entryName = zipEntry.getName();
        byte[] bytes = IoUtil.readBytes(zipInputStream);
        if (bytes!=null) {
          addStreamSource(entryName, new ByteArrayStreamSource(bytes));
        }
        zipEntry = zipInputStream.getNextEntry();
      }
    } catch (Exception e) {
      throw new PvmException("couldn't read zip archive", e);
    }
  }
  
  /** recursively adds all files in a directory using the relative file names */
  public void addDirectory(String directory) {
    if (directory==null) {
      throw new PvmException("directory is null");
    }
    addDirectory(new File(directory), "", false); 
  }
  
  /** recursively adds all files in a directory using the canonical file names */
  public void addDirectoryCanonical(String directory) {
    if (directory==null) {
      throw new PvmException("directory is null");
    }
    addDirectory(new File(directory), "", true); 
  }
  
  /** recursively adds all files in a directory using the relative file names */
  public void addDirectory(File directory) {
    addDirectory(directory, "", false); 
  }
  
  /** recursively adds all files in a directory using the canonical file names */
  public void addDirectoryCanonical(File directory) {
    addDirectory(directory, "", true); 
  }
  
  protected void addDirectory(File directory, String relativeDirectoryName, boolean canonicalPathNames) {
    if (directory==null) {
      throw new PvmException("directory is null");
    }
    if (!directory.isDirectory()) {
      throw new PvmException(directory.getAbsolutePath()+" is not a directory");
    }

    File[] files = directory.listFiles();
    if (files!=null) {
      for (File file: files) {
        String relativeFileName = (canonicalPathNames ? null : relativeDirectoryName+"/"+file.getName());
        if (file.isFile()) {
          if (canonicalPathNames) {
            try {
              addStreamSource(file.getCanonicalPath(), new FileStreamSource(file));
            } catch (IOException e) {
              throw new PvmException("can't get canonical path name for "+file);
            }
          } else {
            addStreamSource(relativeFileName, new FileStreamSource(file));
          }
        } else if (file.isDirectory()) {
          addDirectory(file, relativeFileName, canonicalPathNames);
        }
      }
    }
  }

  protected void addStreamSource(String name, StreamSource streamSource) {
    if (this.name==null) {
      this.name = name;
    }
    if (files==null) {
      files = new HashMap<String, StreamSource>();
    }
    files.put(name, streamSource);
  }
  
  public InputStream getFile(String name) {
    if (files==null) {
      return null;
    }
    StreamSource streamSource = files.get(name);
    return (streamSource!=null ? streamSource.openStream() : null);
  }
  
  public Set<String> getFileNames() {
    if (files==null) {
      return Collections.EMPTY_SET;
    }
    return files.keySet();
  }
  
  public Document getDocument(String name) {
    if ( (documents!=null)
         && (documents.containsKey(name))
       ) {
      return documents.get(name);
    }
    if ( (files!=null)
         && (files.containsKey(name))
       ) {
      InputStream fileStream = getFile(name);
      return parser.createParse()
                   .setInputStream(fileStream)
                   .execute()
                   .checkProblems("deployment file "+name)
                   .getDocument();
    }
    return null;
  }
  
  public ProcessDefinition getProcessDefinition() {
    return processDefinition;
  }
  
  public void setProcessDefinition(ProcessDefinition processDefinition) {
    if (language==null) {
      language = "api";
    }
    this.processDefinition = processDefinition;
  }
  
  public String getName() {
    return name;
  }
  public void setName(String name) {
    this.name = name;
  }
}
