/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.deploy;

import org.jbpm.pvm.Deployment;
import org.jbpm.pvm.PvmException;
import org.jbpm.pvm.client.ClientProcessDefinition;
import org.jbpm.pvm.env.Environment;
import org.jbpm.pvm.internal.log.Log;
import org.jbpm.pvm.internal.model.ProcessDefinitionImpl;
import org.jbpm.pvm.session.PvmDbSession;

/**
 * @author Tom Baeyens
 */
public class CheckVersion implements Deployer {
 
  private static final Log log = Log.getLog(CheckVersion.class.getName());
  
  protected boolean assign = true;

  public void deploy(Deployment deployment) {
    ProcessDefinitionImpl processDefinition = (ProcessDefinitionImpl) deployment.getProcessDefinition();
    
    String name = processDefinition.getName();
    if (name == null) {
      throw new PvmException("process must have a name to deploy it");
    }

    int version = processDefinition.getVersion();
    if ( (version==ProcessDefinitionImpl.UNASSIGNED_VERSION)
         && ! assign
       ) {
      throw new PvmException("no version specified in process definition "+name);
    }
      
    Environment environment = Environment.getCurrent();
    if (environment==null) {
      throw new PvmException("environment is required by deployer check-version");
    }

    PvmDbSession pvmDbSession = environment.get(PvmDbSession.class);
    if (pvmDbSession==null) {
      throw new PvmException(PvmDbSession.class.getName()+" is required in the environment by "+getClass().getName());
    }

    if ( (version==ProcessDefinitionImpl.UNASSIGNED_VERSION)
        && assign
       ) {
      ClientProcessDefinition latestDeployedVersion = pvmDbSession.findLatestProcessDefinitionByName(name);
      if (latestDeployedVersion!=null) {
        version = latestDeployedVersion.getVersion() + 1;
      } else {
        version = 1;
      }
      log.debug("assigning version "+version+" to process definition "+name);
      processDefinition.setVersion(version);

    } else {
      if (pvmDbSession.findProcessDefinitionByName(name, version) != null) {
        throw new PvmException("process '" + name + "' version " + version + " already exists");
      }
    }
  }
}
