/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.job;

import java.text.SimpleDateFormat;
import java.util.Date;

import org.jbpm.pvm.PvmException;
import org.jbpm.pvm.env.Environment;
import org.jbpm.pvm.env.Transaction;
import org.jbpm.pvm.internal.cal.BusinessCalendar;
import org.jbpm.pvm.internal.cal.Duration;
import org.jbpm.pvm.internal.env.JobContext;
import org.jbpm.pvm.internal.jobexecutor.JobAddedNotification;
import org.jbpm.pvm.internal.jobexecutor.JobDbSession;
import org.jbpm.pvm.internal.jobexecutor.JobExecutor;
import org.jbpm.pvm.internal.log.Log;
import org.jbpm.pvm.internal.util.Clock;
import org.jbpm.pvm.job.Timer;
import org.jbpm.pvm.model.ObservableElement;
import org.jbpm.pvm.session.DbSession;

/** a runtime timer instance.
 * 
 * <p>Timers can be
 * </p>
 * 
 * @author Tom Baeyens
 * @author Pascal Verdage
 * @author Alejandro Guizar
 */
public class TimerImpl extends JobImpl<Boolean> implements Timer {

  private static final long serialVersionUID = 1L;
  private static final Log log = Log.getLog(TimerImpl.class.getName());

  private final static String dateFormat = "yyyy-MM-dd HH:mm:ss,SSS";

  protected String signalName;
  protected String eventName;
  protected String repeat;
  
  public static final String EVENT_TIMER = "timer";

  public TimerImpl() {
  }

  public void setDueDateDescription(String dueDateDescription) {
    Duration duration = new Duration(dueDateDescription);
    Date now = Clock.getCurrentTime();
    
    if ( duration.isBusinessTime()
         || duration.getMonths()>0 
         || duration.getYears()>0
       ) {
      Environment environment = Environment.getCurrent();
      if (environment==null) {
        throw new PvmException("no environment to get business calendar for calculating dueDate "+dueDateDescription);
      }
      BusinessCalendar businessCalendar = environment.get(BusinessCalendar.class);
      dueDate = businessCalendar.add(now, duration);

    } else {
      long millis = duration.getMillis() + 
                    1000*( duration.getSeconds() + 
                           60*( duration.getMinutes() + 
                                60*( duration.getHours() + 
                                     24*( duration.getDays() + 
                                          7*duration.getWeeks()))));
      dueDate = new Date(now.getTime() + millis);
    }
  }

  public Boolean execute(Environment environment) throws Exception {
    if (log.isDebugEnabled()) log.debug("executing " + this);

    if (environment==null) {
      throw new PvmException("environment is null");
    }
    
    JobContext jobContext = new JobContext(this);
    environment.addContext(jobContext);
    try {
      if (signalName!=null) {
        if (log.isDebugEnabled()) log.debug("feeding timer signal "+signalName+" into "+execution);
        execution.signal(signalName);
      }
      
      if (eventName!=null) {
        ObservableElement eventSource = execution.getNode();
        if (log.isDebugEnabled()) log.debug("firing event "+signalName+" into "+eventSource);
        execution.fire(eventName, eventSource);
      }
      
    } finally {
      environment.removeContext(jobContext);
    }
    
    boolean deleteThisJob = true;
    // if there is no repeat on this timer
    if (repeat==null) {
      // delete the jobImpl
      if (log.isDebugEnabled()) log.debug("deleting " + this);
      JobDbSession dbSession = environment.get(JobDbSession.class);
      if (dbSession==null) {
        throw new PvmException("no "+JobDbSession.class.getName()+" in environment"); 
      }
      dbSession.delete(this);

    } else { // there is a repeat on this timer
      deleteThisJob = false;
      // suppose that it took the timer runner thread a very long time to execute the timers
      // then the repeat action dueDate could already have passed
      do {
        setDueDateDescription(repeat);
      } while (dueDate.getTime() <= Clock.getCurrentTime().getTime());

      if (log.isDebugEnabled()) log.debug("rescheduled "+this+" for "+formatDueDate(dueDate));
      
      // release the lock on the timer
      setLockOwner(null);
      setLockExpirationTime(null);
      
      // notify the jobExecutor at the end of the transaction
      JobExecutor jobExecutor = environment.get(JobExecutor.class);
      if (jobExecutor!=null) {
        Transaction transaction = environment.get(Transaction.class);
        if (transaction==null) {
          throw new PvmException("no transaction in environment");
        }
        JobAddedNotification jobNotificator = new JobAddedNotification(jobExecutor);
        transaction.registerSynchronization(jobNotificator);
      }
    }

    return deleteThisJob;
  }
  
  public String toString() {
    StringBuffer buffer = new StringBuffer();
    buffer.append("timer[");
    buffer.append(dbid);
    if (dueDate!=null) {
      buffer.append("|");
      buffer.append(formatDueDate(dueDate));
    }
    if (signalName!=null) {
      buffer.append("|");
      buffer.append(signalName);
    }
    if (eventName!=null) {
      buffer.append("|");
      buffer.append(eventName);
    }
    buffer.append("]");
    return buffer.toString();
  }

  public static String formatDueDate(Date date) {
	  return new SimpleDateFormat(dateFormat).format(date);
  }

  public String getSignalName() {
    return signalName;
  }
  public void setSignalName(String signalName) {
    this.signalName = signalName;
  }
  public String getEventName() {
    return eventName;
  }
  public void setEventName(String eventName) {
    this.eventName = eventName;
  }
  public String getRepeat() {
    return repeat;
  }
  public void setRepeat(String repeat) {
    this.repeat = repeat;
  }
}
