/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.jobexecutor;

import java.util.Date;

import org.hibernate.Session;
import org.jbpm.pvm.PvmException;
import org.jbpm.pvm.env.Environment;
import org.jbpm.pvm.env.Transaction;
import org.jbpm.pvm.internal.cmd.Command;
import org.jbpm.pvm.internal.cmd.CommandService;
import org.jbpm.pvm.internal.job.JobImpl;
import org.jbpm.pvm.internal.log.Log;
import org.jbpm.pvm.job.Job;



/**
 * @author Tom Baeyens
 */
public class ExecuteJobCmd implements Command<Job> {

  private static final long serialVersionUID = 1L;

  private static final Log log = Log.getLog(ExecuteJobCmd.class.getName());
  
  protected long jobDbid;

  public ExecuteJobCmd(long jobDbid) {
    this.jobDbid = jobDbid;
  }

  public Job execute(Environment environment) throws Exception {
  	JobDbSession session = environment.get(JobDbSession.class);
  	if (session==null) {
  	  throw new PvmException("no job session configured to get job "+jobDbid+" for execution");
  	}
    JobImpl<?> job = (JobImpl<?>) session.get(JobImpl.class, jobDbid);

  	// in case of exclusive jobs, the jobImpl might have been deleted
  	// before we execute it (they are in a list)
  	if (job != null) {
	    try {
	      log.debug("executing job "+job+"...");
	      job.execute(environment);
	      log.debug("executed job "+job);

	      // if this job is locked too long, it could be unlocked by the lockmonitor and 
	      // executed by another thread.
	      Date lockExpirationDate = job.getLockExpirationTime();
	      // can be null if it was rescheduled
	      if (lockExpirationDate != null) {
		      long lockExpiration = lockExpirationDate.getTime();
		      long currentTime = System.currentTimeMillis();
		      if (currentTime>lockExpiration) {
		        throw new PvmException("job took too long: lock expired "+(currentTime-lockExpiration)+"ms ago");
		      }
	      }
	    } catch (Throwable exception) {
	      log.error("exception while executing '"+job+"'", exception);
	      handleJobExecutionException(environment, job, exception);
	    }

  	} else {
  		log.debug("job " + jobDbid + " no longer exists");
  	}
    return job;
  }

  /** This transaction will be marked for rollback.  A command will be associated with the 
   * Transaction.EVENT_AFTERCOMPLETION (after the jobImpl locks of the current transaction are 
   * released).  Then the command will update the JobImpl with the exception details in a separate 
   * transaction. */
  protected void handleJobExecutionException(Environment environment, JobImpl<?> job, Throwable exception) {
    Transaction transaction = environment.get(Transaction.class);
    transaction.setRollbackOnly();

    CommandService commandService = (CommandService) environment.get(CommandService.class);
    JobExceptionHandler jobExceptionHandler = new JobExceptionHandler(job.getDbid(), exception, commandService);
    transaction.registerSynchronization(jobExceptionHandler);
  }
}
