/**
 * Copyright (C) 2007  Bull S. A. S.
 * Bull, Rue Jean Jaures, B.P.68, 78340, Les Clayes-sous-Bois
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation
 * version 2.1 of the License.
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301, USA.
 **/
package org.jbpm.pvm.internal.jobexecutor;

import org.jbpm.pvm.PvmException;
import org.jbpm.pvm.env.Transaction;
import org.jbpm.pvm.internal.job.TimerImpl;
import org.jbpm.pvm.internal.log.Log;
import org.jbpm.pvm.job.Timer;
import org.jbpm.pvm.session.TimerSession;

/**
 * Timers created with this service are committed at the end of the transaction,
 * so their execution will be late if the delay is shorter than the transaction.
 * In that case, they will be executed at the end of the transaction.
 * 
 * @author Tom Baeyens, Pascal Verdage
 */
public class JobExecutorTimerSession implements TimerSession {
  
	private static final Log log = Log.getLog(TimerSession.class.getName());
	
  /* injected */
  Transaction transaction;

	/* injected */
	JobExecutor jobExecutor; 
	
  /* injected */
  JobDbSession jobDbSession;

  boolean jobExecutorNotificationScheduled = false;

	public void schedule(Timer timer) {
    log.debug("scheduling "+timer);
    validate(timer);
    jobDbSession.save(timer);
    if (!jobExecutorNotificationScheduled) {
      jobExecutorNotificationScheduled = true;
      transaction.registerSynchronization(new JobAddedNotification(jobExecutor));
    }
	}

  protected void validate(Timer timer) {
    if (timer == null)
      throw new PvmException("null timer scheduled");
    if (timer.getExecution() == null) {
      throw new PvmException("timer has no execution specified");
    }
    if ( (timer.getSignalName() == null)
         && (timer.getEventName() == null)
       )
      throw new PvmException("timer has no signalName or eventName specified");
    if (timer.getDueDate() == null)
      throw new PvmException("timer scheduled at null date");
    if (timer.getDueDate().getTime() < 0)
      throw new PvmException("timer scheduled with a negative date");
  }

	public void cancel(Timer timer) {
		log.debug("canceling "+timer);
		if (timer != null) {
		  jobDbSession.delete(timer);
		} else {
			throw new PvmException("timer is null");
		}
	}
}
