/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.model;

import java.util.List;

import org.jbpm.pvm.internal.wire.Descriptor;
import org.jbpm.pvm.internal.wire.WireContext;
import org.jbpm.pvm.model.Condition;
import org.jbpm.pvm.model.Event;
import org.jbpm.pvm.model.Transition;


/** 
 * @author Tom Baeyens
 */ 
public class TransitionImpl extends ObservableElementImpl implements Transition {

  private static final long serialVersionUID = 1L;

  protected NodeImpl source;
  protected NodeImpl destination;
  protected Descriptor conditionDescriptor;
  protected Descriptor waitConditionDescriptor;
  protected boolean isTakeAsync;
  
  /* Use one of the NodeImpl.createOutgoingTransition methods instead. */
  TransitionImpl() {
    super();
  }
  
  public void makeDefault() {
    source.setDefaultTransition(this);
  }
  
  public String toString() {
    String destinationName = (destination!=null ? destination.getName() : null);
    String sourceName = (source!=null ? source.getName() : null);
    return (sourceName!=null ? "("+sourceName+")--" : "--") +
           (name!=null ? name+"-->" : ">") +
           (destinationName!=null ? "("+destinationName+")" : "");
  }
  
  public EventImpl createEvent() {
    EventImpl event = new EventImpl();
    event.setName(Event.TRANSITION_TAKE);
    addEvent(event);
    return event;
  }
  
  // get parent ///////////////////////////////////////////////////////////////
  
  /** the first common parent between the source and the destination node. The 
   * source and destination itself are included in the search except if 
   * source and destination are equal.  In that case (self-transition), then 
   * it's the parent of the node. */
  public ObservableElementImpl getParent() {
    // if one of the two ends is null
    if ( (source==null)
         || (destination==null) 
       ) {
      // the process definition is returned
      return processDefinition;
    }
    // if this is a self transition
    if (source.equals(destination)) {
      // the parent of the node is returned
      return source.getParent();
    }
    // if none of the above, we search for the first common element in the parent chains 
    List<ObservableElementImpl> sourceChain = source.getParentChain();
    List<ObservableElementImpl> destinationChain = destination.getParentChain();
    for (ObservableElementImpl sourceElement : sourceChain) {
      for (ObservableElementImpl destinationElement : destinationChain) {
        if (sourceElement.equals(destinationElement)) {
          return sourceElement;
        }
      }
    }
    return null;
  }


  // getters and setters //////////////////////////////////////////////////////
  
  public void setName(String name) {
    // if there is no source node associated with this transition
    if (source==null) {
      // it s just a setter
      this.name = name;

    } else { // otherwise
      // make sure the source node's leavingTransitionsMap remains up to date
      if (this.name!=null) {
        source.removeOutgoingTransition(this);
      }
      this.name = name;
      if (name!=null) {
        source.addOutgoingTransition(this);
      }
    }
  }
  
  public Condition getCondition() {
    if (conditionDescriptor==null) {
      return null;
    }
    return (Condition) WireContext.create(conditionDescriptor);
  }

  public Condition getWaitCondition() {
    if (waitConditionDescriptor==null) {
      return null;
    }
    return (Condition) WireContext.create(waitConditionDescriptor);
  }

  public EventImpl getEvent() {
    return getEvent(Event.TRANSITION_TAKE);
  }
  public NodeImpl getSource() {
    return source;
  }
  public void setSource(NodeImpl source) {
    this.source = source;
  }
  public NodeImpl getDestination() {
    return destination;
  }
  public void setDestination(NodeImpl destination) {
    this.destination = destination;
  }
  /** see <a href="#conditions">Conditions</a> */
  public Descriptor getConditionDescriptor() {
    return conditionDescriptor;
  }
  /** see <a href="#conditions">Conditions</a> */
  public void setConditionDescriptor(Descriptor conditionDescriptor) {
    this.conditionDescriptor = conditionDescriptor;
  }
  /** see <a href="#waitconditions">Wait conditions</a> */
  public Descriptor getWaitConditionDescriptor() {
    return waitConditionDescriptor;
  }
  /** see <a href="#waitconditions">Wait conditions</a> */
  public void setWaitConditionDescriptor(Descriptor waitConditionDescriptor) {
    this.waitConditionDescriptor = waitConditionDescriptor;
  }
  public boolean isTakeAsync() {
    return isTakeAsync;
  }
  public void setTakeAsync(boolean isTakeAsync) {
    this.isTakeAsync = isTakeAsync;
  }
}
