/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.model.op;

import java.util.LinkedList;
import java.util.List;

import org.jbpm.pvm.internal.job.MessageImpl;
import org.jbpm.pvm.internal.model.ExecutionImpl;
import org.jbpm.pvm.internal.model.NodeImpl;
import org.jbpm.pvm.internal.model.ObservableElementImpl;

/**
 * @author Tom Baeyens
 */
public class ProceedToDestination implements AtomicOperation {
  
  public boolean isAsync(ExecutionImpl execution) {
    return false;
  }

  public void perform(ExecutionImpl execution) {
    NodeImpl destination = execution.getTransition().getDestination();
    execution.setNode(destination);
    List<NodeImpl> enteredNodes = getNodesEntered(execution.getTransitionOrigin(), destination);
    
    ExecutionImpl propagatingExecution = execution;
    for (NodeImpl enteredNode : enteredNodes) {
      propagatingExecution = propagatingExecution.beginNode(enteredNode);
    }
    
    // Restore original transition source node 
    // (so that the call to moveTo updates the previous node) 
    propagatingExecution.setNode(execution.getTransitionOrigin());
    propagatingExecution.moveTo(destination);
    
    propagatingExecution.performAtomicOperation(ExecutionImpl.EXECUTE_NODE);
  }
  
  public List<NodeImpl> getNodesEntered(NodeImpl origin, NodeImpl destination) {
    LinkedList<NodeImpl> nodesEntered = new LinkedList<NodeImpl>();
    
    if (origin.equals(destination)) {
      nodesEntered.add(destination);
      
    } else {
      List<ObservableElementImpl> sourceChain = origin.getParentChain();
      
      if (!sourceChain.contains(destination)) {
        NodeImpl destinationNode = destination;
        while ( (destinationNode!=null)
                && (!sourceChain.contains(destinationNode))
              ) {
          nodesEntered.addFirst(destinationNode);
          destinationNode = destinationNode.getParentNode(); 
        }
      }
    }

    return nodesEntered;
  }

  
  public String toString() {
    return "proceed-to-destination";
  }

  public MessageImpl<?> createAsyncMessage(ExecutionImpl execution) {
    return new ProceedToDestinationMessage(execution);
  }
}
